/***************************************************************************
****************************************************************************
****************************************************************************
*
* Clobberd - By Jason Nunn
* Copyright (C) Jason Nunn
*
* Clobberd comes under the GNU General Public License. Please read the
* COPYING notice in this distribution.
*
* ================================================================
* gui
*
****************************************************************************
****************************************************************************
***************************************************************************/
extern "C" {
#include <curses.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <utmp.h>
#include <malloc.h>
#include "../common/defs.h"
#include "../common/common.h"
#include "../common/common_client.h"
}
#include "gui_obj.h"

int ch = 0,password_mode = 0;

/***************************************************************************
*   
***************************************************************************/
void fatal_memory_error_gui(void)
{ 
  clear();
  refresh();
  printf("Fatal memory error. Aborting program.");
  fflush(stdout);
  sleep(3);
  exit(-1);
}

/***************************************************************************
* Bg top bar colour         : BLUE
* Fg top bar colour         : WHITE
* Bg dialog border colour   : RED
* Fg dialog border colour   : WHITE
* Bg dialog colour          : RED
* Fg dialog colour          : YELLOW
* Bg window border colour   : BLUE
* Fg window border colour   : GREEN
* Bg window colour          : BLUE
* Fg window colour          : CYAN
*
* nb/ in clobberd 3.x, this was all configurable in clobberedit ;)
***************************************************************************/
void init_colour_pairs(void)
{
  init_pair(0,COLOR_BLACK,COLOR_BLACK);

/*topbar*/
  init_pair(1,COLOR_WHITE,COLOR_BLUE);

/*dialog box border*/
  init_pair(2,COLOR_WHITE,COLOR_RED);
/*dialog box box*/
  init_pair(3,COLOR_YELLOW,COLOR_RED);

/*window border*/
  init_pair(4,COLOR_GREEN,COLOR_BLUE);
/*window*/
  init_pair(5,COLOR_CYAN,COLOR_BLUE);

/*special*/
  init_pair(6,COLOR_BLUE,COLOR_BLUE);
}

/***************************************************************************
*
***************************************************************************/
void display_topbar(void)
{
  register int x;
  char s[81];

  if(has_colors())
    attrset(COLOR_PAIR(1) | A_BOLD);
  else
    attron(A_REVERSE);
  move(0,0);
  sprintf(s," Clobberd %-4s  Remote Maintenance      "
            "    Copyright (C) 1996-2001  Jason Nunn ",VERSION);
  addstr(s);
  move(LINES - 1,0);
  for(x = 0;x < 80;x++)
    addch(' ');
  if(has_colors())
    attrset(COLOR_PAIR(6) | A_BOLD);
  else
    attron(A_REVERSE);
  move(LINES - 1 ,1);
  sprintf(s,"[%s]",addr);
  addstr(s);
}

void clear_area(int y1,int x1,int y2,int x2)
{
  for(y1 = y1;y1 <= y2;y1++)
  {
    register int a;

    move(y1,x1);
    for(a = x1;a <= x2;a++)
      addch(' ');
  }
}

void update(void)
{
  move(0,0);
  refresh();
}

/***************************************************************************
*tstring
***************************************************************************/
tstring::tstring(char y,char x,char *str,int slen,int dlen)
{
  end = 0;
  x_pos = x;
  y_pos = y;
  edit_str = str;
  str_len = slen;
  dis_len = dlen;
  rp = dp = 0;
}

void tstring::redraw(void)
{
  move(y_pos,x_pos);

  if(password_mode)
  {
    register int x;

    for(x = 0;x < dis_len;x++)
      if(x < dp)
        addch('*');
      else
        addch(' ');
  }
  else
    addnstr(edit_str + rp,dis_len);
  move(y_pos,x_pos + dp);
  refresh();
}

void tstring::dec_dp(void)
{
  if(dp > 0)
  {
    dp--;
    redraw();
  }
  else
  {
    if(rp > 0)
      rp--;
    redraw();
  }
}

void tstring::inc_dp(void)
{
  if(dp < (dis_len - 1))
  {
    if(dp < str_len)
    {
      dp++;
      redraw();
    }
  }
  else
  {
    if(rp < (str_len - dis_len))
      rp++;
    redraw();
  }
}

void tstring::run(void)
{
  register int a;

  attron(A_REVERSE);
  for(a = 0;a < dis_len;a++)
  {
    move(y_pos,x_pos + a);
    addch(' ');
  }
  for(a = strlen(edit_str);a < str_len;a++)
    edit_str[a] = ' ';
  redraw();
  while(!end)
  {
    register int ap;

    ch = getch();
    ap = rp + dp;
    switch(ch)
    {
      case KEY_UP:
      case KEY_DOWN:
        break;
      case KEY_LEFT:
        dec_dp();
        break;
      case KEY_RIGHT:
        inc_dp();
        break;
      case 10:
        end = 1;
        break;
      case KEY_BACKSPACE:
      case KEY_DC:
        if(ap > 0)
        {
          edit_str[ap - 1] = ' ';
          dec_dp();
        }
        break;
      default:
        if(ap < str_len)
        {
          edit_str[ap] = ch;
          inc_dp();
        }
        break;
    }
  }
  attroff(A_REVERSE);
  clear_area(y_pos,x_pos,y_pos,x_pos + dis_len - 1);
  edit_str[str_len] = 0;
  do
  {
    str_len--;
    if(edit_str[str_len] == ' ')
      edit_str[str_len] = 0;
    else
      break;
  } while(str_len >= 0);
}

/***************************************************************************
*
***************************************************************************/
int get_int(int y,int x,int strlen) // 0123456789
{
  int v = 0;
  char str[11] = "";

  tstring ts(y,x,str,10,strlen);
  ts.run();
  sscanf(str,"%d",&v);
  return v;
}

float get_real(int y,int x,int strlen) //0123456789.99
{
  float v = 0;
  char str[14] = "";

  tstring ts(y,x,str,13,strlen);
  ts.run();
  sscanf(str,"%f",&v);
  return v;
}

int get_mm_ss(int y,int x,int strlen) // MM:SS
{
  char str[6] = "";
  int m = 0,s = 0;

  tstring ts(y,x,str,5,strlen);
  ts.run();
  sscanf(str,"%u:%u",&m,&s);
  if(s > 59) s = 59;
  return s + (m * 60);
}

int get_hh_mm(int y,int x,int strlen) // HHH:MM
{
  char str[7] = "";
  int h = 0,m = 0;

  tstring ts(y,x,str,6,strlen);
  ts.run();
  sscanf(str,"%u:%u",&h,&m);
  if(m > 59) m = 59;
  return (m + (h * 60)) * 60;
}

void scroll_set_line(int y,int x,int hl_bar_size)
{
  register int a;

  move(y,x);
  for(a = 0;a < hl_bar_size;a++)
    addch((char)inch());
}

void scroll_un_hl_line(int y,int x,int hl_bar_size)
{
  attroff(A_REVERSE);
  scroll_set_line(y,x,hl_bar_size);
}

void scroll_hl_line(int y,int x,int hl_bar_size)
{
  attron(A_REVERSE);
  scroll_set_line(y,x,hl_bar_size);
}

void scroll_draw_sb(int y,int x,int list_size,int dlen,int pos,int hl)
{
  register int a;

  if(!list_size) attroff(A_REVERSE);
  for(a = 0;a < dlen;a++)
  {
    move(y + a,x);
    if(list_size)
    {
      register int x = ((pos + hl) * dlen) / list_size;
      if(x > (dlen - 1)) x = (dlen - 1);
      if(a == x) 
        addch('*');
      else
        addch(' ');
    }
    else
      addch(' ');
  }
}

void scroll_clear_line(int hl_bar_size)
{
  register int p;

  for(p = 0;p < hl_bar_size;p++) addch(' ');
}

void scroll_append_sync_ptrs(int list_size,int dlen,int *pos,int *hl)
{
  if(list_size < dlen)
  {
    *pos = 0;
    *hl = list_size - 1;
  }
  else
  {
    *pos = list_size - dlen;
    *hl = dlen - 1;
  }
}

/***************************************************************************
*
***************************************************************************/

/*********************************
*tllist
*********************************/
tllist::tllist(void)
{
  curr = NULL;
  next = NULL;
  curr_ptr = this;
}

tllist::~tllist(void)
{
  if(curr != NULL) free(curr);
  if(next != NULL) delete(next);
}

void tllist::append(void *data,int size)
{
  if(curr == NULL)
  {
    if((curr = malloc(size)) != NULL)
      memcpy(curr,data,size);
    else
      fatal_memory_error_gui();
  }
  else
  {
    if(next == NULL)
    {
      next = new(tllist);
      if(next == NULL)
        fatal_memory_error_gui();
    }
    next->append(data,size);
  }
}

void tllist::rn(tllist *o,int index)
{
  if(index)
  {
    if(o->next == NULL)
      return;
    else
    {
      rn(o->next,index - 1);
      if(index == 1) //one up
      {
        tllist *tmp = o->next;
        o->next = o->next->next;
        tmp->next = NULL;
        delete(tmp);
      }
    }
  }
}

void tllist::remove_node(int index)
{
  if(index)
    rn(this,index);
  else
  {
    if(curr != NULL)
    {
      free(curr);
      curr = NULL;
    }
    if(next != NULL)
    {
      curr = next->curr;
      next->curr = NULL;
      rn(this,1);
    }
  }
}

void tllist::restart(void)
{
  curr_ptr = this;
}

int tllist::read_next(void **p)
{
  if(curr_ptr != NULL)
  {
    *p = curr_ptr->curr;
    curr_ptr = curr_ptr->next;
    return 1;
  }
  return 0;
}

int tllist::get_ls(tllist *o,int n)
{
  register int t = n;

  if(o->next != NULL)
    t = get_ls(o->next,n + 1);
  return t;
}

int tllist::get_list_size(void)
{
  return get_ls(this,curr == NULL ? 0 : 1);
}

/*********************************
*tllist_obj
*********************************/
tllist_obj::tllist_obj(void)
{
  curr = NULL;
  next = NULL;
  curr_ptr = this;
}

tllist_obj::~tllist_obj(void)
{
  if(next != NULL) delete(next);
}

void tllist_obj::append(tobj *obj)
{
  if(curr == NULL)
    curr = obj;
  else
  {
    if(next == NULL)
    {
      next = new(tllist_obj);
      if(next == NULL)
        fatal_memory_error_gui();
    }
    next->append(obj);
  }
}

void tllist_obj::r(tllist_obj *o)
{
  if(o->next == NULL)
    o->curr = NULL;
  else
  {
    r(o->next);
    if(o->next->curr == NULL)
    {
      delete(o->next);
      o->next = NULL;
    }
  }
}

void tllist_obj::remove(void)
{
  r(this);
}

void tllist_obj::restart(void)
{
  curr_ptr = this;
}

int tllist_obj::read_next(tobj **p)
{
  if(curr_ptr != NULL)
  {
    *p = curr_ptr->curr;
    curr_ptr = curr_ptr->next;
    return 1;
  }
  return 0;
}

/*********************************
*tllist_str
*********************************/
void tllist_str::append(char *str)
{
  tllist::append(str,strlen(str) + 1);
}

void tllist_str::get_str(int index,int offs,char **src)
{
  tllist *o = this;

  while(index--)
    if(o->next != NULL)
      o = o->next;
    else
    {
      *src = NULL;
      return;
    }
  *src = (char *)o->curr + offs;
}

void tllist_str::ammend_str(int index,int offs,char *src)
{
  tllist *o = this;

  while(index--)
    if(o->next != NULL)
      o = o->next;
    else
      return;
  strcpy((char *)o->curr + offs,src);
}

/***************************************************************************
*     
***************************************************************************/
tllist_obj list_obj;

void blit_list_obj(void)
{
  attrset(COLOR_PAIR(0));
  clear_area(1,0,stdscr->_maxy - 2,stdscr->_maxx);
  list_obj.restart();
  while(1)
  {
    tobj *o;

    if(list_obj.read_next(&o))
    {
      if(o != NULL) o->redraw();
    }
    else
      break;
  }
  update();
}

/***************************************************************************
*tobj
***************************************************************************/
void tobj::create(tobj *obj)
{
  list_obj.append(obj);
  blit_list_obj();
}

void tobj::dest(void)
{
  list_obj.remove();
  blit_list_obj();
}

void tobj::redraw(void)
{
}

/***************************************************************************
*tdialog
***************************************************************************/
tdialog::tdialog(void)
{
  running = 1;
  touched = 0;
}

void tdialog::create(char *t,int sy,int sx,int py,int px,int c)
{
  title = t;
  size_x = sx;
  size_y = sy;
  pos_x = px;
  pos_y = py;
  cpair = c;
}

void tdialog::redraw(void)
{
  register int a,b,asx = size_x - 2;

  attrset(cpair);
  move(pos_y,pos_x);
  addch(ACS_ULCORNER);
  for(a = 0;a < asx;a++) addch(ACS_HLINE);
  addch(ACS_URCORNER);
  for(a = 1;a < (size_y - 1);a++)
  {
    move(pos_y + a,pos_x);
    addch(ACS_VLINE);
    for(b = 0;b < asx;b++) addch(' ');
    addch(ACS_VLINE);
  }
  move(pos_y + size_y - 1,pos_x);
  addch(ACS_LLCORNER);
  for(a = 0;a < asx;a++) addch(ACS_HLINE);
  addch(ACS_LRCORNER);

  a = strlen(title);
  b = size_x - 2;
  if(a > b) a = b;
  move(pos_y,pos_x + ((b - a) >> 1) + 1);
  addnstr(title,size_x - 2);
}

/***************************************************************************
* Notify Dialog
***************************************************************************/
void tnotify::create(char *name,char *cont)
{
  register int x,y;

  x = strlen(cont) + 2;
  if(x > 60)
  {
    y = (x / 60) + 3;
    x = 60;
  }
  else
    y = 3;
  tdialog::create(
    name,y,x,
    ((stdscr->_maxy - y) >> 1) - 3,
    (stdscr->_maxx - x) >> 1,
    COLOR_PAIR(2) | A_BOLD);
  contents = cont;
  tobj::create(this);
}

void tnotify::redraw(void)
{
  register int x = 0,y = pos_y + 1;

  tdialog::redraw();
  attrset(COLOR_PAIR(3) | A_BOLD);
  while(1)
  {
    register int inc = 58;

    move(y++,pos_x + 1);
    if((x + inc) < (int)strlen(contents))
    {
      while(inc > 0)
      {
        if(*(contents + x + inc) == ' ')
          break;
        inc--;
      }
      addnstr(contents + x,inc);
      x += inc + 1;
    }
    else
    {
      addstr(contents + x);
      break;
    }
  }
}

int tnotify::run(void)
{
  beep();
  refresh();
  return getch();
}

/***************************************************************************
*
***************************************************************************/
int tquit::run(void)
{
  ch = tnotify::run();
  if((ch == 'y') || (ch == 'Y'))
    return 0;
  ch = 0;
  return 1;
}

/***************************************************************************
*
***************************************************************************/
void tscroll_select::append(char *str)
{
  tllist_str::append(str);
  list_size = get_list_size();
}

void tscroll_select::remove_node(void)
{
  tllist::remove_node(pos + hl);
  list_size = get_list_size();
}

void tscroll_select::init(int y,int x,int l,int hlbs,int sbf)
{
  pos_x = x;
  pos_y = y;
  length = l;
  pos = 0;
  hl = 0;
  hl_bar_size = hlbs;
  list_size = 0;
  scroll_bar_f = sbf;
}

void tscroll_select::redraw(void)
{
  register int a;
  char *s;

  attroff(A_REVERSE);
  restart();
  for(a = 0;a < pos;a++)
    read_next((void **)&s);
  for(a = 0;a < length;a++)
  {
    move(pos_y + a,pos_x);
    if(read_next((void **)&s))
    {
      if(s != NULL)
      {
        register int p;

        addstr(s);
        for(p = 0;p < (hl_bar_size - (int)strlen(s));p++)
          addch(' ');
      }
      else
        scroll_clear_line(hl_bar_size);
    }
    else
      scroll_clear_line(hl_bar_size);
  }
  scroll_hl_line(pos_y + hl,pos_x,hl_bar_size);
  if(scroll_bar_f)
    scroll_draw_sb(pos_y,pos_x + hl_bar_size,list_size,length,pos,hl);
}

void tscroll_select::cursor_up(void)
{
  if(hl > 0)
  {
    scroll_un_hl_line(pos_y + hl,pos_x,hl_bar_size);
    hl--;
    scroll_hl_line(pos_y + hl,pos_x,hl_bar_size);
    if(scroll_bar_f)
      scroll_draw_sb(pos_y,pos_x + hl_bar_size,list_size,length,pos,hl);
  }
  else
  {
    if(pos > 0) pos--;
    redraw();
  }
}

void tscroll_select::cursor_down(void)
{
  if(list_size)
    if(hl < (length - 1))
    {
      if((hl + pos) < (list_size - 1))
      {
        scroll_un_hl_line(pos_y + hl,pos_x,hl_bar_size);
        hl++;
        scroll_hl_line(pos_y + hl,pos_x,hl_bar_size);
        if(scroll_bar_f)
          scroll_draw_sb(pos_y,pos_x + hl_bar_size,list_size,length,pos,hl);
      }
    }
    else
    {
      if(pos < (list_size - length))
        pos++;
      redraw();
    }
}

void tscroll_select::page_up(void)
{
  register int x;

  for(x = 0;x < length;x++)
    cursor_up();
}

void tscroll_select::page_down(void)
{
  register int x;

  for(x = 0;x < length;x++)
    cursor_down();
}

void tscroll_select::goto_end(void)
{
  if(list_size)
  {
    if(list_size < length)
    {
      pos = 0;
      hl = list_size - 1;
    }
    else
    {
      pos = list_size - length;
      hl = length - 1;
    }
    redraw();
  }
}

int tscroll_select::get_option(void)
{
  return pos + hl;
}

int tscroll_select::get_integer(int o)
{
  return get_int(pos_y + hl,pos_x + o,10);
}

float tscroll_select::get_float(int o)
{
  return get_real(pos_y + hl,pos_x + o,10);
}

int tscroll_select::get_min_sec(int o)
{
  return get_mm_ss(pos_y + hl,pos_x + o,5);
}

int tscroll_select::get_hr_min(int o)
{
  return get_hh_mm(pos_y + hl,pos_x + o,8);
}

void tscroll_select::get_string(char *str,int o,int slen,int dlen)
{
  tstring ts(
    pos_y + hl,
    pos_x + o,str,slen,dlen);
  ts.run();
}

int tscroll_select::goto_user(char *user)
{
  register int uls,s = 1;

  pos = 0;
  hl = 0;
  uls = list_size - 1;
  for(;;)
  {
    register int p;
    char *entry;

    p = get_option();
    get_str(p,0,&entry);
    if(!strcmp(user,entry + 1))
      break;   
    if(p == uls)
    {
      s = 0;
      break;
    }
    cursor_down();
  }
  redraw();
  return s;
}

/***************************************************************************
* string box
***************************************************************************/
tstrbox::tstrbox(void)
{
}

int tstrbox::create(char *title,int sy,int sx,int py,int px,int sl)
{
  tdialog::create(
    title,
    sy,sx + 2,py,px,
    COLOR_PAIR(4) | A_BOLD);
  tobj::create(this);
  str_len = sl;
  dstr_len = sx;
  return 1;
}

void tstrbox::run(char *key_str)
{
  tstring ts(pos_y + 1,pos_x + 1,key_str,str_len,dstr_len);
  ts.run();
}
