/* main.c -- Parallel Libra Tester entry routines.
 * Marc SCHAEFER <schaefer@dilog.ch>
 * Creation: 01/08/96
 * Update:   10/12/98
 * V1.0 PV011
 * NOTES
 * BUGS
 *    - Rethink the way we give the MC device name (and tape)
 * TODO
 * MODIFICATION-HISTORY
 *    10/12/98  schaefer Applying new RCS scheme and adding support for
 *                       no INITIALIZE ELEMENT STATUS (for Adic) and
 *                       POSITION TO ELEMENT (also for Adic).
 * $Id: main.c,v 1.3 1999/04/19 15:05:09 schaefer Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include "types.h"
#include "debug_if.h"
#include "plt_if.h"
#include "mc_if.h"
#include "time_if.h"
#ifdef ENABLE_INTERACTIVE_MODE
#include "tests_if.h"
#endif

/* Private type definitions */

/* Macros */
#define PLT_PV "PLT V1.2 PV002 $Id: main.c,v 1.3 1999/04/19 15:05:09 schaefer Exp $"
#define PLT_PV004_COMPATIBILITY /* device name alone */
/* #define SHOW_COPYRIGHT */

/* #define DEBUG_PARSER */

/* Private function definitions */

/* NAME
 *    usage
 * DESCRIPTION
 *    Prints a text about the use of this software. Will
 *    act differently on the base of some macro definitions.
 * RESULT
 *    NONE
 * NOTES
 * BUGS
 * TODO
 */
static void usage(char *name);

/* Global variables */

debug_level_t debug_level; /* global */
BOOL do_show_elapsed = FALSE;
BOOL do_continuous = FALSE;
BOOL do_label = FALSE;
BOOL do_write = FALSE;
BOOL do_full_unload = FALSE;
unsigned short max_time = 0xFFFF; /* 65535 minutes, so appx. 1000 hours */

/* Private variables */

/* Function implementation */

/* NAME
 *    main
 * DESCRIPTION
 *    This function performs the basic initialization, argument
 *    parsing and general checks.
 * RESULT
 *    int   0   ok
 *          1   error, test failed
 *          2   fatal software or user error
 * NOTES
 * BUGS
 *    - Not using getopt.
 * TODO
 *    - Implement time limit, see specification ../SPEC
 */
int main(int argc, char **argv) {
   plt_test_result_t result;
   int i;
   BOOL device_set = FALSE;
   plt_do_t opcode = PLT_DO_RANDOM;
   BOOL direct_command = FALSE;
   BOOL skip_initialization = FALSE;
   plt_direct_cmd_t cmd;
#ifdef ENABLE_INTERACTIVE_MODE
   BOOL interactive = FALSE;
#endif

   if (argc < 2) {
      usage(argv[0]);
      exit(2);
   }

   debug_level = DEBUG_INFO; /* all debug messages */

   i = 1;
   while (i < argc) {
#ifdef DEBUG_PARSER
      printf("argc %d arg %s.\n", i, argv[i]);
#endif
      if (!strcmp("--show_elapsed", argv[i])
          || !strcmp("-s", argv[i])) {
         do_show_elapsed = TRUE;
      }
      else if (!strcmp("--check_labels", argv[i])
          || !strcmp("-l", argv[i])) {
         do_label = TRUE;
      }
      else if (!strcmp("--full_unload", argv[i])) {
         do_full_unload = TRUE;
      }
      else if (!strcmp("--write", argv[i])
          || !strcmp("-w", argv[i])) {
         do_write = TRUE;
      }
      else if (!strcmp("--continuous", argv[i])
          || !strcmp("-c", argv[i])) {
         do_continuous = TRUE;
      }
      else if (!strcmp("--debug", argv[i])
               || !strcmp("-d", argv[i])) {
         if ((i + 1) < argc) {
            i++;
	    if (!strcmp(argv[i], "INFO")) {
	       debug_level = DEBUG_INFO;
	    }
	    else if (!strcmp(argv[i], "NOTICE")) {
	       debug_level = DEBUG_NOTICE;
	    }
	    else if (!strcmp(argv[i], "WARN")) {
	       debug_level = DEBUG_WARN;
	    }
	    else if (!strcmp(argv[i], "ERROR")) {
	       debug_level = DEBUG_ERROR;
	    }
	    else if (!strcmp(argv[i], "FATAL")) {
	       debug_level = DEBUG_FATAL;
	    }
	    else {
	       usage(argv[0]);
	       exit(2);
	    }
         }
         else {
           usage(argv[0]);
           exit(2);
         }  
      }
      else if (!strcmp("--operation", argv[i])
               || !strcmp("-o", argv[i])) {
         if ((i + 1) < argc) {
            i++;
	    if (!strcmp(argv[i], "random")) {
	       opcode = PLT_DO_RANDOM;
	    }
	    else if (!strcmp(argv[i], "seq")) {
	       opcode = PLT_DO_SEQ;
	    }
	    else if (!strcmp(argv[i], "quick")) {
	       opcode = PLT_DO_QUICK;
	    }
	    else if (!strcmp(argv[i], "seq_h")) {
	       opcode = PLT_DO_SEQUENTIAL_H;
	    }
	    else if (!strcmp(argv[i], "seq_l")) {
	       opcode = PLT_DO_SEQUENTIAL_L;
	    }
	    else if (!strcmp(argv[i], "16x16")) {
	       opcode = PLT_DO_16x16;
	    }
	    else if (!strcmp(argv[i], "adsm_1")) {
	       opcode = PLT_DO_ADSM_1;
	    }
	    else if (!strcmp(argv[i], "adsm_2")) {
	       opcode = PLT_DO_ADSM_2;
	    }
	    else if (!strcmp(argv[i], "label")) {
	       opcode = PLT_DO_LABEL;
	    }

	    else {
	       usage(argv[0]);
	       exit(2);
	    }
         }
         else {
           usage(argv[0]);
           exit(2);
         }  
      }
      else if (!strcmp("--direct_command", argv[i])) {
         if ((i + 1) < argc) {
            i++;
	    if (!strcmp(argv[i], "load")) {
	       cmd.cmd = PLT_CMD_LOAD;
               direct_command = TRUE;
	    }
	    else if (!strcmp(argv[i], "unload")) {
	       cmd.cmd = PLT_CMD_UNLOAD;
               direct_command = TRUE;
	    }
	    else if (!strcmp(argv[i], "move_medium")) {
	       cmd.cmd = PLT_CMD_MOVE_MEDIUM;
               if ((i + 3) < argc) {
                  i++;
                  cmd.arg1 = atoi(argv[i]);
                  i++;
                  cmd.arg2 = atoi(argv[i]);
                  i++;
                  cmd.arg3 = atoi(argv[i]);
                  direct_command = TRUE;
               }
               else {
	          usage(argv[0]);
	          exit(2);
               }
	    }
	    else if (!strcmp(argv[i], "position")) {
	       cmd.cmd = PLT_CMD_POSITION;
               if ((i + 2) < argc) {
                  i++;
                  cmd.arg1 = atoi(argv[i]);
                  i++;
                  cmd.arg2 = atoi(argv[i]);
                  direct_command = TRUE;
               }
               else {
	          usage(argv[0]);
	          exit(2);
               }
	    }
	    else {
	       usage(argv[0]);
	       exit(2);
	    }
         }
         else {
           usage(argv[0]);
           exit(2);
         }  
      }
      else if (!strcmp("--drive", argv[i])) {
         if ((i + 2) < argc) {
            mc_set_tape_device(argv[i + 1], atoi(argv[i + 2]));
            i += 2;
         }
      }
      else if (!strcmp("--no_initialization", argv[i])) {
         skip_initialization = TRUE;
      }
#ifdef ENABLE_INTERACTIVE_MODE
      else if (!strcmp("--interactive", argv[i])
               || !strcmp("-i", argv[i])) {
         interactive = TRUE;
         break;
      }
#endif
      else if (!strcmp("--medium_changer", argv[i])
               || !strcmp("-m", argv[i])) {
         if ((i + 1) < argc) {
            mc_set_mc_device(argv[i + 1]);
            device_set = TRUE;
            i++;
         }
         else {
           usage(argv[0]);
           exit(2);
         }  
      }
      else if (!strcmp("--time", argv[i])) {
         if ((i + 1) < argc) {
            max_time = atoi(argv[i + 1]);
            i++;
         }
         else {
           usage(argv[0]);
           exit(2);
         }
      }
#ifdef PLT_PV004_COMPATIBILITY
      else if (argv[i][0] == '/') {
         mc_set_mc_device(argv[i]);
         device_set = TRUE;
      }
#endif
      else {
         usage(argv[0]);
         exit(2);
      }
      i++;  
   }

   if (!device_set) {
      usage(argv[0]);
      exit(2);
   }
      

   MESSAGE(DEBUG_INFO, "", "", "",
           PLT_PV);

   MESSAGE(DEBUG_INFO, "plt", "main", "DEBUG_LEVEL",
           DEBUG_LEVEL_DESCR(debug_level));

   time_init();

#ifdef ENABLE_INTERACTIVE_MODE
   if (interactive) {
#ifdef DEBUG_PARSER
      printf("passing %d arguments starting at %s.\n", argc -i, argv[i + 1]);
#endif
      result = (tests_main(argc - i, &argv[i]) == 0) ? TEST_SUCCEEDED
                                                     : TEST_FAILED;
               /* and not i + 1 */
   }
   else
#endif

   if (plt_initialize(!direct_command, skip_initialization)) {
      if (direct_command) {
	 result = plt_direct_command(cmd) ? TEST_SUCCEEDED : TEST_FAILED ;
      }
      else {
	 result = plt_do(opcode);
      }
   }
   else {
      MESSAGE(DEBUG_ERROR, "plt", "main", "plt_initialize()",
	     	           "can't initialize");
      exit(2);
   }

   switch (result) {
      case TEST_SUCCEEDED: 
         MESSAGE(DEBUG_INFO, "plt", "main", "test", "succeeded");
         if (do_show_elapsed) {
            printf("Elapsed: ");
            time_show_elapsed_time();
            printf(".\n");
         }
         exit(0);
         break;
      case TEST_FAILED:
         MESSAGE(DEBUG_ERROR, "plt", "main", "test", "failed");
         if (do_show_elapsed) {
            printf("Elapsed: ");
            time_show_elapsed_time();
            printf(".\n");
         }
         exit(1);
         break;
      default:
         MESSAGE(DEBUG_FATAL, "plt", "main", "test",
                 "internal inconsistency");
         if (do_show_elapsed) {
            printf("Elapsed: ");
            time_show_elapsed_time();
            printf(".\n");
         }
         exit(2);
         break;
   }
}


/* Private function implementation */

#ifndef OLD_PARSER
static void usage(char *name) {
#ifdef ENABLE_INTERACTIVE_MODE
   fprintf(stderr, "%s [--debug {INFO|NOTICE|WARN|ERROR|FATAL}] [--operation {random|seq_l|seq_h|16x16|adsm_1|adsm_2|label|quick|seq}] [--direct_command {unload|load|move_medium ARM FROM TO|position ARM TO}] [--drive /dev/stX {1|2}] --medium_changer /dev/sg[a-z] [--show_elapsed] [--no_initialization] [--time MINUTES] [--full_unload] [--check_labels] [--write] [--continuous] [--interactive]\n", name);
   fprintf(stderr, "%s [-d {INFO|NOTICE|WARN|ERROR|FATAL}] [-o {random|seq_l|seq_h|16x16|adsm_1|adsm_2|label|quick|seq}] -m /dev/sg[a-z] [-s] [-l] [-w] [-c] [-i]\n", name);
#else
   fprintf(stderr, "%s [--debug {INFO|NOTICE|WARN|ERROR|FATAL}] [--operation {random|seq_l|seq_h|16x16|adsm_1|adsm_2|label|quick|seq}] [--direct_command {unload|load|move_medium ARM FROM TO|position ARM TO}] [--drive /dev/stX {1|2}] --medium_changer /dev/sg[a-z] [--show_elapsed]  [--no_initialization] [--time MINUTES] [--full_unload] [--check_labels] [--write] [--continuous]\n", name);
   fprintf(stderr, "%s [-d {INFO|NOTICE|WARN|ERROR|FATAL}] [-o {random|seq_l|seq_h|16x16|adsm_1|adsm_2|label|quick|seq}] -m /dev/sg[a-z] [-s] [-l] [-w] [-c]\n", name);
#endif
#ifdef PLT_PV004_COMPATIBILITY
   fprintf(stderr, "INFO: you can also specify the device /dev/sg[a-z] without the -m.\n");
#endif
#ifdef SHOW_COPYRIGHT
   fprintf(stderr, "\nCOPYRIGHT: DILOG SA 1996. Author: Marc SCHAEFER <schaefer@dilog.ch>\n");
   fprintf(stderr, "           This software contains parts copyright\n");
   fprintf(stderr, "           Marc SCHAEFER <schaefer@alphanet.ch>\n\n");
#endif
}
#else
static void usage(char *name) {
   fprintf(stderr, "%s [-debug {INFO|NOTICE|WARN|ERROR|FATAL}] /dev/sg[a-z]\n%s: bad args.\n",
                   name, name);
}
#endif

/* Private function implementation */
