/* mc.c -- Parallel Libra Tester Medium Changer Object.
 * Marc SCHAEFER <schaefer@dilog.ch>
 * Creation: 01/08/96
 * Update:   14/12/98
 * V1.0 PV010
 * NOTES
 *    - open/close sg done in scsi.c at each command.
 * BUGS
 *    - local storage returned, hard limits.
 *    - mc_tape_element_to_device not implementerd, misc all too.
 *    - xfer to UNIX device *very bad*
 *    - something bad in read_elt_status
 *    - we try to discriminate from what scsi returns, but handle_command
 *      only returns TRUE/FALSE.
 *    - only two or one drives, and should be independant of elt number.
 * TODO
 *    - we should check for mc_device null in each command
 * MODIFICATION-HISTORY
 *    05/11/96  schaefer  Adding sense information to mc_move_medium
 *    10/12/98  schaefer  Added support for adic's Position command.
 *                        Fixed Unload to use it.
 *    14/12/98  schaefer  Added (undefined) dump_data() function in read_elt().
 * $Id: mc.c,v 1.4 1998/12/14 11:34:38 schaefer Exp $
 */

#define BUGGY_SOMETHING /* in read_elt_status ... */

#undef DEBUG_DUMP_DATA /* in same */

#include <stdio.h>
#include <unistd.h>
#include "types.h"
#include "debug_if.h"
#include "mc_if.h"
#include "mc_private.h"
#include "scsi_if.h"

/* Private types */

#define TAPE_MAX_NUMBER 2

typedef struct {
   char *device_name;
   mc_element_t elt;
} mc_tape_dev_t;

/* Private function definitions */

/* NAME
 *    mc_sense_to_mc_result
 * DESCRIPTION
 *    Converts SCSI sense to MC_RESULT.
 * RESULT
 *    MC_RESULT_OK if everything ok, else mc_result_t.
 * NOTES
 * BUGS
 * TODO
 */
static mc_result_t mc_sense_to_mc_result(int s);

/* NAME
 */
#ifdef DEBUG_DUMP_DATA
/* NAME
 *    dump_data
 * DESCRIPTION
 *    Dumps some data to stderr.
 * RESULT
 *    NONE
 * NOTES
 * BUGS
 * TODO
 */
static void dump_data(char *name,
                      char *type,
                      u_char *data,
                      u_char length);
#endif /* DEBUG_DUMP_DATA */

/* Private variables */

static char *mc_device = NULL;

static mc_tape_dev_t tape_device_array[TAPE_MAX_NUMBER] = { { "/dev/st0", 0},
                                                            { "/dev/st1", 1 }
                                                          };

/* Function implementation */

void mc_set_mc_device(char *device_name) {
   if (mc_device != NULL) {
      MESSAGE(DEBUG_WARN, "plt", "mc", "mc_set_mc_device",
              "was already set, request to change accepted");
   }
   mc_device = device_name;
   MESSAGE(DEBUG_INFO, "plt", "mc", "mc_set_mc_device", device_name);
}

void mc_set_tape_device(char *device_name, mc_element_t elt) {
   MESSAGE(DEBUG_INFO, "plt", "mc", "mc_set_tape_device", device_name);
   if (elt == 1) {
      tape_device_array[0].device_name = device_name;
   }
   else {
      tape_device_array[1].device_name = device_name;
   }
}

static char local_buffer[4]; /* BEURK ! */

char *mc_element_to_string(mc_element_t e) {
   sprintf(local_buffer, "%.3d", e);
   return local_buffer; /* return * local storage intended */
}

char *mc_data_transfer_element_to_device(mc_element_t data_transfer) {
   if (data_transfer == 1) {
      return tape_device_array[0].device_name;
   }
   else {
      return tape_device_array[1].device_name;
   }
}

mc_result_t mc_get_data_transfer_elements(mc_element_t *array, int number,
                                          BOOL *tape_full_array) {
   return mc_read_element_status(NULL, 0, NULL, array, number,
                                                tape_full_array, FALSE);
}

char *mc_result_descr(mc_result_t r) {
   switch (r) {
      case MC_RESULT_OK:    return "MC_RESULT_OK";
                            break;
      case MC_RESULT_FATAL: return "MC_RESULT_FATAL";
                            break;
      case MC_RESULT_SCSI_ERROR1: return "MC_RESULT_SCSI_ERROR1";
                            break;
      case MC_RESULT_SCSI_ERROR2: return "MC_RESULT_SCSI_ERROR2";
                            break;
      case MC_RESULT_SCSI_NOT_READY: return "MC_RESULT_SCSI_NOT_READY";
                            break;
      case MC_RESULT_SCSI_HARD_ERROR: return "MC_RESULT_SCSI_HARD_ERROR";
                            break;
      case MC_RESULT_SCSI_ILLEGAL: return "MC_RESULT_SCSI_ILLEGAL";
                            break;
      case MC_RESULT_SCSI_ATTENTION: return "MC_RESULT_SCSI_ATTENTION";
                            break;
      default:              return "MC_RESULT_UNKNOWN";
                            break;
   }
   /* NOT REACHED */
}

mc_result_t mc_storage_elements(mc_element_t *array, int number,
                                BOOL *elt_full_array) {
   return mc_read_element_status(array, number, elt_full_array,
                                 NULL, 0, NULL, FALSE);
}

mc_result_t mc_initialize(void) {
   unsigned char cmd[SCSI_OFF + 18]; /* magic */
   unsigned char cmdblk[MC_SCSI_CMD_LEN_INITIALIZE_ELEMENT_STATUS];

   MESSAGE(DEBUG_INFO, "plt", "mc", "mc_initialize", "called");

   cmdblk[0] = MC_SCSI_CMD_INITIALIZE_ELEMENT_STATUS;
   cmdblk[1] = 0x00; /* LUN, RESERVED */
   cmdblk[2] = 0;
   cmdblk[3] = 0;
   cmdblk[4] = 0;
   cmdblk[5] = 0;

   memcpy( cmd + SCSI_OFF, cmdblk, sizeof(cmdblk) );

   if (!scsi_handle_scsi_cmd(mc_device, sizeof(cmdblk), 0, cmd,
	  0, NULL)) {
      return MC_RESULT_SCSI_ERROR1;
   }

   if (! ((struct sg_header *) cmd)->sense_buffer[2]) {
      return MC_RESULT_OK;
   }
   else {
      return mc_sense_to_mc_result(((struct sg_header *) cmd)->sense_buffer[2]);
   }
   /* NOT REACHED */
}

mc_result_t mc_unload(void) {
   /* Implemented by a Position_To_Element(0, 0); */

   MESSAGE(DEBUG_INFO, "plt", "mc", "mc_unload", "called");
   return mc_position(0, 0);
}

mc_result_t mc_position(mc_element_t transport_element,
                        mc_element_t destination_address) {
   unsigned char cmd[SCSI_OFF + 18]; /* magic */
   unsigned char cmdblk[MC_SCSI_CMD_LEN_POSITION_TO_ELEMENT];

   MESSAGE(DEBUG_INFO, "plt", "mc", "mc_position", "called");

   cmdblk[0] = MC_SCSI_CMD_POSITION_TO_ELEMENT;
   cmdblk[1] = 0x00; /* LUN, RESERVED */
   cmdblk[2] = HIGH(transport_element);
   cmdblk[3] = LOW(transport_element);
   cmdblk[4] = HIGH(destination_address);
   cmdblk[5] = LOW(destination_address);
   cmdblk[6] = 0;
   cmdblk[7] = 0;
   cmdblk[8] = 0;
   cmdblk[9] = 0;

   memcpy( cmd + SCSI_OFF, cmdblk, sizeof(cmdblk) );

   if (!scsi_handle_scsi_cmd(mc_device, sizeof(cmdblk), 0, cmd,
	  0, NULL)) {
      return MC_RESULT_SCSI_ERROR1;
   }

   if (! ((struct sg_header *) cmd)->sense_buffer[2]) {
      return MC_RESULT_OK;
   }
   else {
      char sense_buffer[100]; /* hacking, hacking ... */

      sprintf(sense_buffer, "SENSE: 0x%2.2x 0x%2.2x 0x%2.2x",
              ((struct sg_header *) cmd)->sense_buffer[2],
              ((struct sg_header *) cmd)->sense_buffer[12],
              ((struct sg_header *) cmd)->sense_buffer[13]);
      MESSAGE(DEBUG_WARN, "plt", "mc", "mc_position", sense_buffer);
      return mc_sense_to_mc_result(((struct sg_header *) cmd)->sense_buffer[2]);
   }
   /* NOT REACHED */
}

mc_result_t mc_move_medium(mc_element_t transport_element,
                           mc_element_t source_address,
                           mc_element_t destination_address) {
   unsigned char cmd[SCSI_OFF + 18]; /* magic */
   unsigned char cmdblk[MC_SCSI_CMD_LEN_MOVE_MEDIUM];

   MESSAGE(DEBUG_INFO, "plt", "mc", "mc_move_medium", "called");

   cmdblk[0] = MC_SCSI_CMD_MOVE_MEDIUM;
   cmdblk[1] = 0x00; /* LUN, RESERVED */
   cmdblk[2] = HIGH(transport_element);
   cmdblk[3] = LOW(transport_element);
   cmdblk[4] = HIGH(source_address);
   cmdblk[5] = LOW(source_address);
   cmdblk[6] = HIGH(destination_address);
   cmdblk[7] = LOW(destination_address);
   cmdblk[8] = 0;
   cmdblk[9] = 0;
   cmdblk[10] = 0;
   cmdblk[11] = 0;

   memcpy( cmd + SCSI_OFF, cmdblk, sizeof(cmdblk) );

   if (!scsi_handle_scsi_cmd(mc_device, sizeof(cmdblk), 0, cmd,
	  0, NULL)) {
      return MC_RESULT_SCSI_ERROR1;
   }

   if (! ((struct sg_header *) cmd)->sense_buffer[2]) {
      return MC_RESULT_OK;
   }
   else {
      char sense_buffer[100]; /* hacking, hacking ... */

      sprintf(sense_buffer, "SENSE: 0x%2.2x 0x%2.2x 0x%2.2x",
              ((struct sg_header *) cmd)->sense_buffer[2],
              ((struct sg_header *) cmd)->sense_buffer[12],
              ((struct sg_header *) cmd)->sense_buffer[13]);
      MESSAGE(DEBUG_WARN, "plt", "mc", "mc_move_medium", sense_buffer);
      return mc_sense_to_mc_result(((struct sg_header *) cmd)->sense_buffer[2]);
   }
   /* NOT REACHED */
}

mc_result_t mc_read_element_status(mc_element_t *stg_elt_array,
                                   int num_stg_elt,
                                   BOOL *full_stg_elt_array,
                                   mc_element_t *xfer_elt_array,
                                   int num_xfer_elt,
                                   BOOL *full_xfer_array,
                                   BOOL dump) {
   unsigned char elementbuffer[SCSI_OFF + READ_ELEMENT_STATUS_ALLOCATION_LEN];
   unsigned char cmd[SCSI_OFF + 18]; /* magic */
   unsigned char cmdblk[MC_SCSI_CMD_LEN_READ_ELEMENT_STATUS];
   char *current;
   long int size;
   int first, last;
   int seen_stg = 0;
   int seen_xfer = 0;

   if (stg_elt_array) {
      int i;

      for (i = 0; i < num_stg_elt; i++) {
         stg_elt_array[i] = MC_ELEMENT_INVALID;
      }
   }
   if (xfer_elt_array) {
      int i;

      for (i = 0; i < num_xfer_elt; i++) {
        xfer_elt_array[i] = MC_ELEMENT_INVALID;
      }
   }

   cmdblk[0] = MC_SCSI_CMD_READ_ELEMENT_STATUS;
   cmdblk[1] = 0x00; /* LUN, RESERVED */
   cmdblk[2] = 0; /* Starting element */
   cmdblk[3] = 0; /* address          */
   cmdblk[4] = 0xFF; /* Number of */
   cmdblk[5] = 0xFF; /* elements  */
   cmdblk[6] = 0;
   cmdblk[7] = HIGHEST(READ_ELEMENT_STATUS_ALLOCATION_LEN);  /* Allocation */
   cmdblk[8] = HIGH(READ_ELEMENT_STATUS_ALLOCATION_LEN);     /* length     */
   cmdblk[9] = LOW(READ_ELEMENT_STATUS_ALLOCATION_LEN);      /*            */
   cmdblk[10] = 0;
   cmdblk[11] = 0;

   memcpy( cmd + SCSI_OFF, cmdblk, sizeof(cmdblk) );

   if (!scsi_handle_scsi_cmd(mc_device, sizeof(cmdblk), 0, cmd,
	  sizeof(elementbuffer) - SCSI_OFF, elementbuffer)) {
      MESSAGE(DEBUG_ERROR, "plt", "mc", "scsi_handle_scsi_cmd", "failed");
      return MC_RESULT_SCSI_ERROR1;
   }

#ifndef BUGGY_SOMETHING
   if (! ((struct sg_header *) cmd)->sense_buffer[2]) {
#else
   if (1) {
#endif
      current = &elementbuffer[SCSI_OFF];


#ifdef DEBUG_DUMP_DATA
      dump_data("mc", "R_E_S",
                current,
                100);
#endif /* DEBUG_DUMP_DATA */
               

      first = EXPAND(0, current[0], current[1]);
      last = first + EXPAND(0, current[2], current[3]) - 1;
      if (dump) {
         printf("\nRange of elements returned: [%d-%d]\n", first, last);
      }

      size = EXPAND(current[5], current[6], current[7]);

      current = &current[8];

      while (size > 0) {
	 char *temp = &current[8];
	 int len = EXPAND(current[5], current[6], current[7]);
	 int block_len = EXPAND(0, current[2], current[3]); /* badly documented */
	 switch (current[0]) {
	    case 1:
                    if (dump) {
                       printf("   Medium Transport");
		    }
		    while (len >= 12) {
                       if (dump) {
 		          printf("\n      Address: %d", EXPAND(0, temp[0], temp[1]));
                       }
		      temp += block_len;
		      len -= block_len;
		    }
                    if (dump) {
   		       printf("\n");
                    }
		    break;
	    case 2:
                    if (dump) {
                       printf("   Storage Element");
                    }
		    while (len >= 12) {
                      if (dump) {
   		         printf("\n      Address: %d", EXPAND(0, temp[0], temp[1]));                         
			 if (temp[2] & 1) {
			    printf(" Is full");
			 }
			 else {
			    printf(" Is empty");
			 }
			 if (temp[2] & 2) {
			    printf(" Is reserved");
			 }
			 else {
			    printf(" Is not reserved");
			 }
			 printf(" Source storage: %d", EXPAND(0, temp[10], temp[11]));
                      }
                      if ((seen_stg < num_stg_elt) && stg_elt_array) {
                         stg_elt_array[seen_stg] = EXPAND(0, temp[0], temp[1]);
                      }
                      if ((seen_stg < num_stg_elt) && full_stg_elt_array) {
                         full_stg_elt_array[seen_stg] = (temp[2] & 1) ? TRUE : FALSE;
                      }
                      seen_stg++;

		      temp += block_len;
		      len -= block_len;
		    }
                    if (dump) {
   		       printf("\n");
                    }
		    break;
	    case 4:
                    if (dump) {
                       printf("   Data Transfer Element");
                    }
		    while (len >= 12) {
                       if (dump) {
			  printf("\n      Address: %d", EXPAND(0, temp[0], temp[1]));
			  if (temp[2] & 1) {
			     printf(" Is full");
			  }
			  else {
			     printf(" Is empty");
			  }
			  printf(" SCSI Address: %d", temp[7]);
			  printf(" Source storage: %d", EXPAND(0, temp[10], temp[11]));
                      }
                      if ((seen_xfer < num_xfer_elt) && xfer_elt_array) {
                         xfer_elt_array[seen_xfer] = EXPAND(0, temp[0], temp[1]);
                      }
                      if ((seen_xfer < num_xfer_elt) && full_xfer_array) {
			 full_xfer_array[seen_xfer] = (temp[2] & 1) ? TRUE : FALSE;
                      }
                      seen_xfer++;
		      temp += block_len;
		      len -= block_len;
		    }
                    if (dump) {
   		       printf("\n");
                    }
		    break;
	    default:
                     if (dump) {
                        printf("   UNKNOWN\n");
                     }
		     len = -1;
		     break;
	 }

	 len = EXPAND(current[5], current[6], current[7]);
	 size -= len + 8; /* 8 == sizeof(header) */
	 current += len + 8;
      }

      return MC_RESULT_OK;
   }
   else {
      MESSAGE(DEBUG_ERROR, "plt", "mc", "scsi_handle_scsi_cmd", "failed 2");
      return mc_sense_to_mc_result(((struct sg_header *) cmd)->sense_buffer[2]);
   }
   /* NOT REACHED */
}

mc_result_t mc_inquiry(char *buffer, int *len) {
   unsigned char cmd[SCSI_OFF + 18]; /* magic */
   unsigned char cmdblk[MC_SCSI_CMD_LEN_INQUIRY];
   unsigned char temp[SCSI_OFF + MC_SCSI_CMD_REPLYLEN_INQUIRY];

   if (*len > MC_SCSI_CMD_REPLYLEN_INQUIRY) {
      *len = MC_SCSI_CMD_REPLYLEN_INQUIRY;
    }

   MESSAGE(DEBUG_INFO, "plt", "mc", "mc_inquiry", "called");

   cmdblk[0] = MC_SCSI_CMD_INQUIRY;
   cmdblk[1] = 0x00; /* LUN, RESERVED */
   cmdblk[2] = 0;
   cmdblk[3] = 0;
   cmdblk[4] = *len;
   cmdblk[5] = 0;

   memcpy( cmd + SCSI_OFF, cmdblk, sizeof(cmdblk) );

   if (!scsi_handle_scsi_cmd(mc_device, sizeof(cmdblk), 0, cmd,
	  *len, temp)) {
      return MC_RESULT_SCSI_ERROR1;
   }

   /* BUGS
    *   - was return (Inqbuffer + SCSI_OFF); in libra2.c
    */
   if (! ((struct sg_header *) cmd)->sense_buffer[2]) {
      memcpy(buffer, temp + SCSI_OFF, (*len > (sizeof(temp) - SCSI_OFF))
                                      ? (sizeof(temp) - SCSI_OFF) : *len);
      return MC_RESULT_OK;
   }
   else {
      return mc_sense_to_mc_result(((struct sg_header *) cmd)->sense_buffer[2]);
   }
   /* NOT REACHED */
}

/* Private function implementation */

static mc_result_t mc_sense_to_mc_result(int s) {
   switch (s) {
      case 0: return MC_RESULT_OK; break;
      case 2: return MC_RESULT_SCSI_NOT_READY; break;
      case 4: return MC_RESULT_SCSI_HARD_ERROR; break;
      case 5: return MC_RESULT_SCSI_ILLEGAL; break;
      case 6: return MC_RESULT_SCSI_ATTENTION; break;
      default: return MC_RESULT_SCSI_ERROR2; break;
   }
   /* NOT REACHED */
}

#ifdef DEBUG_DUMP_DATA
static void dump_data(char *name,
                      char *type,
                      u_char *data,
                      u_char length) {
   if (length) {
      printf("%s: dumping %s: ", name, type);

      while (length--) {
         printf("0x%x%s", *data, length ? " " : "");
         data++;
      }

      printf(".\n");
   }
}
#endif /* DEBUG_DUMP_DATA */
