/* mt_operations.c -- Common ``mt''-like operations.
 * Marc SCHAEFER <schaefer@dilog.ch>
 * Creation: 12/11/96
 * Update:   12/11/96
 * V1.0 PV001
 * NOTES
 *    - It seems we have to close/reopen fd if we want accurate information.
 *      Well, that might be untrue.
 *    - Using or sequencement C assumption.
 * BUGS
 *    - c_ronly is not clear.
 * TODO
 *    - what about a mt_open/mt_close interface with private keys ?
 * COPYRIGHT
 *    Most of this is taken from mt.c,
 *       Copyright (c) 1980 The Regents of the University of California.
 *       All rights reserved.
 * MODIFICATION-HISTORY
 *    10/12/98  schaefer Applying new RCS scheme.
 * $Id: mt_operations.c,v 1.2 1999/04/01 16:07:11 schaefer Exp $
 */
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mtio.h>
#include <fcntl.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/time.h>
#include "types.h"
#include "debug_if.h"
#include "mt_operations_if.h"

#ifndef USE_MT_SCRIPT

/* Types definitions */

#define USLEEP_LENGTH 5*1000000
#define MAX_ERRORS 10

#undef DEBUG

/* Private function definitions */

/* NAME
 *    mt_open_tape
 * DESCRIPTION
 *    Opens the tape device
 * RESULT
 *    fd_t or -1 if error
 * NOTES
 * BUGS
 * TODO
 */
static fd_t mt_open_tape(char *tape, BOOL c_ronly);

/* NAME
 *    mt_get_status
 * DESCRIPTION
 *    Get the tape status.
 * RESULT
 *    TRUE if structure valid.
 * NOTES
 * BUGS
 * TODO
 */
static BOOL mt_get_status(char *tape, struct mtget *mt);

/* Function implementation */

BOOL mt_do(char *tape) {
   if (mt_is_online(tape, TRUE)) {
      return mt_setblk(tape, 0); /* variable block size */
   }
   else {
      return FALSE;
   }
}

BOOL mt_is_online(char *tape, BOOL do_wait) {
   BOOL result = FALSE;
   struct mtget m;
   int error_count = 0;

   do {
      if (!mt_get_status(tape, &m)) {
         error_count++;
      }
      else {
         if (GMT_ONLINE((m.mt_gstat))) {
            result = TRUE;
         }
         else {
            MESSAGE(DEBUG_INFO, "mt", "mt_is_online", tape,
                    "is not yet online");
#ifdef DEBUG
            printf("Status: %8.8lx.\n", m.mt_gstat);
#endif
         }
      }
      
      if (!result && do_wait) {
         usleep(USLEEP_LENGTH);
      }
   } while ((do_wait && !result) && (error_count < MAX_ERRORS));

   return result;
}

BOOL mt_setblk(char *tape, int block_size) {
   BOOL result = FALSE;
   fd_t mtfd;
   struct mtop m;
   int error_count = 0;

   do {
      mtfd  = mt_open_tape(tape, TRUE);

      m.mt_op = MTSETBLK;
      m.mt_count = block_size;

      if ((mtfd == -1) || (ioctl(mtfd, MTIOCTOP, (char *) &m) < 0)) {
	 MESSAGE(DEBUG_ERROR, "mt", "mt_setblk", "MTIOCTOP",
		 debug_perror());
	 error_count++;
      }
      else {
	 result = TRUE;
      }

      if (mtfd != -1) {
	 close(mtfd);
      }

   } while (!result && (error_count < MAX_ERRORS));

   return result;
}

/* NAME
 *    mt_get_retry_count
 * DESCRIPTION
 *    Returns the retry count of the drive.
 * RESULT
 *    -1 if no retry count could be returned.
 * NOTES
 * BUGS
 * TODO
 */
int mt_get_retry_count(char *tape) {
   struct mtget m;

   if (!mt_get_status(tape, &m)) {
      return -1;
   }
   else {
      return (m.mt_erreg & MT_ST_SOFTERR_MASK) >> MT_ST_SOFTERR_SHIFT;
   }
   /* NOT REACHED */
}

/* Private function implementation */

static fd_t mt_open_tape(char *tape, BOOL c_ronly) {
   fd_t mtfd;

   mtfd = open(tape, c_ronly ? O_RDONLY : O_RDWR);

   if (mtfd == -1) {
      MESSAGE(DEBUG_ERROR, "mt", "mt_open_tape", "open",
              debug_perror());
   }

   return mtfd;
}

static BOOL mt_get_status(char *tape, struct mtget *mt) {
   fd_t mtfd;
   BOOL result = FALSE;

   mtfd = mt_open_tape(tape, TRUE);

   if ((mtfd == -1) || (ioctl(mtfd, MTIOCGET, (char *) mt) < 0)) {
      MESSAGE(DEBUG_ERROR, "mt", "mt_get_status", (mtfd == -1)
                                                 ? "no fd" : "MTIOCGET",
              debug_perror());
   }
   else {
      result = TRUE;
   }

   if (mtfd != -1) {
      close(mtfd);
   }

   return result;
}

#endif /* !USE_MT_SCRIPT */