/* plt.c -- Parallel Libra Tester main engine.
 * Marc SCHAEFER <schaefer@dilog.ch>
 * Creation: 01/08/96
 * Update:   14/12/98
 * V1.0 PV012
 * NOTES
 *    - random() is not initialized, so the sequence is always the same.
 *      This is a feature.
 * BUGS
 *    - Should use dynamic lists and pointers. Should not use
 *      fixed length anywhere.
 *    - Cleanup not done: fatal error supposes usually quitting the
 *      program. So we don't close files, etc.
 *    - if one of the tape quits, buggy.
 * TODO
 *    - timeout de 5 sec, et reboucle pour random load/unload
 *    - random load/unload disable posisbility
 * MODIFICATION-HISTORY
 *    10/12/98  schaefer  Added support for POSITION (for Adic).
 *    14/12/98  schaefer  Added support for non random movement. Implemented
 *                        sequential UP, DOWN then UP again. #ifdef'd out
 *                        the unimplemented modes.
 * $Id: plt.c,v 1.3 1999/04/19 15:09:09 schaefer Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <string.h>
#include "types.h"
#include "debug_if.h"
#include "tape_if.h"
#include "mc_if.h"
#include "time_if.h"
#include "plt_if.h"

#ifdef __sun
long random();
#endif /* __sun */

#undef  FAKE_MOVES

#define RETURN_BEFORE_STATS
#define MAXLINELENGTH 80 /* only used for dumb gets() */

/* External variables */

extern BOOL do_label;
extern BOOL do_full_unload;
extern unsigned short max_time;

/* Private type definitions */

#define MAXBUFLEN 100 /* for keyboard input */

#define PLT_MAX_STORAGE_ELEMENTS 16
#define PLT_RANDOM_LOAD_UNLOAD 50

#define SLOT_OFFSET (tape_number + 1) /* this MUST disappear */

#define PLT_MAX_TAPES 2
#define PLT_TAPE_DESCR_LENGTH 30
typedef struct {
   fd_t result_pipe;  /* From medium changer object to tape object */
   fd_t command_pipe; /* From tape object to medium changer object */
   mc_element_t element; /* Tape drive element number in medium changer
                          * point of view.
                          */
} plt_tape_t;

typedef struct {
   int load_count;
   int slot_activity_count[PLT_MAX_TAPES][PLT_MAX_STORAGE_ELEMENTS];
} plt_stats_t;

/* Private function definitions */

/* NAME
 *    plt_load
 * DESCRIPTION
 *    This functions executes the mc_initialize function.
 * RESULT
 *    mc_result_t
 * NOTES
 * BUGS
 * TODO
 */
static mc_result_t plt_load(void);

/* NAME
 *    plt_unload
 * DESCRIPTION
 *    This functions executes the mc_unload function.
 * RESULT
 *    mc_result_t
 * NOTES
 * BUGS
 * TODO
 */
static mc_result_t plt_unload(void);

/* NAME
 *    plt_move_medium
 * DESCRIPTION
 *    This functions executes the mc_move_medium function.
 * RESULT
 *    mc_result_t
 * NOTES
 * BUGS
 * TODO
 */
static mc_result_t plt_move_medium(int arg1, int arg2, int arg3);

/* NAME
 *    plt_move_medium
 * DESCRIPTION
 *    This functions executes the mc_position function.
 * RESULT
 *    mc_result_t
 * NOTES
 * BUGS
 * TODO
 */
static mc_result_t plt_position(int arg1, int arg2);

/* NAME
 *    plt_main_loop
 * DESCRIPTION
 *    This is the main loop. We wait for messages from
 *    the tape(s) object(s) with a select(2) and we
 *    call the mc routines to get or give tapes. We
 *    communicate results to the tape object.
 * RESULT
 * NOTES
 *    - There is one MC unit which can only process one
 *      request at a time, thus once we receive a request,
 *      we must synchronously execute it. No other messages
 *      will be processed.
 * BUGS
 * TODO
 */
static plt_test_result_t plt_main_loop(void);

/* NAME
 *    plt_get_tape
 * DESCRIPTION
 *    We must found a full slot (with a specified algorithm) and
 *    then Move_Medium to the tape device.
 * RESULT
 * NOTES
 *    - No validity checks
 * BUGS
 *    - infinite loop is possible with badly full pack
 * TODO
 */
static plt_test_result_t plt_get_tape(int tape_num, mc_element_t xfer_element);

/* NAME
 *    plt_give_tape
 * DESCRIPTION
 *    We move the tape from the drive to an algorithm-generated empty slot.
 * RESULT
 * NOTES
 *    - No validity checks
 * BUGS
 *    - infinite loop is possible with badly full pack
 * TODO
 */
static plt_test_result_t plt_give_tape(int tape_num, mc_element_t xfer_element);

/* NAME
 *    plt_send_result
 * DESCRIPTION
 *    Sends a result through a result pipe, translating type
 *    BOOL and int to plt_result_t in plt_to_tape_t;
 * RESULT
 *    NONE
 * NOTES
 * BUGS
 *    - No way of saying error
 * TODO
 */
static void plt_send_result(BOOL result, int arg, fd_t result_pipe);

/* NAME
 *    plt_dump_stats
 * DESCRIPTION
 *    This function dumps the overall statistics.
 * RESULT
 *    NONE
 * NOTES
 *    - Does not consider debug level.
 * BUGS
 * TODO
 */
static void plt_dump_stats(void);

/* NAME
 *    check_nothing_in_tapes
 * DESCRIPTION
 *    This function checks if all tapes are empty.
 * RESULT
 *    TRUE if all tapes are empty
 * NOTES
 *    - Main use is for deciding unload if ! --full_unload.
 * BUGS
 * TODO
 */
static BOOL check_nothing_in_tapes(void);

/* NAME
 *    return_random
 * DESCRIPTION
 *    Function returning a random delta, upto modulo - 1.
 * RESULT
 *    mc_element_t
 * NOTES
 * BUGS
 * TODO
 */
static mc_element_t return_random(mc_element_t modulo, BOOL put);

/* NAME
 *    return_sequential
 * DESCRIPTION
 *    Function returning a sequential delta, upto modulo - 1.
 * RESULT
 *    mc_element_t
 * NOTES
 * BUGS
 * TODO
 */
static mc_element_t return_sequential(mc_element_t modulo, BOOL put);

/* Private variables */

static plt_tape_t plt_tapes[PLT_MAX_TAPES];
static int tape_number = 0;
static int slot_number = PLT_MAX_STORAGE_ELEMENTS; /* for pre-init */
static plt_stats_t the_stats;
static BOOL interrupted = FALSE;

mc_element_t last_slot = 0;
#define DIRECTION_UP 0
#define DIRECTION_DOWN 1
#define DIRECTION_LAST_TIME 2
#define DIRECTION_IS_UP(d) (!((d) & 1))
int direction = DIRECTION_UP;
static mc_element_t (*slot_function)(mc_element_t modulo, BOOL put)
   = return_random;

/* Function implementation */

BOOL plt_initialize(BOOL get_tapes, BOOL skip_initialization) {
   mc_element_t tape_element_array[PLT_MAX_TAPES];
   BOOL tape_full_array[PLT_MAX_TAPES];
   mc_result_t mc_result;
   fd_t pipes1[2];
   fd_t pipes2[2];
   int i, j;
   char buffer[MAXBUFLEN];
   int len = MAXBUFLEN;

   for (i = 0; i < PLT_MAX_STORAGE_ELEMENTS; i++) {
      for (j = 0; i < tape_number; j++) {
         the_stats.slot_activity_count[j][i] = 0;
      }
   }

   mc_result = mc_inquiry(buffer, &len);

   if (mc_result != MC_RESULT_OK) {
      MESSAGE(DEBUG_ERROR, "plt", "plt_init", "mc_inquiry",
              MC_RESULT_DESCR(mc_result));
      return FALSE;
   }

   /* BUGS
    *    - Not supposed to be null-terminated, so we use this hack.
    */
   buffer[MC_SCSI_INQUIRY_VENDOR + 15] = 0;

   MESSAGE(DEBUG_INFO, "plt", "plt_init", "INQUIRY", buffer + MC_SCSI_INQUIRY_VENDOR);

   the_stats.load_count = 1; /* implicit */

#ifdef ENABLE_INITIALIZE
   if (!skip_initialization) {
      MESSAGE(DEBUG_INFO, "plt", "plt_init", "mc_initialize", "Initialize Element Status");

      mc_result = mc_initialize();

      if (mc_result != MC_RESULT_OK) {
	 MESSAGE(DEBUG_ERROR, "plt", "plt_init", "mc_initialize",
		 MC_RESULT_DESCR(mc_result));
	 return FALSE;
      }   
   }
#endif

   mc_result = mc_get_data_transfer_elements(tape_element_array,
                                             PLT_MAX_TAPES, tape_full_array);
   if (mc_result != MC_RESULT_OK) {
      MESSAGE(DEBUG_ERROR, "plt", "plt_init", "mc_get_tape_elements",
              MC_RESULT_DESCR(mc_result));
      return FALSE;
   }

   for (i = 0; i < PLT_MAX_TAPES; i++) {
      if (tape_element_array[i] != MC_ELEMENT_INVALID) {
         MESSAGE(DEBUG_INFO, "plt", "plt_init", "identified tape at element",
                 MC_ELEMENT_TO_STRING(tape_element_array[i]));
         tape_number++;
         plt_tapes[tape_number - 1].element = tape_element_array[i];
         if (pipe(pipes1) == -1) {
            MESSAGE(DEBUG_FATAL, "plt", "plt_init", "pipe(pipes1)", 
                    MESSAGE_PERROR);
            return FALSE;
         }
         else {
            if (pipe(pipes2) == -1) {
               MESSAGE(DEBUG_FATAL, "plt", "plt_init", "pipe(pipes2)", 
                       MESSAGE_PERROR);
               return FALSE;
            }
            else {
               char *data_transfer_element
                   = mc_data_transfer_element_to_device
                        (plt_tapes[tape_number - 1].element);

               if (data_transfer_element) {
		  plt_tapes[tape_number - 1].result_pipe = pipes1[1];
		  plt_tapes[tape_number - 1].command_pipe = pipes2[0];
		  if (tape_instanciate(pipes1[0], pipes2[1], 
				       pipes1[1], pipes2[0],
				       data_transfer_element)) {
		     MESSAGE(DEBUG_INFO, "plt", "plt_init",
			     "tape object", "instanciated");

                     if (get_tapes && tape_full_array[tape_number - 1]) {
		        MESSAGE(DEBUG_INFO, "plt", "plt_init",
			        "tape object", "is full; emptying");
                        plt_give_tape(tape_number - 1,
                                      plt_tapes[tape_number - 1].element);
                        /* Ignoring the result. May not be a good thing
                         * anyway.
                         */

                     }
		  }
		  else {
		     MESSAGE(DEBUG_FATAL, "plt", "plt_init",
                             "tape_instanciate", 
			     "failed");
		     return FALSE;
		  }
	       }
               else {
		     MESSAGE(DEBUG_FATAL, "plt", "plt_init",
                             "mc_data_transfer_element_to_device", 
			     "returned NULL");
		     return FALSE;
               }
            }   
         }
      }
      else {
         break;
      }
   }

   if (tape_number) {
      mc_result_t r;
      mc_element_t stg_array[PLT_MAX_STORAGE_ELEMENTS];

      r = mc_storage_elements(stg_array, PLT_MAX_STORAGE_ELEMENTS,
                              NULL);

      if (r == MC_RESULT_OK) {
         int i;

         slot_number = 0;

         for (i = 0; i < PLT_MAX_STORAGE_ELEMENTS; i++) {
            if (stg_array[i] != MC_ELEMENT_INVALID) {
               slot_number++;
            }
         }
         return TRUE;
      }
      else {
         MESSAGE(DEBUG_ERROR, "plt", "plt_init", "no elements",
                 MC_RESULT_DESCR(r));
         return FALSE;
      }
   }
   else {
     MESSAGE(DEBUG_ERROR, "plt", "plt_init", "no tapes", "");
     return FALSE;
   }
}

plt_test_result_t plt_do(plt_do_t what) {
   plt_test_result_t r;

   switch (what) {
      case PLT_DO_RANDOM:
         slot_function = return_random;
         if (do_label) {
            MESSAGE(DEBUG_WARN, "plt", "plt_do",
             "--check_labels is incompatible with the RANDOM test currently.",
             "do_label disabled");
             do_label = FALSE;
         }
         else {
            r = plt_main_loop();
         }
         break;

      case PLT_DO_SEQ:
         slot_function = return_sequential;
         r = plt_main_loop();
         break;
#if 0
      case PLT_DO_QUICK:
         break;
      case PLT_DO_SEQUENTIAL_H:
         break;
      case PLT_DO_SEQUENTIAL_L:
         break;
      case PLT_DO_16x16:
         break;
      case PLT_DO_ADSM_1:
         break;
      case PLT_DO_ADSM_2:
         break;
      case PLT_DO_LABEL:
         break;
#endif
      default:
        MESSAGE(DEBUG_FATAL, "plt", "plt_do",
             "Internal inconsistency", "function not implemented");
        r = TEST_FATAL;
        break; 
   }

#ifdef RETURN_BEFORE_STATS
   {
      char buffer[MAXLINELENGTH];

      printf("<Type RETURN for statistics>");
      fflush(stdout);

      gets(buffer);
   }
   
#endif
   plt_dump_stats();

   return r;
}

BOOL plt_direct_command(plt_direct_cmd_t cmd) {
   BOOL result = TRUE;

   switch (cmd.cmd) {
      case PLT_CMD_UNLOAD:
         if (plt_unload() != MC_RESULT_OK) {
            result = FALSE;
         }
         break;
      case PLT_CMD_LOAD:
         if (plt_load() != MC_RESULT_OK) {
            result = FALSE;
         }
         break;
      case PLT_CMD_MOVE_MEDIUM:
         if (plt_move_medium(cmd.arg1, cmd.arg2, cmd.arg3) != MC_RESULT_OK) {
            result = FALSE;
         }
         break;
      case PLT_CMD_POSITION:
         if (plt_position(cmd.arg1, cmd.arg2) != MC_RESULT_OK) {
            result = FALSE;
         }
         break;
      default:
        MESSAGE(DEBUG_FATAL, "plt", "plt_direct_command",
             "Internal inconsistency", "");
        result = FALSE;
        break; 
   }

   return result;
}

/* Private function implementation */
   
static plt_test_result_t plt_main_loop(void) {
   int i;
   plt_test_result_t result = TEST_SUCCEEDED;
   int numfds = 0; /* magic number */
   fd_set readfds, writefds, exceptfds;
   struct timeval timeout;
   BOOL ok = TRUE;
   int ending = 0;

   while (ok && (tape_number) && (ending < tape_number) && !interrupted
          && ((time_elapsed_time() / 60) < max_time)) {
      FD_ZERO(&readfds);
      FD_ZERO(&writefds);
      FD_ZERO(&exceptfds);
      timerclear(&timeout);

      if (!(random() % PLT_RANDOM_LOAD_UNLOAD)) {
         /* NOTES
          *    - Libra8 does not support ejection when the drive is
          *      full. TwinLibra does. Default is not to do unloads
          *      if full, except if --full_unload is given.
          */
	 if (do_full_unload || (check_nothing_in_tapes())) {
	    MESSAGE(DEBUG_INFO, "plt", "plt", "plt_main_loop", "reloading");
	    if (plt_unload() == MC_RESULT_OK) {
	       if (plt_load() == MC_RESULT_OK) {
		  the_stats.load_count++;
	       }
	       else {
		  ok = FALSE;
		  result = TEST_FAILED;
		  break;
	       }
	    }
	    else {
	       ok = FALSE;
	       result = TEST_FAILED;
	       break;
	    }
         }
         MESSAGE(DEBUG_INFO, "plt", "plt", "plt_main_loop", "reloading done");
      }

      /* Enable selected input */

      for (i = 0; i < tape_number; i++) { /* changed from 0 */
         FD_SET(plt_tapes[i].command_pipe, &readfds);
         numfds = MAX(plt_tapes[i].command_pipe + 1, numfds);
         MESSAGE(DEBUG_INFO, "plt", "plt", "plt_main_loop",
                 "registered one input event pipe");
      }

      FD_SET(0, &readfds); /* stdin */

      if (select(numfds, &readfds, &writefds, &exceptfds, NULL) /*&timeout) */
          == -1) {
         MESSAGE(DEBUG_FATAL, "plt", "plt", "select()", 
                 MESSAGE_PERROR);
         ok = FALSE;
         result = TEST_FATAL;
      }
      else {
 	 if (FD_ISSET(0, &readfds)) {
            char buffer[MAXBUFLEN];

            if (read(0, buffer, MAXBUFLEN) > 0) {
               if (buffer[0] == 'q') {
                  interrupted = TRUE;
                  MESSAGE(DEBUG_NOTICE, "plt", "plt", "plt_main_loop",
	   	       "user interrupted");
                  continue;
               }
               else {
                  MESSAGE(DEBUG_INFO, "plt", "plt", "plt_main_loop",
	   	       "ignored stdin garbage");
               }
            }
            else {
               MESSAGE(DEBUG_WARN, "plt", "plt", "can't read std input",
		       MESSAGE_PERROR);
            }
         }

         MESSAGE(DEBUG_INFO, "plt", "plt", "plt_main_loop",
		    "processing events");

	 for (i = 0; i < tape_number; i++) {
	    if (FD_ISSET(plt_tapes[i].command_pipe, &readfds)) {
               plt_cmd_t the_command;

               if (read(plt_tapes[i].command_pipe, &the_command,
                        sizeof(the_command)) == sizeof(the_command)) {
                  MESSAGE(DEBUG_INFO, "plt", "plt", "plt_main_loop executing",
                          PLT_PLT_CMD_TO_STRING(the_command));
                  switch (the_command) {
                     case PLT_CMD_GET_TAPE: result =
                                            plt_get_tape(i, plt_tapes[i].element);
                                            if (result != TEST_SUCCEEDED) {
                                               ok = FALSE;
                                            }
                                            plt_send_result(ok,
                                                             plt_tapes[i].element, plt_tapes[i].result_pipe);
                                            break;
                     case PLT_CMD_GIVE_TAPE: result =
                                             plt_give_tape(i, plt_tapes[i].element);
                                             if (result != TEST_SUCCEEDED) {
                                                ok = FALSE;
                                             }
                                             plt_send_result(ok,
                                                             plt_tapes[i].element, plt_tapes[i].result_pipe);
                                             break;
                     case PLT_CMD_FAIL:  result = TEST_FAILED;
                                         ok = FALSE;
                                         break;
                     case PLT_CMD_FATAL: result = TEST_FATAL;
                                         ok = FALSE;
                                         break;
                     case PLT_CMD_ENDING: ending++;
                                          break;
                     default: MESSAGE(DEBUG_FATAL, "plt", "plt",
                                      "plt_main_loop", 
			              "Unknown command");
			      ok = FALSE;
			      result = TEST_FATAL;
                              break;
                  }
               }
               else {
		  MESSAGE(DEBUG_FATAL, "plt", "plt", "read()", 
			  MESSAGE_PERROR);
		  ok = FALSE;
		  result = TEST_FATAL;
               }
            }
	 }
      }
   }

   if ((time_elapsed_time() / 60) >= max_time) {
      MESSAGE(DEBUG_NOTICE, "plt", "plt", "time", "expired");
    }

   if (do_full_unload || (check_nothing_in_tapes())) {
      plt_unload(); /* result ignored */
   }

   /* ... send ending to both tapes ... */
   return result;
}

static mc_element_t return_random(mc_element_t modulo, BOOL put) {
   return random() % modulo;
}

static mc_element_t return_sequential(mc_element_t modulo, BOOL put) {
   /* only works with all slots full */

   if (put) {
      mc_element_t value = last_slot;

      if (DIRECTION_IS_UP(direction)) {
         mc_element_t real_modulo = modulo - 1;
         last_slot++;

         if (direction & DIRECTION_LAST_TIME) {
            real_modulo = modulo;
         }
	 
	 if (last_slot >= real_modulo) {
	    if (direction & DIRECTION_LAST_TIME) {
	 	  interrupted = TRUE; /* some hacking */
	    }
	    direction = DIRECTION_DOWN;
	    last_slot = 1;
	 }
      }
      else {
         if (last_slot >= (modulo - 1)) {
            last_slot = 0;
            direction = DIRECTION_UP | DIRECTION_LAST_TIME;
         }
         else {
            last_slot++;
         }

         value = modulo - value;
      }
      return value;
   }
   else {
      return 0;
   }
   /* NOT REACHED */
}

static plt_test_result_t plt_get_tape(int tape_num,
                                      mc_element_t xfer_element) {
   mc_element_t stg_array[PLT_MAX_STORAGE_ELEMENTS];
   BOOL elt_full_array[PLT_MAX_STORAGE_ELEMENTS];

   MESSAGE(DEBUG_INFO, "plt", "plt", "plt_get_tape", "entering");

   if (mc_storage_elements(stg_array, PLT_MAX_STORAGE_ELEMENTS,
                           elt_full_array)
       == MC_RESULT_OK) {
      int i = 0;
      int j = slot_function(slot_number, TRUE);
      mc_element_t current;

      /* NOTES
       *    We want to have the j'st' full element
       */

      while (j >= 0) {
         current = MC_ELEMENT_INVALID;
         while (current == MC_ELEMENT_INVALID) {
            if ((stg_array[i] != MC_ELEMENT_INVALID)
                && (i < PLT_MAX_STORAGE_ELEMENTS)) {
               if (elt_full_array[i]) {
                  current = stg_array[i];
               }
               i++;
            }
            else {
               i = 0;
            }
         }
         j--;
      }

      if (current == MC_ELEMENT_INVALID) {
         return TEST_FAILED;
      }

      MESSAGE(DEBUG_INFO, "plt", "plt", "plt_get_tape from",
              MC_ELEMENT_TO_STRING(current));
      MESSAGE(DEBUG_INFO, "plt", "plt", "plt_get_tape to",
              MC_ELEMENT_TO_STRING(xfer_element));

      if (plt_move_medium(0, current, xfer_element) == MC_RESULT_OK) {
         the_stats.slot_activity_count[tape_num][current - SLOT_OFFSET]++;
         return TEST_SUCCEEDED;
      }
      else {
         MESSAGE(DEBUG_ERROR, "plt", "plt", "plt_move_medium", "failed");
         return TEST_FAILED;
      }
      /* NOT REACHED */
   }
   else {
      MESSAGE(DEBUG_ERROR, "plt", "plt", "mc_get_storage_elements", "failed");
      return TEST_FAILED;
   }
   /* NOT REACHED */
}

static plt_test_result_t plt_give_tape(int tape_num,
                                       mc_element_t xfer_element) {
   mc_element_t stg_array[PLT_MAX_STORAGE_ELEMENTS];
   BOOL elt_full_array[PLT_MAX_STORAGE_ELEMENTS];

   MESSAGE(DEBUG_INFO, "plt", "plt", "plt_give_tape", "entering");

#ifndef FAKE_MOVES
   if (mc_storage_elements(stg_array, PLT_MAX_STORAGE_ELEMENTS,
                           elt_full_array)
       == MC_RESULT_OK) {
      int i = 0;
      int j = slot_function(slot_number, FALSE);
      mc_element_t current;

      /* NOTES
       *    We want to have the j'st' empty element
       */

      while (j >= 0) {
         current = MC_ELEMENT_INVALID;
         while (current == MC_ELEMENT_INVALID) {
            if ((stg_array[i] != MC_ELEMENT_INVALID)
                && (i < PLT_MAX_STORAGE_ELEMENTS)) {
               if (!elt_full_array[i]) {
                  current = stg_array[i];
               }
               i++;
            }
            else {
               i = 0;
            }
         }
         j--;
      }

      if (current == MC_ELEMENT_INVALID) {
         return TEST_FAILED;
      }

      MESSAGE(DEBUG_INFO, "plt", "plt", "plt_give_tape from",
              MC_ELEMENT_TO_STRING(xfer_element));
      MESSAGE(DEBUG_INFO, "plt", "plt", "plt_give_tape to",
              MC_ELEMENT_TO_STRING(current));

      if (plt_move_medium(0, xfer_element, current) == MC_RESULT_OK) {
         the_stats.slot_activity_count[tape_num][current - SLOT_OFFSET]++;
         return TEST_SUCCEEDED;
      }
      else {
         MESSAGE(DEBUG_ERROR, "plt", "plt", "plt_move_medium", "failed");
         return TEST_FAILED;
      }
      /* NOT REACHED */
   }
   else {
      MESSAGE(DEBUG_ERROR, "plt", "plt", "mc_get_storage_elements", "failed");
      return TEST_FAILED;
   }
   /* NOT REACHED */
#else /* FAKE_MOVES */
   return TEST_SUCCEEDED;
#endif /* FAKE_MOVES */
}

static void plt_send_result(BOOL ok, int arg, fd_t result_pipe) {
   plt_to_tape_t r;

   if (ok) {
      r.result = PLT_RESULT_OK;
   }
   else {
      r.result = PLT_RESULT_END;
   }

   r.arg = arg;

   if (write(result_pipe, &r, sizeof(r)) != sizeof(r)) {
      MESSAGE(DEBUG_FATAL, "plt", "plt", "write()", MESSAGE_PERROR);
   }
}

static void plt_dump_stats(void) {
   int i, j;

   printf("STATISTICS\n");
   printf("   Load/Unload: %5d.\n", the_stats.load_count);

   for (j = 0; j < tape_number; j++) {
      printf("                TAPE %1d", j);
   }

   printf("\n");

   for (i = 0; i < slot_number; i++) {
      printf("   Slot %2d:", i + 1);
      for (j = 0; j < tape_number; j++) {
         printf("     %5d operations", 
                the_stats.slot_activity_count[j][i]);
      } 
      printf("\n");
  }
}

static mc_result_t plt_load(void) {
   mc_result_t r;
   MESSAGE(DEBUG_NOTICE, "plt", "plt_load", "Loading",
                                   "");   
   r = mc_initialize();

   if (r != MC_RESULT_OK) {
      IFDEBUG(DEBUG_ERROR)
      printf("%s\n", MC_RESULT_DESCR(r));
      ENDIFDEBUG
   }

   return r;
}

static mc_result_t plt_unload(void) {
   mc_result_t r;
   MESSAGE(DEBUG_NOTICE, "plt", "plt_load", "Unloading",
                                   "");   
   r = mc_unload();

   if (r != MC_RESULT_OK) {
      IFDEBUG(DEBUG_ERROR)
      printf("%s\n", MC_RESULT_DESCR(r));
      ENDIFDEBUG
   }

   return r;
}

#define MAXPATHLEN 200
static mc_result_t plt_move_medium(int arg1, int arg2, int arg3) {
   mc_result_t r;

   IFDEBUG(DEBUG_NOTICE)
   time_show_elapsed_time();
   printf(" Loading element %d to element %d with arm %d ... ",
   arg2, arg3, arg1);
   fflush(stdout);
   ENDIFDEBUG

#ifdef FAKE_MOVES
   r = MC_RESULT_OK;
#else /* !FAKE_MOVES */
   r = mc_move_medium(arg1, arg2, arg3);
#endif /* !FAKE_MOVES */

   IFDEBUG(DEBUG_NOTICE)
   printf("%s\n", MC_RESULT_DESCR(r));
   ENDIFDEBUG

   return r;
}

static mc_result_t plt_position(int arg1, int arg2) {
   mc_result_t r;

   IFDEBUG(DEBUG_NOTICE)
   time_show_elapsed_time();
   printf(" Positionning to element %d with arm %d ... ",
   arg2, arg1);
   fflush(stdout);
   ENDIFDEBUG

   r = mc_position(arg1, arg2);

   IFDEBUG(DEBUG_NOTICE)
   printf("%s\n", MC_RESULT_DESCR(r));
   ENDIFDEBUG

   return r;
}

static BOOL check_nothing_in_tapes(void) {
   BOOL tape_full_array[PLT_MAX_TAPES];
   mc_result_t mc_result;
   int i;

   mc_result = mc_get_data_transfer_elements(NULL,
                                             PLT_MAX_TAPES,
                                             tape_full_array);

   if (mc_result != MC_RESULT_OK) {
      MESSAGE(DEBUG_ERROR, "plt", "plt_init", "mc_get_tape_elements",
              MC_RESULT_DESCR(mc_result));
      return FALSE; /* well ... */
   }
   else {
      for (i = 0; i < PLT_MAX_TAPES; i++) {
         if (tape_full_array[i]) {
            return FALSE;
         }
      }
      return TRUE;
      
   }
   /* NOT REACHED */
}
