/* scsi.c -- Parallel Libra Tester low level SCSI functions.
 * Marc SCHAEFER <schaefer@dilog.ch>
 * Creation: 01/08/96
 * Update:   03/10/96
 * V1.0 PV003
 * NOTES
 *    - see the Makefile: -DOLD_KERNEL
 *                        -DWAIT_LONGER=secs
 *    - we open/close the generic each time, this means this is slower,
 *      but maybe more reliable.
 * BUGS
 *    - does not seem to detect a disconnected unit (kernel problem)
 *    - first command usually fails. (kernel 2.x problem ?)
 * TODO
 * $Id: scsi.c,v 1.3 1999/04/01 16:57:47 schaefer Exp $
 */

#undef  DEBUG_DUMP_DATA

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#ifdef __linux
#ifdef WAIT_LONGER
#include <asm/param.h>
#include <sys/ioctl.h>
#endif
#endif /* __linux */

#ifdef __sun
#include <sys/ioctl.h>
#include <sys/ioccom.h>
#include <sys/scsi/targets/scgio.h>
#endif /* __sun */
#include "types.h"
#include "debug_if.h"
#include "scsi_if.h"

#define DEBUG

/* Private type definitions */

#define MIN(x, y) ((x) < (y) ? (x) : (y))

/* Private function definitions */

#ifdef DEBUG_DUMP_DATA
/* NAME
 *    dump_data
 * DESCRIPTION
 *    Dumps some data to stderr.
 * RESULT
 *    NONE
 * NOTES
 * BUGS
 * TODO
 */
static void dump_data(char *name,
                      char *type,
                      u_char *data,
                      u_char length);
#endif /* DEBUG_DUMP_DATA */

/* Function implementation */

#ifdef __linux
BOOL scsi_handle_scsi_cmd(char *generic_device,
                     unsigned cmd_len,
                     unsigned in_size,
                     unsigned char *i_buff,
                     unsigned out_size,
                     unsigned char *o_buff) {
#ifdef WAIT_LONGER
  int timeout = WAIT_LONGER * HZ; /* WAIT_LONGER in seconds */
#endif
  int status = 0;
  fd_t fd;
  struct sg_header *sg_hd;

  while (TRUE) {

  MESSAGE(DEBUG_INFO, "plt", "scsi", "opening", generic_device);

  fd = open(generic_device, O_RDWR);
  if (fd == -1) {
    MESSAGE(DEBUG_FATAL, "plt", "scsi", "open()", MESSAGE_PERROR);
    return FALSE;
  }

#ifdef WAIT_LONGER
  if (ioctl(fd, SG_SET_TIMEOUT, &timeout) == -1) {
     MESSAGE(DEBUG_FATAL, "plt", "scsi", "ioctl() timeout", MESSAGE_PERROR);
     close(fd);
     return FALSE;
  }
#endif

  /* safety checks */
  if (!cmd_len) { close(fd); return FALSE; }    /* need a cmd_len != 0 */
  if (!i_buff) { close(fd); return FALSE; }    /* need an input buffer != NULL */
#ifdef SG_BIG_BUFF
  if ((SCSI_OFF + cmd_len + in_size) > SG_BIG_BUFF) {
     close(fd); return FALSE;
  }
  if ((SCSI_OFF + out_size) > SG_BIG_BUFF) {
     close(fd); return FALSE;
  }
#else
  if ((SCSI_OFF + cmd_len + in_size) > 4096) { close(fd); return FALSE; }
  if ((SCSI_OFF + out_size) > 4096) { close(fd); return FALSE; }
#endif

  if (!o_buff) out_size = 0;

    /* generic scsi device header construction */
  sg_hd = (struct sg_header *) i_buff;
  sg_hd->reply_len  = SCSI_OFF + out_size;
  sg_hd->twelve_byte = cmd_len == 12;
  sg_hd->result = 0;
#if 0
  sg_hd->pack_len  = SCSI_OFF + cmd_len + in_size;
  sg_hd->pack_id;   /* not used */
  sg_hd->other_flags; /* not used */
#endif

  /* send command */
  status = write(fd, i_buff, SCSI_OFF + cmd_len + in_size);
  if ((status < 0)
      || (status != (SCSI_OFF + cmd_len + in_size))
      || sg_hd->result) {
    /* some error happened */
    fprintf( stderr, "write(generic) result = 0x%x cmd = 0x%x\n",
	     sg_hd->result, i_buff[SCSI_OFF] );
    perror("");
#if 0
    close(fd);
    return FALSE;
#endif
  }

  if (!o_buff) o_buff = i_buff;   /* buffer pointer check */

    /* retrieve result */
  status = read(fd, o_buff, SCSI_OFF + out_size);
  if ((status < 0)
      || status != (SCSI_OFF + out_size)
      || sg_hd->result ) {
    /* some error happened */
    fprintf( stderr, "read(generic) result = 0x%x cmd = 0x%x\n",
	     sg_hd->result, o_buff[SCSI_OFF] );
    fprintf( stderr, "read(generic) sense "
	     "%x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x\n",
	     sg_hd->sense_buffer[0],   sg_hd->sense_buffer[1],
	     sg_hd->sense_buffer[2],   sg_hd->sense_buffer[3],
	     sg_hd->sense_buffer[4],   sg_hd->sense_buffer[5],
	     sg_hd->sense_buffer[6],   sg_hd->sense_buffer[7],
	     sg_hd->sense_buffer[8],   sg_hd->sense_buffer[9],
	     sg_hd->sense_buffer[10],   sg_hd->sense_buffer[11],
	     sg_hd->sense_buffer[12],   sg_hd->sense_buffer[13],
	     sg_hd->sense_buffer[14],   sg_hd->sense_buffer[15]);
    if (status < 0) {
       perror("");
    }
  }
  close(fd);

  if (sg_hd->sense_buffer[2] == 6) continue;

  break;
  }

  /* Look if we got what we expected to get */
  if (status == SCSI_OFF + out_size) status = 0; /* got them all */

  if (status) {
     return FALSE;
  }

  MESSAGE(DEBUG_INFO, "plt", "scsi", "closing", generic_device);
  return TRUE;
}

#endif /* __linux */

#ifdef __sun
BOOL scsi_handle_scsi_cmd(char *generic_device,
                          unsigned cmd_len,
                          unsigned in_size,
                          unsigned char *i_buff,
                          unsigned out_size,
                          unsigned char *o_buff) {
   /* This is really a quick emulation */
   unsigned int scg_number;
   unsigned int scsi_id;
   unsigned int scsi_lun;
   unsigned char *buffer;


#ifdef DEBUG_DUMP_DATA
   printf("dev %s cmd_len %d in %d in_b 0x%x out %d out_b 0x%x\n",
          generic_device, cmd_len, in_size, i_buff, out_size, o_buff);
#endif

   if (out_size) {
      buffer = valloc(out_size);
      if (buffer == NULL) {
	 MESSAGE(DEBUG_FATAL, "plt", "scsi", "valloc()", MESSAGE_PERROR);
	 return FALSE;
      }
   }
   else {
      buffer = NULL;
   }

   /* BUGS
    *    - No support for DATAOUT (write).
    */

   if (sscanf(generic_device,
              "/dev/changer_%d_%d_%d",
              &scg_number,
              &scsi_id,
              &scsi_lun) == 3) {
      int f;
      unsigned char devname[20];

      sprintf(devname, "/dev/scg%d", scg_number);
      f = open(devname, O_RDWR);
      if (f < 0) {
	 MESSAGE(DEBUG_FATAL,
                 "plt", "scsi",
                 "open()",
                 MESSAGE_PERROR);
         if (buffer) {
            free(buffer);
         }
	 return FALSE;
      }
      else {
         struct scg_cmd scmd;

         scmd.debug = 0;
         scmd.timeout = 600; /* 10 minutes */
         scmd.flags = (out_size) ? SCG_RECV_DATA : 0;
         scmd.cdb_len = cmd_len;
         scmd.sense_len = SG_SENSE_LEN;

         scmd.target = scsi_id;

         scmd.size = out_size;
         scmd.addr = buffer;

         memcpy(&scmd.cdb.cmd_cdb, i_buff + SCSI_OFF, scmd.cdb_len);

         memset(&(((struct sg_header *) i_buff)->sense_buffer[0]),
                0,
                SG_SENSE_LEN);

         if (ioctl((f), (SCGIO_CMD), (&scmd)) < 0) {
	    MESSAGE(DEBUG_FATAL,
		    "plt", "scsi",
		    "ioctl()",
		    MESSAGE_PERROR);
            close(f);
	    if (buffer) {
	       free(buffer);
	    }
            return FALSE;
         }
         else {
            if (scmd.sense_len) {
	       memcpy(&(((struct sg_header *) i_buff)->sense_buffer[0]),
		      scmd.u_sense.cmd_sense,
		      MIN(SG_SENSE_LEN, MIN(scmd.sense_len, SCG_MAX_SENSE)));
            }

            if (out_size) {
               memcpy(o_buff + SCSI_OFF, buffer, out_size);
#ifdef DEBUG_DUMP_DATA
	       dump_data("scsi", "data",
			 buffer,
			 100);
#endif /* DEBUG_DUMP_DATA */
            }
            close(f);
	    if (buffer) {
	       free(buffer);
	    }
            return TRUE;
         }
      }
   }
   else {
      MESSAGE(DEBUG_FATAL, "plt", "scsi", "parse device name", MESSAGE_PERROR);
      printf("Syntax is /dev/changer_%%d_%%d_%%d, bus/id/lun.\n");
      if (buffer) {
         free(buffer);
      }
      return FALSE;
   }
}
#endif /* __sun */

#ifdef DEBUG_DUMP_DATA
static void dump_data(char *name,
                      char *type,
                      u_char *data,
                      u_char length) {
   if (length) {
      printf("%s: dumping %s: ", name, type);

      while (length--) {
         printf("0x%x%s", *data, length ? " " : "");
         data++;
      }

      printf(".\n");
   }
}
#endif /* DEBUG_DUMP_DATA */