/* tape.c -- Parallel Libra Tester Tape Object.
 * Marc SCHAEFER <schaefer@dilog.ch>
 * Creation: 01/08/96
 * Update:   12/11/96
 * V1.0 PV004
 * NOTES
 *    - The more_{read|write} functions have been implemented to
 *      keep the small 20k buffer (for memory reason on the test
 *      machines) and to allow bigger size easily.
 *    - This means however that only the last chunk will be checked
 *      against the random pattern; however experience has shown that
 *      errors come through an I/O error most of the time.
 * BUGS
 * TODO
 * MODIFICATION-HISTORY
 *    10/12/98  schaefer Applying new RCS scheme.
 * $Id: tape.c,v 1.2 1999/04/01 16:06:48 schaefer Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include "types.h"
#include "debug_if.h"
#include "tape_if.h"
#include "plt_if.h"
#include "process_if.h"

#ifdef SHOW_SOFT_ERRORS
#define MAXLINELENGTH 100 /* for count buffer sprintf */
#endif

#ifndef USE_MT_SCRIPT
#include "mt_operations_if.h"
#endif

#define MORE_REDO 10 /* Redo read/write 10 times for a 10x20k == 200k,
                      * put 1 for perfect read/write execution.
                      * WARNING: 0 is not a legal value.
                      */

/* External variables */

extern BOOL do_write, do_label;

/* Private type definitions */

#define TAPE_TAPE_DESCR_LENGTH 20 /* must be strncpy by process instanci. */
typedef struct {
   fd_t result_pipe;  /* From medium changer object to tape object */
   fd_t command_pipe; /* From tape object to medium changer object */
   char description[TAPE_TAPE_DESCR_LENGTH]; /* String representing the
                                             * UNIX tape device. Better
                                             * would be to use the
                                             * SCSI ID/LUN in the
                                             * element structure.
                                             */
   /* Statistics */

   unsigned long tape_inserted;
   unsigned long tape_retrieved;
   unsigned long megabytes_written;
   unsigned long megabytes_read;
   unsigned long media_errors;
} tape_tape_t;

/* Private function definitions */

/* NAME
 *    tape_instance_init
 * DESCRIPTION
 *    This function initializes a Tape Object (which is a
 *    child (another process) communicating with the father
 *    through pipes). It first reads the pipe to get the
 *    UNIX tape device description.
 * RESULT
 * NOTES
 *    - read(2) on a type are supposed atomic, the assumption is
 *    enforced. This may not be true on sockets, eg.
 *    - We also send binary data without network conversion.
 *    This is not a network, BTW.
 * BUGS
 * TODO
 */
static void tape_instance_init(fd_t from_father, fd_t to_father);


/* NAME
 *    tape_say
 * DESCRIPTION
 *    This function sends a specific command to the father.
 * RESULT
 * NOTES
 *    - write(2) on a type are supposed atomic, the assumption is
 *    enforced. This may not be true on sockets, eg.
 * BUGS
 * TODO
 */
static BOOL tape_say(fd_t command_pipe,
                     fd_t result_pipe,
                     plt_cmd_t cmd,
                     char *instance_name,
                     int *arg);

/* NAME
 *    tape_test_cycle
 * DESCRIPTION
 *    This function does a test cycle on a tape unit and updates
 *    diverse statistics.
 * RESULT
 * NOTES
 * BUGS
 * TODO
 */
static BOOL tape_test_cycle(tape_tape_t *tinfo, int arg);

/* NAME
 *    more_read
 * DESCRIPTION
 *    This function executes a number of times the equivalent
 *    of the function which is implied (ie read).
 * RESULT
 *    Result of the last executed function. See the manual of
 *    the read(2) system call.
 * NOTES
 * BUGS
 * TODO
 */
static int more_read(int fd, char *buf, size_t count);

/* NAME
 *    more_write
 * DESCRIPTION
 *    This function executes a number of times the equivalent
 *    of the function which is implied (ie write).
 * RESULT
 *    Result of the last executed function. See the manual of
 *    the write(2) system call.
 * NOTES
 * BUGS
 * TODO
 */
static int more_write(int fd, const char *buf, size_t count);

/* Function implementation */

BOOL tape_instanciate(fd_t child_result_pipe, fd_t child_command_pipe,
                      fd_t father_result_pipe, fd_t father_command_pipe,
                      char *unit_descr) {

   if (process_create(child_result_pipe, child_command_pipe,
                      father_result_pipe, father_command_pipe,
                      tape_instance_init)) {
      char name_buffer[TAPE_TAPE_DESCR_LENGTH];

      MESSAGE(DEBUG_INFO, "plt", "tape", "tape_instanciate", 
              "instance created");

      /* NOTES
       *    - See SVR4 manpage: the result of strncpy will not be
       *      null terminated if source_len >= len.
       */
      strncpy(name_buffer, unit_descr, TAPE_TAPE_DESCR_LENGTH - 1);
      name_buffer[TAPE_TAPE_DESCR_LENGTH - 1] = '\0';

      /* DESCRIPTION
       *    Send the name through the result pipe for initialization
       * ASSUMPTION
       *    - The write is complete and non-blocking since it is
       *      the first, and less than 4096 bytes (or the appropriate
       *      definition in sys/param.h
       */

      if (write(father_result_pipe, name_buffer, TAPE_TAPE_DESCR_LENGTH)
          == TAPE_TAPE_DESCR_LENGTH) {
         MESSAGE(DEBUG_INFO, "plt", "tape", "tape_instanciate", 
                 "Initialization message sent to child");
         return TRUE;
      }
      else {
         MESSAGE(DEBUG_FATAL, "plt", "tape", "write()", 
                 MESSAGE_PERROR);
         return FALSE;
         /* BUGS
          *    - Can't kill child
          */
      }
      /* NOT REACHED */
   }
   else {
      MESSAGE(DEBUG_FATAL, "plt", "tape", "tape_instanciate", 
              "process_create failed");
      return FALSE;
   }
   /* NOT REACHED */
}

/* Private function implementation */

static void tape_instance_init(fd_t from_father, fd_t to_father) {
   BOOL ok = TRUE;
   tape_tape_t tape_info;
   int arg;

   /* Initialize */

   tape_info.result_pipe = from_father;
   tape_info.command_pipe = to_father;
   tape_info.tape_inserted = 0;
   tape_info.tape_retrieved = 0;
   tape_info.megabytes_written = 0;
   tape_info.megabytes_read = 0;
   tape_info.media_errors = 0;

   /* ASSUMPTION
    *    - See tape_instanciate() implementation.
    */
   if (read(tape_info.result_pipe, tape_info.description,
            TAPE_TAPE_DESCR_LENGTH) != TAPE_TAPE_DESCR_LENGTH) {
      MESSAGE(DEBUG_FATAL, "plt", "tape", "read()", 
              MESSAGE_PERROR);
      tape_say(tape_info.command_pipe, 
               tape_info.result_pipe,
               PLT_CMD_FATAL,
               "tape", &arg); /* result ignored */
      ok = FALSE;
   }
   else {
      MESSAGE(DEBUG_INFO, "plt", "tape", "Received initialization", 
              tape_info.description);
   }

   while (ok) {
      ok = FALSE;

      if (tape_say(tape_info.command_pipe, tape_info.result_pipe,
                   PLT_CMD_GET_TAPE, tape_info.description, &arg)) {
         if (tape_test_cycle(&tape_info, arg)) {
            if (tape_say(tape_info.command_pipe, 
                         tape_info.result_pipe,
                         PLT_CMD_GIVE_TAPE,
                         tape_info.description, &arg)) {
               ok = TRUE;
            }
         }
      }
   }

   tape_say(tape_info.command_pipe,
            tape_info.result_pipe,
            PLT_CMD_ENDING,
            tape_info.description,
            &arg); /* result ignored */
   exit(0);
}

static BOOL tape_say(fd_t command_pipe,
                     fd_t result_pipe,
                     plt_cmd_t cmd,
                     char *description,
                     int *arg) {
   plt_to_tape_t result;

   if (write(command_pipe, &cmd, sizeof(cmd))
       == sizeof(cmd)) {
      MESSAGE(DEBUG_INFO, "plt", description, "tape_say",
              PLT_PLT_CMD_TO_STRING(cmd));

      if (read(result_pipe, &result, sizeof(result)) == sizeof(result)) {
         if (result.result == PLT_RESULT_OK) {
            MESSAGE(DEBUG_INFO, "plt", description, "tape_say", "result is OK");
            *arg = result.arg;
            return TRUE;
         }
         else {
            MESSAGE(DEBUG_INFO, "plt", description, "tape_say", "result is END");
            return FALSE;
         }
      }
      else {
         return FALSE;
      }
      /* NOT REACHED */
   }
   else {
      MESSAGE(DEBUG_FATAL, "plt", description, "write()",
              MESSAGE_PERROR);
      return FALSE;
   }
   /* NOT REACHED */
}

#define MAXPATHLEN 1024
#define TEST_BUFLEN 20000 /* should be bigger and allocated */
static BOOL tape_test_cycle(tape_tape_t *tinfo, int number) {
   int f;
   int label;
#ifdef USE_MT_SCRIPT
   char buf[MAXPATHLEN];
#endif /* USE_MT_SCRIPT */

   if (!do_write && !do_label) {
      return TRUE;
   }

   MESSAGE(DEBUG_INFO, "plt", tinfo->description, "setting density",
           "");

   /* NOTES
    *    - The following is needed because the Libra 8 does not
    *      return MC_OK when the drive says OK, but before.
    *      We could ask the drive via the generic interface,
    *      or via mtio.
    *    - The following script does it for us via the mt command
    *      by waiting upto 40 seconds, and doing the setblk.
    */
#ifdef USE_MT_SCRIPT
   sprintf(buf, "wait_tape_ready.sh %s", tinfo->description);
   system(buf); /* no return value check */
#else
   if (!mt_do(tinfo->description)) {
      return FALSE;
    }
#endif

   MESSAGE(DEBUG_INFO, "plt", tinfo->description, "test_tape",
           "opening tape device");
   f = open(tinfo->description, O_RDONLY);
   if (f != -1) {
      if (do_label) {
         /* Read the label from the tape */
         MESSAGE(DEBUG_INFO, "plt", tinfo->description, "test_tape",
           "reading label");
      }

      if (!do_label || (read(f, &label, sizeof(label)) == sizeof(label))) {
         if (!do_label || (label == number)) {
            int i;
            char buffer[TEST_BUFLEN];
            char buf2[TEST_BUFLEN];

            close(f);

            if (!do_write) {
               return TRUE;
            }

            f = open(tinfo->description, O_RDWR);
            if (f != -1) {
               /* Try write, then read again */

               for (i = 0; i < sizeof(buffer); i++) {
                  buffer[i] = random() % 256;
               }
 
               MESSAGE(DEBUG_INFO, "plt", tinfo->description, "test_tape",
                                   "write test");

               if (more_write(f, buffer, sizeof(buffer)) == sizeof(buffer)) {
#ifdef SHOW_SOFT_ERRORS
                  int cnt;
#endif
                  /* Verify read */
                  close(f);

#ifdef SHOW_SOFT_ERRORS
                  if ((cnt = mt_get_retry_count(tinfo->description)) > 0) {
                     char soft_error_buffer[MAXLINELENGTH];

                     sprintf(soft_error_buffer, "soft error count is %d", cnt);
                     MESSAGE(DEBUG_NOTICE, "plt", tinfo->description,
                             "tape_test", soft_error_buffer);
                  }
#endif
                  f = open(tinfo->description, O_RDONLY);
                  if (f != -1) {
                     MESSAGE(DEBUG_INFO, "plt", tinfo->description, "test_tape",
                                   "reading back");
                  if (more_read(f, buf2, sizeof(buf2)) == sizeof(buf2)) {
                      if (!memcmp(buffer, buf2, sizeof(buffer))) {
                         close(f); /* error on close ignored
                                    * everywhere
                                    */

                         /* Reopen and write the label */

                         if (!do_label) {
                            return TRUE;
                         }
                         f = open(tinfo->description, O_WRONLY);
                         if (f != -1) {
                            MESSAGE(DEBUG_INFO, "plt", tinfo->description, "test_tape",
                                   "writing label back");
                          if (write(f, &number, sizeof(number)) == sizeof(number)) {
                               close(f); /* no check */
                               MESSAGE(DEBUG_NOTICE, "plt", tinfo->description, "test_tape",
                                   "Test OK");
                               return TRUE;
                            }
                            else {
                                MESSAGE(DEBUG_ERROR, "plt", tinfo->description, "test_tape",
                                   "Can't write label");
                                close(f);
                                return FALSE;
                            }
                         }
                         else {
                            MESSAGE(DEBUG_ERROR, "plt", tinfo->description, "test_tape",
                                   "Can't reopen");
                            return FALSE;
                         }
                      }
                      else {
                         MESSAGE(DEBUG_ERROR, "plt", tinfo->description, "BLOCK ERROR: buffer does not match", MESSAGE_PERROR);
                         close(f);
                         return FALSE;
                      }
                  }
                  else {
                     MESSAGE(DEBUG_ERROR, "plt", tinfo->description, "cannot read 2", MESSAGE_PERROR);
                     close(f);
                     return FALSE;
                   }
                  }
                  else {
                     MESSAGE(DEBUG_ERROR, "plt", tinfo->description, "cannot open for read again", MESSAGE_PERROR);
                     return FALSE;
                  }
               }
               else {
                  MESSAGE(DEBUG_ERROR, "plt", tinfo->description, "cannot write", MESSAGE_PERROR)
                  close(f);
                  return FALSE;
               }
            }
            else {
               MESSAGE(DEBUG_ERROR, "plt", tinfo->description, "cannot open for writing", MESSAGE_PERROR);
               return FALSE;
            }
         }
         else {
            MESSAGE(DEBUG_ERROR, "plt", tinfo->description, "TAPE NUMBER ERROR", "");
            IFDEBUG(DEBUG_ERROR)
            printf("TAPE NUMBER ERROR: read %d expected %d.\n",
                   label, number);
            ENDIFDEBUG
            close(f);
            return FALSE;
         }
      }
      else {
         MESSAGE(DEBUG_ERROR, "plt", tinfo->description, "cannot read label", MESSAGE_PERROR);
         close(f);
         return FALSE;
      }
   }
   else {
      MESSAGE(DEBUG_ERROR, "plt", tinfo->description, "cannot access tape", MESSAGE_PERROR);
      return FALSE;
   }
}

static int more_read(int fd, char *buf, size_t count) {
   int i = MORE_REDO;
   int result;

   while (i--) {
      result = read(fd, buf, count);
      if (result != count) {
         break;
      }
   }

   return result;
}

static int more_write(int fd, const char *buf, size_t count) {
   int i = MORE_REDO;
   int result;

   while (i--) {
      result = write(fd, buf, count);
      if (result != count) {
         break;
      }
   }

   return result;
}
