/* tests.c -- (non) Parallel Libra Tester miscellaneous other tests.
 * Marc SCHAEFER <schaefer@dilog.ch>
 * Creation: 08/08/96
 * Update:   10/12/98
 * V1.0 PV013
 * NOTES
 * BUGS
 *    - Bypasses most of the PLT system. Too low-level.
 *      Dirty port of libra2.
 *    - Missing function definitions.
 * TODO
 *    - All tests should be backwardable. See my Wj cnfutil implementation.
 * BASED-ON
 *    libra2.c V1.0 PV023 of 06/08/96.
 *    NOTES
 *       - This software is (C) Marc SCHAEFER <schaefer@alphanet.ch>,
 *         not DILOG SA, except the plt compatilibity parts, and some
 *         of the tests.
 * MODIFICATION-HISTORY
 *    10/12/98  schaefer  Applying new RCS scheme, and adding real
 *                        position to element.
 *    22/04/99  schaefer  Fix to scanf() so that doesn't infinitely loop.
 *                        Added sun's prototype so doesn't complain.
 * $Id: tests.c,v 1.2 1999/04/22 07:51:07 schaefer Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <fcntl.h>
#include <errno.h>
#include "types.h"
#include "debug_if.h"
#include "mc_if.h"
#include "tests_if.h"

#ifdef __sun
long random();
#endif /* __sun */

#define TWIN_LIBRA

/* Compatibility defines */

/* Assumption: MC_RESULT_OK == 0 */
/* But you will get types warning */
#define SHOW_RESULT(x) printf("[%s]", MC_RESULT_DESCR(x));

#define Initialize_Element_Status() mc_initialize()
#define Mode_Sense() -1
#define Position_To_Element(x, y) mc_position((x), (y))
#define Move_Medium(x, y, z) mc_move_medium((x), (y), (z))
#define Exchange_Medium(x, y, z) -1
#define Do_Read_Element_Status() { mc_read_element_status(NULL, 0, NULL, NULL, 0, NULL, TRUE); sleep(5); }

/* Configuration defines */

/* #define LOW_LEVEL_ELEMENT_DEBUG */
/* The following could be get by elt_status */
#ifdef TWIN_LIBRA
#define FIRST_SLOT_OFFSET 3
#define SLOT_NUMBER num_slots
#else
#define FIRST_SLOT_OFFSET 2
#define SLOT_NUMBER num_slots
#endif
#define SCSI_OFF sizeof(struct sg_header)

int Test_Tape(int number, int drive);

static int test_mode = 0;
static int write_mode = 0;
static int no_retry = -1;
static int continuous = 0;
static int num_slots =
#ifdef TWIN_LIBRA
16
#else
8
#endif
;

static char *tape_devices[2];

int Do_Load(void) {
   int result;

   printf("\nLoading ... ");
   fflush(stdout);
   result = Initialize_Element_Status();
   SHOW_RESULT(result);
   return result;
}

int Do_Mode_Sense(void) {
   int result;

   printf("\nMode Sense ... ");
   fflush(stdout);
   result = Mode_Sense();
   SHOW_RESULT(result);
   return result;
}

int Do_Unload(void) {
   int result;

   printf("\nUnloading ... ");
   fflush(stdout);
   result = Position_To_Element(0, 0);
   SHOW_RESULT(result);
   return result;
}

int Do_Position(int arm, int destination) {
   int result;

   printf("\nPositionning arm %d to element %d ... ", arm, destination);
   fflush(stdout);
   result = Position_To_Element(arm, destination);
   SHOW_RESULT(result);
   return result;
}

#define TIME_BUFFER_LEN 10
int Do_Move(int arm, int source, int destination) {
   char the_time[TIME_BUFFER_LEN];
   int result;
   time_t t;

   time(&t);  

   strncpy(the_time, ctime(&t) + 11, 8);
   the_time[8] = '\0'; 

   printf("\n(%s) Loading element %d to element %d with arm %d ... ",
                                                           the_time,
                                                           source,
                                                           destination,
                                                           arm);
                                                                    
   fflush(stdout);
   result = Move_Medium(arm, source, destination);
   SHOW_RESULT(result);
   if (destination < FIRST_SLOT_OFFSET) {
      switch(Test_Tape(source + 1 - FIRST_SLOT_OFFSET, destination)) {
      case -1:
      case -2: result = -1;
	       break;
      }
   }
   return result;
}

int Forever_Do_Move(int arm, int source, int destination) {
   int i = 0;
   int result = 1;

#ifdef FAKE_MOVE
   printf("%d to %d.\n", source, destination);
   return 0;
#endif

   while (result && (i < 10)) {
      if (i) {
         printf("RETRY %d.\n", i);
       }
      result = Do_Move(arm, source, destination);
      if (no_retry) {
         break;
      }
      i++;
   }

   return result;
}

int Do_Exchange(int arm, int source, int destination) {
   int result;

   printf("\nExchanging element %d and element %d with arm %d ... ",
                                                           source,
                                                           destination,
                                                           arm);
                                                                    
   fflush(stdout);
   result = Exchange_Medium(arm, source, destination);
   SHOW_RESULT(result);
   return result;
}

/* NAME
 *    Test_Tape
 * DESCRIPTION
 *    Test read/write on tape. Puts a number after
 *    to be able to check the tape number.
 * RESULT
 *    -1 if crashed
 *    -2 if label does not match
 *    0 if ok.
 * NOTES
 *    - Current implementation maybe wrong
 *    - Device supposed auto-rewind.
 *    - Label supposed to be written already.
 * BUGS
 * TODO
 */
#define TEST_BUFLEN 20000
#define MAXPATHLEN 100

int Set_Tape_Label(int number, int drive) {
   int f;
   char *device;
   char buf[MAXPATHLEN];

   if ((drive == 1) || (drive == 2)) {
      device = tape_devices[drive - 1];
   }
   else {
      printf("Set_Tape_Label: ILLEGAL drive.\n");
      return -1;
   }

   sprintf(buf, "/bin/mt -f %s setblk 0", device);
   system(buf);

   f = open(device, O_RDWR);
   if (f != -1) {
      /* Write the label to the tape */
      if (write(f, &number, sizeof(number)) == sizeof(number)) {
         close(f);
         return 0;
      }
      else {
         printf("TAPE ERROR: Can't write label.\n");
         perror(".");
         close(f);
         return -1;
      }
   }
   else {
      printf("TAPE ERROR: Can't access tape %s.\n", device);
      perror(".");
      return -1;
   }
}

int Test_Tape(int number, int drive) {
   int f;
   int label;
   char *device;
   char buf[MAXPATHLEN];

   if (!test_mode && !write_mode) {
      return 0;
   }

   if ((drive == 1) || (drive == 2)) {
      device = tape_devices[drive - 1];
   }
   else {
      printf("Test_Tape: ILLEGAL drive.\n");
      return -1;
   }

   printf("Test_Tape: setting density.\n");

   sprintf(buf, "/bin/mt -f %s setblk 0", device);
   system(buf);

   printf("Test_Tape: opening tape device.\n");

   f = open(device, O_RDONLY);
   if (f != -1) {
      if (test_mode) {
      /* Read the label from the tape */
      printf("Test_Tape: reading the label.\n");
      }

      if (!test_mode || (read(f, &label, sizeof(label)) == sizeof(label))) {
         if (!test_mode || (label == number)) {
            int i;
            char buffer[TEST_BUFLEN];
            char buf2[TEST_BUFLEN];

            close(f);

            if (!write_mode) {
               return 0;
            }

            f = open(device, O_RDWR);
            if (f != -1) {
               /* Try write, then read again */

               for (i = 0; i < sizeof(buffer); i++) {
                  buffer[i] = random() % 256;
               }
 
               printf("Test_Tape: write test.\n");

               if (write(f, buffer, sizeof(buffer)) == sizeof(buffer)) {
                  /* Verify read */
                  close(f);
                  f = open(device, O_RDONLY);
                  if (f != -1) {
                     printf("Test_Tape: reading back.\n");
                  if (read(f, &buf2, sizeof(buf2)) == sizeof(buf2)) {
                      if (!memcmp(buffer, buf2, sizeof(buffer))) {
                         close(f); /* error on close ignored
                                    * everywhere
                                    */

                         /* Reopen and write the label */

                         if (!test_mode) {
                            return 0;
                         }
                         f = open(device, O_WRONLY);
                         if (f != -1) {
                          printf("Test_Tape: writing label back.\n");
                            if (write(f, &number, sizeof(number)) == sizeof(number)) {
                               close(f);
                               printf("Test_Tape: test OK.\n");
                               return 0;
                            }
                            else {
                                printf("TAPE ERROR: cannot write label.\n");
                                close(f);
                                return -1;
                            }
                         }
                         else {
                            printf("TAPE ERROR: cannot re-open.\n");
                            return -1;
                         }
                      }
                      else {
                         printf("BLOCK ERROR: buffer does not match.\n");
                         perror(".");
                         close(f);
                         return -1;
                      }
                  }
                  else {
                     printf("TAPE ERROR: cannot read 2.\n");
                     perror(".");
                     close(f);
                     return -1;
                   }
                  }
                  else {
                     printf("TAPE ERROR: cannot open for read again.\n");
                     perror(".");
                     return -1;
                  }
               }
               else {
                  printf("TAPE ERROR: cannot write.\n");
                  perror(".");
                  close(f);
                  return -1;
               }
            }
            else {
               printf("TAPE ERROR: cannot open for writing.\n");
               perror(".");
               return -1;
            }
         }
         else {
            printf("TAPE NUMBER ERROR: read %d expected %d.\n",
                   label, number);
            close(f);
            return -2;
         }
      }
      else {
         printf("TAPE ERROR: Can't read label.\n");
         perror(".");
         close(f);
         return -1;
      }
   }
   else {
      printf("TAPE ERROR: Can't access tape %s.\n", device);
      perror(".");
      return -1;
   }
}


/* NAME
 *    Do_Test_Random
 * DESCRIPTION
 *    Moves tape around in a (pseudo-)random manner.
 * RESULT
 *    -1 if crashed
 * NOTES
 *    - You should use `script' to save the result to a file
 *    for testing.
 *    - This is an infinite test
 * BUGS
 *    - Does not initialize the random generator. So we
 *      get the same sequence. Idea: print the number obtained
 *      so we can re-run the sequence.
 * TODO
 *    - Label the tapes so we can verify if we are on the
 *      correct one visually, and label the media to
 *      check automatically at each load.
 *    - Save internal representation of which initial
 *      tape moved to where for extended testing and
 *      visual verification.
 *    - Verify internal with Read_Element_Status.
 */

int get_random_full_slot(int *tape_array) {
   int i = random() % SLOT_NUMBER; /* 0..(SLOT_NUMBER-1) */
   int current = 0; /* 0..(SLOT_NUMBER-1) */

   while (i >= 0) {
      /* find a full slot sequentially */
      while (1) {
         if (tape_array[current] != 0) {
            break;
         }
         else {
            current = (current + 1) % SLOT_NUMBER;
         }
      }
      i--;
   }

   return current + 1;
}

int get_random_empty_slot(int *tape_array) {
   int i = random() % SLOT_NUMBER; /* 0..(SLOT_NUMBER-1) */
   int current = 0; /* 0..(SLOT_NUMBER-1) */

   while (i >= 0) {
      /* find an empty slot sequentially */
      while (1) {
         if (tape_array[current] == 0) {
            break;
         }
         else {
            current = (current + 1) % SLOT_NUMBER;
         }
      }
      i--;
   }

   return current + 1;
}

int Do_Test_Random() {
   int result = 0;
   int i;
   int drive_1_full = 0; 
   int drive_2_full = 0;
   int next_slot; /* where to insert or to retrieve */
   int insert = 0;
   int next_drive;
   int drive_1_tape; /* the cassette in drive 1 effectively was originally
                      * this tape.
                      */
   int drive_2_tape;
   int slot_array[SLOT_NUMBER];

   for (i = 0; i < SLOT_NUMBER; i++) {
      slot_array[i] = i + 1;
   }

   while (!result) {
      if (!drive_1_full && !drive_2_full) {
         /* No choice. We must load one of the tapes */
         insert = 1;
         next_drive = (random() % 2) + 1;
         next_slot = (random() % SLOT_NUMBER) + 1;
      }
      else if (drive_1_full && drive_2_full) {
         /* We must eject */

         next_drive = (random() % 2) + 1;
         next_slot = get_random_empty_slot(slot_array);
         if (slot_array[next_slot - 1] != 0) {
            printf("ASSUMPTION VIOLATED #1.\n");
	 }

         insert = 0;
      }
      else {
         insert = random() % 2;
         if (insert) {
            next_drive = (drive_1_full) ? 2 : 1;
            next_slot = get_random_full_slot(slot_array);
            if (slot_array[next_slot - 1] == 0) {
               printf("ASSUMPTION VIOLATED #2.\n");
   	    }
         }
         else {
            next_drive = (drive_1_full) ? 1 : 2;
            next_slot = get_random_empty_slot(slot_array);
            if (slot_array[next_slot - 1] != 0) {
               printf("ASSUMPTION VIOLATED #3.\n");
   	    }
         } 
      }

      /* Now do the movement */

     if (insert) {
        result = Forever_Do_Move(0, next_slot + 2, next_drive);
        if (next_drive == 1) {
           drive_1_tape = slot_array[next_slot - 1];
           slot_array[next_slot - 1] = 0;
           drive_1_full = 1;
        }
        else {
           drive_2_tape = slot_array[next_slot - 1];
           slot_array[next_slot - 1] = 0;
           drive_2_full = 1;
        }
     }
     else {
        result = Forever_Do_Move(0, next_drive, next_slot + 2);
        if (next_drive == 1) {
           slot_array[next_slot - 1] = drive_1_tape;
           drive_1_full = 0;
        }
        else {
           slot_array[next_slot - 1] = drive_2_tape;
           drive_2_full = 0;
        }
        
        for (i= 0; i < SLOT_NUMBER; i++) {
           printf("slot[%d] is tape %d.\n", i + 1, slot_array[i]);
        }
     }
   }
   return result;
}

int Do_Test_All_Slots(int val) {
  int result = 0;
  int i;
  int arm = 0;
  int j = (val == 3) ? 1 : val;

  if ((val == 3) && (test_mode || write_mode)) {
     test_mode = 0;
     write_mode = 0;
     printf("WARNING: test_mode/write_mode have been disabled.\n");
  }

  for (i = 1; i <= SLOT_NUMBER ; i++) {
    if (!Forever_Do_Move(arm, i + FIRST_SLOT_OFFSET - 1, j)) {
      if (val == 3) {
        switch(Set_Tape_Label(i, j)) {
        case -2:
	case -1: result = -1;
	  break;
        }
        if (result == -1)  {
	  break;
        }
      }
      if (!Forever_Do_Move(arm, j, i + FIRST_SLOT_OFFSET - 1)) {
	;
      }
      else {
	fprintf(stderr, "test: Do_Move(%d, %d, %d) failed.\n",
		arm, j, i + FIRST_SLOT_OFFSET - 1);
	result = -1;
        break;
      }
    }
    else {
      fprintf(stderr, "test: Do_Move(%d, %d, %d) failed.\n",
	      arm, i + FIRST_SLOT_OFFSET - 1, j);
      result = -1;
      break;
    }
  }
  return result;
}

int Do_Test_16x16(void) {
  int result = 0;
  int i, j;
  int arm = 0;

  for (i = 1; i <= SLOT_NUMBER; i++) {
            if (result == -1) break;
     for (j = 1; j <= SLOT_NUMBER; j++) {
            if (result == -1) break;
        if (!Forever_Do_Move(arm, i + FIRST_SLOT_OFFSET - 1, 1)) {
           if (!Forever_Do_Move(arm, 1, i + FIRST_SLOT_OFFSET - 1)) {
            if (result == -1) break;
              if (!Forever_Do_Move(arm, j + FIRST_SLOT_OFFSET - 1, 2)) {
                 if (!Forever_Do_Move(arm, 2, j + FIRST_SLOT_OFFSET - 1)) {
                   ;;
                 }
                 else {
                    fprintf(stderr, "test: Do_Move(%d, %d, %d) failed.\n",
	                            arm, 2, j + FIRST_SLOT_OFFSET - 1);
                    result = -1;
                    break;
                 }
              }
              else {
                    fprintf(stderr, "test: Do_Move(%d, %d, %d) failed.\n",
	                            arm, j + FIRST_SLOT_OFFSET - 1, 2);
                    result = -1;
                    break;
              }
           }
           else {
              fprintf(stderr, "test: Do_Move(%d, %d, %d) failed.\n",
	                      arm, 1, i + FIRST_SLOT_OFFSET - 1);
              result = -1;
              break;
           }
        }
        else {
            fprintf(stderr, "test: Do_Move(%d, %d, %d) failed.\n",
                            arm, i + FIRST_SLOT_OFFSET - 1, 1);
            result = -1;
            break;
        }
      }
   }
  return result;
}

int Do_Test_Quick(void) {
   int result = 0;
   int i, j;
   int elt_array[5] = {3, 18, 10, 7, 14};

   for (j = 1; j < 3; j++) {
      for (i = 0; i < sizeof(elt_array)/sizeof(int); i++) {
	 result = Forever_Do_Move(0, elt_array[i], j);      
         if (result) {
            break;
         }
         else {
   	    result = Forever_Do_Move(0, j, elt_array[i]);      
         }
      }
      if (result) {
        break;
      }
   }

   return result;
}

#if 0
/* NAME
 *    Forever_Do_Move_Filter
 * DESCRIPTION
 *    Filters unwanted operations.
 * RESULT
 *    int
 * NOTES
 * BUGS
 * TODO
 * BASED-ON
 *    windjammer's cnfutil.c CNF_lib_adsm_variant of 15/01/97.
 */
static int Forever_Do_Move_Filter(int arm, int from, int to) {
   if ((from > (num_slots + 2)) || (to > (num_slots + 2))
       || ((from < 3) && (to < 3))) {
       return 0;
   }
   else {
      return Forever_Do_Move(arm, from, to);
   }
}

/* NAME
 *    Do_Test_ADSM_NEW
 * DESCRIPTION
 * RESULT
 *    int
 * NOTES
 * BUGS
 * TODO
 * BASED-ON
 *    windjammer's cnfutil.c CNF_lib_adsm_variant of 15/01/97.
 */
static int Do_Test_ADSM_NEW(BOOL reverse, BOOL second_way) {
   int slot = 3;
   int last_slot = num_slots + 2;
   int drive = 1;
   BOOL has_error = FALSE;
   int increment = 1;
 
   if (reverse) {
      slot = (last_slot);
      last_slot = 4;
      increment = -1;
   }
 
   if (second_way) {
      drive = 2;
   }
 
   /* Start */
 
   has_error = Forever_Do_Move_Filter(0, slot, drive);
     
   while (!has_error) {
 
      drive = (2 - drive) + 1;
      has_error 
         = Forever_Do_Move_Filter(0, slot + increment, drive);
      if (!has_error) {
         drive = (2 - drive) + 1;
         has_error = Forever_Do_Move_Filter(0, drive, slot);
         if (!has_error) {
            has_error = Forever_Do_Move_Filter(0, slot + increment + increment, drive);
            if (!has_error) {
               drive = (2 - drive) + 1;
               has_error = Forever_Do_Move_Filter(0, drive, slot + increment);
            }
            drive = (2 - drive) + 1;
         }
      }
 
      if (slot == last_slot) {
         break; /* finished */
      }
      slot += increment;
      slot += increment;
   }
 
   if (has_error) {
      printf("FAILED.");
   }

   return has_error ? 1 : 0;
}
#endif /* 0 */

int Do_Test_ADSM(void) {
   int i = 1;
   int a = 0;
   int result = 0;

   while ((i <= SLOT_NUMBER) && (!result)) {
      switch (a) {
         case 0:
                 result = Forever_Do_Move(0, i + FIRST_SLOT_OFFSET - 1, 1);
                 a++;
                 break; /* load 0 */
         case 1: result = Forever_Do_Move(0, i + FIRST_SLOT_OFFSET, 2);
                 a++;
                 break;
         case 2: result = Forever_Do_Move(0, 1, i + FIRST_SLOT_OFFSET - 1);
                 if ((!result) && (i < (SLOT_NUMBER - 1))) {
                    result = Forever_Do_Move(0, i + FIRST_SLOT_OFFSET + 1, 1);
                 }
                 a++;
                 break;
         case 3: result = Forever_Do_Move(0, 2, i + FIRST_SLOT_OFFSET);
                 i += 2;
                 a = 1;
                 break;      
      }
   }
   return result;
}

int Do_Test_ADSM2(void) {
   int i = 1;
   int a = 0;
   int result = 0;

   while ((i <= SLOT_NUMBER) && (!result)) {
      switch (a) {
         case 0:
                 result = Forever_Do_Move(0, i + FIRST_SLOT_OFFSET - 1, 2);
                 a++;
                 break; /* load 0 */
         case 1: result = Forever_Do_Move(0, i + FIRST_SLOT_OFFSET, 1);
                 a++;
                 break;
         case 2: result = Forever_Do_Move(0, 2, i + FIRST_SLOT_OFFSET - 1);
                 if ((!result) && (i < (SLOT_NUMBER - 1))) {
                    result = Forever_Do_Move(0, i + FIRST_SLOT_OFFSET + 1, 2);
                 }
                 a++;
                 break;
         case 3: result = Forever_Do_Move(0, 1, i + FIRST_SLOT_OFFSET);
                 i += 2;
                 a = 1;
                 break;      
      }
   }
   return result;
}

   

/* see testit.c in 8097bh code. This one also indirectly tests
 * the tape labels. Values might not be corresponding to the
 * new params. */
#define FOREVER_DO_MOVE_AND_RETURN(x, y) if (Forever_Do_Move(0, x, y)) return -1; \
                                         else if (Forever_Do_Move(0, y, x)) return -1;
int Do_Test_0x56(void) {
  int i, j;

  printf("Now testing the UPPER border case, going to 1/1.\n");
  printf("NOTE: intentionnally we will also go into the normal zone.\n");

  for (i = 1; i < 6; i++) {
    FOREVER_DO_MOVE_AND_RETURN(i + 2, 2);
    FOREVER_DO_MOVE_AND_RETURN(1 + 2, 1);
  }

  printf("Now testing the LOWER border case, going to 16/2.\n");
  printf("NOTE: intentionnally we will also go into the normal zone.\n");

  for (i = 16; i > 10; i--) {
    FOREVER_DO_MOVE_AND_RETURN(i + 2, 1);
    FOREVER_DO_MOVE_AND_RETURN(16 + 2, 2);
  }

  printf("Now checking move after resync.\n");

  FOREVER_DO_MOVE_AND_RETURN(1 + 2, 1);
  FOREVER_DO_MOVE_AND_RETURN(1 + 2, 2);
  FOREVER_DO_MOVE_AND_RETURN(2 + 2, 1);

  printf("Now checking move after resync.\n");

  FOREVER_DO_MOVE_AND_RETURN(16 + 2, 1);
  FOREVER_DO_MOVE_AND_RETURN(15 + 2, 2);

  printf("Normal zone test (without dead sides).\n");

  FOREVER_DO_MOVE_AND_RETURN( 1 + 2, 1);
  FOREVER_DO_MOVE_AND_RETURN( 4 + 2, 2);

  FOREVER_DO_MOVE_AND_RETURN( 1 + 2, 1);
  FOREVER_DO_MOVE_AND_RETURN( 5 + 2, 2);

  FOREVER_DO_MOVE_AND_RETURN( 1 + 2, 1);
  FOREVER_DO_MOVE_AND_RETURN( 8 + 2, 2);

  FOREVER_DO_MOVE_AND_RETURN( 1 + 2, 1);
  FOREVER_DO_MOVE_AND_RETURN( 7 + 2, 2);

  FOREVER_DO_MOVE_AND_RETURN( 5 + 2, 1);
  FOREVER_DO_MOVE_AND_RETURN( 4 + 2, 2);

  FOREVER_DO_MOVE_AND_RETURN( 5 + 2, 1);
  FOREVER_DO_MOVE_AND_RETURN( 5 + 2, 2);

  FOREVER_DO_MOVE_AND_RETURN( 5 + 2, 1);
  FOREVER_DO_MOVE_AND_RETURN( 8 + 2, 2);

  FOREVER_DO_MOVE_AND_RETURN( 5 + 2, 1);
  FOREVER_DO_MOVE_AND_RETURN( 7 + 2, 2);

  FOREVER_DO_MOVE_AND_RETURN( 16 + 2, 2);
  FOREVER_DO_MOVE_AND_RETURN( 12 + 2, 1);

  FOREVER_DO_MOVE_AND_RETURN( 16 + 2, 2);
  FOREVER_DO_MOVE_AND_RETURN( 11 + 2, 1);

  FOREVER_DO_MOVE_AND_RETURN( 16 + 2, 2);
  FOREVER_DO_MOVE_AND_RETURN( 1 + 2, 1);

  FOREVER_DO_MOVE_AND_RETURN( 16 + 2, 2);
  FOREVER_DO_MOVE_AND_RETURN( 2 + 2, 1);

  FOREVER_DO_MOVE_AND_RETURN( 12 + 2, 2);
  FOREVER_DO_MOVE_AND_RETURN( 12 + 2, 1);

  FOREVER_DO_MOVE_AND_RETURN( 12 + 2, 2);
  FOREVER_DO_MOVE_AND_RETURN( 11 + 2, 1);

  FOREVER_DO_MOVE_AND_RETURN( 12 + 2, 2);
  FOREVER_DO_MOVE_AND_RETURN( 1 + 2, 1);

  FOREVER_DO_MOVE_AND_RETURN( 12 + 2, 2);
  FOREVER_DO_MOVE_AND_RETURN( 2 + 2, 1);

  printf("Normal zone test (with dead sides)\n");

  for (i = 5; i <= 8; i++) {
    for (j = 1; j <= 8; j++) {
      FOREVER_DO_MOVE_AND_RETURN(i + 2, 1);
      FOREVER_DO_MOVE_AND_RETURN(j + 2, 2);
    }
  }

  for (j = 12; j >= 9; j--) {
    for (i = 1; i >= 8; i++) {
      FOREVER_DO_MOVE_AND_RETURN(j + 2, 2);
      FOREVER_DO_MOVE_AND_RETURN(i + 2, 2);
    }
  }

  printf("General tests.\n");

  for (i = 0; i < 16; i++) {
    FOREVER_DO_MOVE_AND_RETURN(i + 3, 1);
    FOREVER_DO_MOVE_AND_RETURN(i + 3, 2);
    FOREVER_DO_MOVE_AND_RETURN(i + 3, 1);
  } 
  return 0;
}
int Do_Test_Alternative(void) {
  int result = 0;
  int i;
  int arm = 0;

  if (!Forever_Do_Move(arm, 1 + FIRST_SLOT_OFFSET - 1, 1)) {
    for (i = 1; i <= SLOT_NUMBER ; i++) {
      if (!Forever_Do_Move(arm, 1, i + FIRST_SLOT_OFFSET - 1)) {
	if (!Forever_Do_Move(arm, i + FIRST_SLOT_OFFSET - 1, 2)) {
	  if (!Forever_Do_Move(arm, 2, i + FIRST_SLOT_OFFSET - 1)) {
	    if (i < SLOT_NUMBER) {
	      if (!Forever_Do_Move(arm, i + 1 + FIRST_SLOT_OFFSET - 1, 1)) {
	      }
	      else {
		fprintf(stderr, "test: Forever_Do_Move(%d, %d, %d) failed.\n",
			arm, i + 1 + FIRST_SLOT_OFFSET - 1, 1);
		result = -1;
                break;
	      }
	    }
	  }
	  else {
	    fprintf(stderr, "test: Do_Move(%d, %d, %d) failed.\n",
		    arm, 1, i + FIRST_SLOT_OFFSET - 1);
	    result = -1;
            break;
	  }
	}
	else {
	  fprintf(stderr, "test: Do_Move(%d, %d, %d) failed.\n",
		  arm, i + FIRST_SLOT_OFFSET - 1, 2);
	  result = -1;
          break;
	}
      }
      else {
	fprintf(stderr, "test: Do_Move(%d, %d, %d) failed.\n",
		arm, 1, i + FIRST_SLOT_OFFSET - 1);
	result = -1;
        break;
      }
    }
  }
  else {
    fprintf(stderr, "test: Do_Move(%d, %d, %d) failed.\n",
	    arm, 1 + FIRST_SLOT_OFFSET - 1, 1);
    result = -1;
  }
  return result;
}

/* IDEA: verify the elt_status */

int Test_Mode(void) {
   int result = 0;

   if (!Do_Load()) {
      int j;

      /* First test is easy */
      for (j = 2; j >= 1; j--) {
         if (Do_Test_All_Slots(j)) {
            result = -1;
            break;
	 }
      }

     /* Second test is a little tricky */

     if (!result && Do_Test_Alternative()) {
        result = -1;
     }

     if (!result && Do_Test_16x16()) {
        result = -1;
     }      

     if (!result && Do_Test_Random()) { /* This one never returns */
        result = -1;
     }      
      
      Do_Unload();
   }
   else {
      fprintf(stderr, "test: Do_Load() failed.\n");
      result = -1;
   }
   return result;
}

int do_command(int argc, char **argv) {
   switch (atoi(argv[1])) {
      case 1: /* load */
              if (!Do_Load()) {
                 return(0);
              }
              else {
                 return(1);
              }
              break;
      case 2: /* unload */
              if (!Do_Unload()) {
                 return(0);
              }
              else {
                 return(1);
              }
              break;
      case 3: /* move medium */
              if (argc != 5) {
                 fprintf(stderr, "move medium args: ARM SOURCE DESTINATION\n");
                 return(2);
              }
              else {
                 if (!Do_Move(atoi(argv[2]), atoi(argv[3]), atoi(argv[4]))) {
                    return(0);
                 }
                 else {
                    return(1);
                 }
	      }
              break;
      case 5: if (!Do_Load()) {
                 if (!Do_Test_All_Slots(3)) {
                    if (!Do_Unload()) {
                       return 0;
                    }
                    else {
                       return 1;
                    }
                 }
                 else {
                    return 1;
                 }
              }
              else {
                 return(1);
              }
              break; 
      case 6: write_mode = 1;
              test_mode = 1;
              if (!Do_Load()) {
                 if (!Do_Test_All_Slots(1)) {
                    if (!Do_Unload()) {
                       return 0;
                    }
                    else {
                       return 1;
                    }
                 }
                 else {
                    return 1;
                 }
              }
              else {
                 return(1);
              }
              break;
      case 7: if (!Do_Load()) {
                 if (!Do_Test_All_Slots(1)) {
                    if (!Do_Unload()) {
                       return 0;
                    }
                    else {
                       return 1;
                    }
                 }
                 else {
                    return 1;
                 }
              }
              else {
                 return(1);
              }
              break; 
 
      default: fprintf(stderr, "commands are:\n1 load\n2 unload\n3 move medium\n5 label tapes\n6 sequential RW upper unit\n7 sequential upper unit\n");
               return(1);
               break;
   }
}

int tests_main(int argc, char **argv) {
  tape_devices[0] = "/dev/st0";
  tape_devices[1] = "/dev/st1";

  if ((argc < 1) || (argc > 5)) {
     fprintf(stderr, "%s [command args ... args]\n%s: bad args.\n", argv[0], argv[0]);
     return(1);
  }
     
  if (argc > 1) {
     return do_command(argc, argv);
  }

  while (1) {
     int choice;
     int arm, source, destination;

     printf("\n\n1. LOAD\n2. UNLOAD\n3. POSITION\n4. MOVE MEDIUM\n5. TEST MENU\n6. READ ELEMENT STATUS\n7. EXCHANGE MEDIUM\n8. MODE SENSE\n0. QUIT\n");
     if (scanf("%d", &choice) == EOF) { return 0; }

     switch (choice) {
        case 0:
           return(0);
        case 1:
           Do_Load();
           break;
        case 2:
           Do_Unload();
           break;
        case 3:
           printf("\nARM: 0\n");
           arm = 0;
           printf("\nPosition to element [2-9]: ");
           scanf("%d", &destination);
           Do_Position(arm, destination);
           break;
        case 4:
           printf("\nARM: 0\n");
           arm = 0;
           printf("SOURCE element [1, 2, 3-18]: ");
           scanf("%d", &source);
           printf("DESTINATION element [1, 2, 3-18]: ");
           scanf("%d", &destination);

           Do_Move(arm, source, destination);
           break;
        case 5:
           printf("NOTE: tests done with %d slots.\n\n", SLOT_NUMBER);
           printf("TEST MENU\n\n1. Automatic test\n2. Higher sequential\n3. Lower sequential\n4. Random\n5. Label tapes\n6. Test 16x16\n7. Higher/lower sequential.\n8. Alternating test\n9. Change mode (now: Test_Tape %s)\n10. Param test (0x56)\n11. Change tape mode (now: write %s)\n12. ADSM-like test\n13. ADSM-like test 2\n14. Change retry mode (now: retry %s)\n15. Quick alignment test.\n16. Continuous mode (now: continuous %s)\n17. Go to %d slots.\n", test_mode ? "enabled" : "disabled", write_mode ? "enabled" : "disabled", no_retry ? "disabled" : "enabled", continuous ? "enabled" : "disabled", num_slots);
           scanf("%d", &choice);

           switch (choice) {
	   case 1:
             choice = Test_Mode();
	     break;
	   case 2:
             while (choice != -1) {
                if (!Do_Load()) {
                   choice = Do_Test_All_Slots(1);
                   if (Do_Unload()) {
                      choice = -1;
                   }
		 }
                 else {
                    choice = -1;
	         }
                 if (!continuous) break;
              }
	     break;
	   case 3:
             while (choice != -1) {
                if (!Do_Load()) {
                   choice = Do_Test_All_Slots(2);
                   if (Do_Unload()) {
                      choice = -1;
                   }
		 }
                 else {
                    choice = -1;
	         }
                 if (!continuous) break;
              }
              break;
           case 4:
             choice = Do_Test_Random();
             break;
           case 5:
                if (!Do_Load()) {
                   choice = Do_Test_All_Slots(3);
                   if (Do_Unload()) {
                      choice = -1;
                   }
		 }
                 else {
                    choice = -1;
	         }
             break;
           case 6:
             while (choice != -1) {
                if (!Do_Load()) {
                   choice = Do_Test_16x16();
                   if (Do_Unload()) {
                      choice = -1;
                   }
		 }
                 else {
                    choice = -1;
	         }
                 if (!continuous) break;
              }
             break;
          case 7:
             while (choice != -1) {
                if (!Do_Load()) {
                   choice = Do_Test_All_Slots(1);
                   if (choice != -1) {
                      choice = Do_Test_All_Slots(2);
                   }
                   if (Do_Unload()) {
                      choice = -1;
                   }
                }
                 if (!continuous) break;
             }
             break;
          case 8:
             while (choice != -1) {
                if (!Do_Load()) {
                   choice = Do_Test_Alternative();
                   if (Do_Unload()) {
                      choice = -1;
                   }
                }
                if (!continuous) break;
             }
             break;
           case 9:
              test_mode = !test_mode;
              printf("Test_Tape is %s.\n", test_mode ? "enabled" : "disabled");
              break;
           case 10:
              if (!Do_Load()) {
                 choice = Do_Test_0x56();
                 if (Do_Unload()) {
                    choice = -1;
                 }
              }
              else {
                 choice = -1;
              }
              break;
           case 11:
              write_mode = !write_mode;
              printf("Write mode is %s.\n", write_mode ? "enabled" : "disabled");
              break;
           case 12:
	     while (choice != -1) {
              if (!Do_Load()) {
                 choice = Do_Test_ADSM();
                 if (Do_Unload()) {
                    choice = -1;
                    break;
                 }
              }
              else {
                 choice = -1;
                 break;
              }
              if (!continuous) break;
	     }
              break;
           case 13:
	     while (choice != -1) {
              if (!Do_Load()) {
                 choice = Do_Test_ADSM2();
                 if (Do_Unload()) {
                    choice = -1;
                    break;
                 }
              }
              else {
                 choice = -1;
                 break;
              }
              if (!continuous) break;
	     }
              break;
           case 14:
              no_retry = !no_retry;
              printf("Retry  is %s.\n", no_retry ? "disabled" : "enabled" );
              break;

           case 15:
              if (!Do_Load()) {
                 choice = Do_Test_Quick();
                 if (Do_Unload()) {
                    choice = -1;
                 }
              }
              else {
                 choice = -1;
              }
              break;
           case 16:
              continuous = !continuous;
              printf("Continuous is %s.\n", !continuous ? "disabled" : "enabled" );
              break;
           case 17:
              printf("Slot number: ");
              scanf("%d", &num_slots);
              break;
           }
           if (choice == -1) {
              printf("Test mode failed.\n");
           }
           break;
           
        case 6:
           Do_Read_Element_Status();
           break;
        case 7:
           printf("\nARM: 0\n");
           arm = 0;
           printf("SOURCE element [1, 2-9]: ");
           scanf("%d", &source);
           printf("DESTINATION element [1, 2-9]: ");
           scanf("%d", &destination);

           Do_Exchange(arm, source, destination);
           break;
        case 8:
           Do_Mode_Sense();
           break;
        default:
     }
  }
}
