<?php

/* smtp.php: smtp routines
*
    Copyright (C) 2002  Hastymail Development group

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; Version 2
    See the COPYING file for complete information.

* $Id: smtp.php,v 1.98 2004/05/06 14:18:34 sailfrog Exp $
*/

require_once('general.php');

/**
 *  the main function for sending mail, is called from compose.php
 *
 *  @param  $to                 string      to addresses string
 *  @param  $cc                 string      cc addresses string
 *  @param  $bcc                string      bcc addresses string
 *  @param  $subject            string      subject string
 *  @param  $message            string      message string
 *  @param  $from               string      from address string
 *  @param  $reply_inreplyto    string      inreplyto header value for replies
 *  @param  $reply_messid       string      the message id of the message replied to
 *  @param  $reply_refs         string      the references header value for replies
 *  @param  $reply              boolean     true if we are replying to a message
 *  @param  $config             array       the configuration array
 *  @param  $config_file        string      location of configuration file
 *  @param  $bounce             boolean     if the message is a fake message bounce
 *  @param  $version            string      hastymail version number
 *  @param  $hasty_key          string      encryption key
 *  @param  $imap_stream        resource    connection to imap
 *  @return                     boolean     false on failure and true on success
 *
 */
function hm_send_mail($to, $cc, $bcc, $subject, $message, $from, $reply_inreplyto,
                      $reply_messid, $reply_refs, $reply, $config, $config_file,
                      $bounce=false, $version, $hasty_key, $imap_stream) {

    $str = $_SESSION['str'];
    if (!isset($config['smtp_helo_name']) || $config['smtp_helo_name'] == 'false') {
        $hostname = preg_replace("/^http(s){0,1}:\/\//", '', $config['hostname']);
        $hostname = preg_replace("/:\d{1,}$/", '', $hostname);
    }
    else {
        $hostname = $config['smtp_helo_name'];
    }
    $domain = $config['domain'];
    $server = $config['smtp_server'];
    $no_reply_refs = false;
    if ($reply) {
        if ($reply_messid == '') {
            $no_reply_refs = true;
        }
        else {
            $irt = $reply_messid;
            if ($reply_refs != '') {
                $refs = $reply_refs.' '.$reply_messid;
            }
            else {
                if ($reply_inreplyto != '') {
                    $refs = $reply_inreplyto.' ';
                }
                $refs = $reply_messid;
            }
            $refs = preg_replace("/(\t|\r\n)/", ' ', $refs);
            $refs = preg_replace("/\s{2,}/", ' ', $refs);
            $refs_array = split(' ', $refs); 
            $refs = implode("\r\n            ", $refs_array);
        }
    }
    if (!$bounce) {
        $from_address = $_SESSION['profiles'][$from]['from_address'];
        if (isset($_SESSION['profiles'][$from]['from_name']) &&
            $_SESSION['profiles'][$from]['from_name'] != '') {
            $from_name = '"'.$_SESSION['profiles'][$from]['from_name'].'"';
        }
        else {
            $from_name = '';
        }
    }
    else {
        $from_address = $from;  
        $from_name = '';
    }
    
    $lost = $str[179];

    /* connect to the SMTP server */
    if (!$smtp_stream = hm_connect_smtp($server)) {
        return false;
    }
    
    /* say hello */
    if (!hm_smtp_helo($smtp_stream, $hostname, $lost)) {
        if ($smtp_stream) {
            fclose($smtp_stream);
        }
        return false;
    }
    
    /* smtp auth */
    if (isset($config['smtp_auth']) && ($config['smtp_auth'] == 'cram-md5' ||
        $config['smtp_auth'] == 'plain' || $config['smtp_auth'] == 'login')) {
        if(!hm_auth_smtp($smtp_stream, $config['smtp_auth'], $lost, $config_file,
            $hasty_key)) {
            if ($smtp_stream) {
                fclose($smtp_stream);
            }
            return false;
        }
    }
       
    if (!hm_smtp_mail($smtp_stream, $lost, $from_address)) {
        if ($smtp_stream) {
            fclose($smtp_stream);
        }
        return false;
    }
    
    /* parse To:, Cc:, and Bcc: addresses */
    $recpts = array($to, $cc, $bcc);
    foreach ($recpts as $val) { 
        if ($val != '') {
            $val_clean = preg_replace("/(\"[^\"]*\"|<|>|\([^\)]*\))/", '', $val);
            $val_array = preg_split("/(\s*,\s*|\s*;\s*|\s+)/", $val_clean);
            foreach ($val_array as $val_addy) {
                $val_addy = trim($val_addy, ",; ");
                if ($val_addy != '' && strstr($val_addy, '@')) {
                    if (!hm_smtp_rcpt($smtp_stream, $val_addy, $lost)) {
                        $i++;
                    }
                }
            }
        }
    }

    /* check fields for required encodings */
    if (hm_special_char_check($from_name)) {
        $from_name = '=?'.$str['charset'].'?B?'.base64_encode($from_name).'?=';
    }
    if (hm_special_char_check($subject)) {
        $subject = '=?'.$str['charset'].'?B?'.base64_encode($subject).'?=';
    }
    $to_parts = explode(' ', $to);
    if (count($to_parts > 1)) {
        $to = '';
        foreach ($to_parts as $to_part) {
            if (hm_special_char_check($to_part)) {
                $to .=  ' =?'.$str['charset'].'?B?'.base64_encode($to_part).'?=';
            }
            else {
                $to .= ' '.$to_part;
            }
        }
    }
    else {
        if (hm_special_char_check($to)) {
            $to = '=?'.$str['charset'].'?B?'.base64_encode($to).'?=';
        }
    }
    $to = ltrim($to);
    $cc_parts = explode(' ', $cc);
    if (count($cc_parts > 1)) {
        $cc = '';
        foreach ($cc_parts as $cc_part) {
            if (hm_special_char_check($cc_part)) {
                $cc .=  ' =?'.$str['charset'].'?B?'.base64_encode($cc_part).'?=';
            }
            else {
                $cc .= ' '.$cc_part;
            }
        }
    }
    else {
        if (hm_special_char_check($cc)) {
            $cc = '=?'.$str['charset'].'?B?'.base64_encode($cc).'?=';
        }
    }
    $cc = ltrim($cc);
    if (hm_special_char_check($message)) {
        $content_charset = $str['charset'];
    }
    else {
        $content_charset = 'us-ascii';
    }

    /* build message headers and body to send to the SMTP server */
    $smtp_message  = '';
    if (isset($config['ip_header']) && $config['ip_header'] == 'true') {
        $ip = false;
        if (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        }
        elseif (isset($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        }
        else {
            $ip = $_SERVER['REMOTE_ADDR'];
        }
        if ($ip) {
            $serv = $_SERVER['SERVER_NAME'];
            $smtp_message .= "Received: from [$ip] by $serv with HTTP; ".date("r")."\r\n";
        }
    }
    $smtp_message .= 'Message-ID: <'.md5(uniqid(rand(),1))."@$domain>\r\n";
    $smtp_message .= 'Date: '.date("r")."\r\n";
    $smtp_message .= "From: $from_name <$from_address>\r\n";
    $smtp_message .= "Subject: $subject\r\n";
    $smtp_message .= "To: $to\r\n";
    $smtp_message .= "Reply-to: <$from_address>\r\n";
    if ($cc != '') {
        $smtp_message .= "Cc: $cc\r\n";
    }
    if ($reply && !$no_reply_refs) {
        $smtp_message .= "In-Reply-To: ".str_replace(array('&gt;', '&lt;'), 
                                         array('>', '<'), $irt)."\r\n";
        $smtp_message .= "References: ".str_replace(array('&gt;', '&lt;'), 
                                        array('>', '<'), $refs)."\r\n";
    }
    $smtp_message .= "X-Mailer: Hastymail $version\r\n";
    $smtp_message .= "MIME-Version: 1.0\r\n";
    $size = 0;
    $bndry = '';
    
    /* calculations and headers for multi-part messages */
    if (isset($_SESSION['attachments']) && count($_SESSION['attachments']) > 0) {
        $bndry = md5(uniqid(rand(),1));
        foreach($_SESSION['attachments'] as $array) {
            $size += ($array['rawsize'] + 6 + strlen($array['header']) + strlen($bndry));
        }
        $size += (strlen($bndry) + 4);
        $smtp_message .= "Content-Type:multipart/mixed; boundary=\"$bndry\"\r\n\r\n";
        $bndry = "--".$bndry;
        $smtp_message .= "$bndry\r\n";
        $smtp_message .= "Content-Type: text/plain; charset=".$content_charset."\r\n";
        $smtp_message .= "Content-Transfer-Encoding: 8bit\r\n";
        $multi = true;
        
    }
    else {
        $smtp_message .= "Content-type:text/plain; charset=".$content_charset."\r\n";
        $multi = false;
    }

    /* properly format the body text */
    $lines = split("\n", $message);
    $new_lines = array();
    $new_imap_lines = array();
    foreach($lines as $string) {
        if (strlen($string) > 78) {
            $line = rtrim(implode("\r\n", hm_split_line($string, 76, false, true)),
                           "\n")."\r\n";
            $new_imap_lines[] = $line;
            $new_lines[] = preg_replace("/^\.\r\n/", "..\r\n", $line);
        }
        else {
            $line = rtrim($string, "\r\n")."\r\n";
            $new_imap_lines[] = $line;
            $new_lines[] = preg_replace("/^\.\r\n/", "..\r\n", $line);
        }
    }

    /* final formatting before the SMTP data command */
    $body = implode("", $new_lines);
    $imap_body = implode("", $new_imap_lines);
    $imap_message = $smtp_message."\r\n".$imap_body;
    $smtp_message .= "\r\n".$body;
    $size += strlen($imap_message);
    if (isset($_SESSION['forward_flag']) && $_SESSION['forward_flag'] == 'true') {
        unset($_SESSION['forward_flag']);
        $size -= 4;
    }
    if ($reply) {
        $size -= 4;
    }
    if (strlen($imap_body) == 2) {
        $imap_message .= "\r\n\r\n";
        $size -= 4;
    }

    /* send the headers/body */
    $no_notice = false;
    if ($reply || $bounce) {
        $no_notice = true;
    }
    if (!hm_smtp_data($smtp_stream, $smtp_message, $lost, $no_notice, $multi, $bndry)) {
        if (is_resource($smtp_stream)) {
            fclose($smtp_stream);
        }
        return false;
    }
    /* clean up */
    else {
        fputs($smtp_stream, "QUIT\r\n");
        fclose($smtp_stream);
        $config = $_SESSION['config'];
        hm_save_message($imap_message, $imap_stream, 'sent', $multi, $bndry, $size);
        hm_imap_logout($imap_stream); 
        return true;
    }
}

/**
 *  connect to the SMTP server
 *
 *  @param      $server     string      smtp server address or hostname
 *  @return                 resource    false on failure
 *
 */
function hm_connect_smtp($server) {
    $str = $_SESSION['str'];
    if (isset($_SESSION['config']['smtp_port'])) {
        $port = $_SESSION['config']['smtp_port'];
    }
    else {
        $port = 25;
    }
    if (isset($_SESSION['config']['smtp_ssl']) && 
        $_SESSION['config']['smtp_ssl'] == 'ssl') {
        $server = 'ssl://'.$server;
    }
    elseif (isset($_SESSION['config']['smtp_ssl']) && 
        $_SESSION['config']['smtp_ssl'] == 'tls') {
        $server = 'tls://'.$server;
    }
    if ($smtp_stream = @fsockopen($server, $port, $errorno, $errorstr, 30)) {
        if ($helo = fgets($smtp_stream, 1024)) {
            if (substr($helo, 0, 4) == '220 ') {
                return $smtp_stream;
            }
            elseif (substr($helo, 0, 4) == '220-') {
                while (substr($helo, 0, 4) == '220-') {
                    $helo = fgets($smtp_stream, 1024);
                }
                return $smtp_stream;
            }
            else {
                $_SESSION['notices'][] = $str[180].':<br>'.$helo;
                return false;
            }
        }
        else {
            $_SESSION['notices'][] = $str[181];
            return false;
        }
    }
    else {
        $_SESSION['notices'][] = $str[181];
        return false;
    }
}
    
/**
 *  sends the SMTP HELO command
 *
 *  @param      $smtp_stream    resource    connection to the smtp server
 *  @param      $hostname       string      the hostname we are sending mail from
 *  @return                     boolean     true on success, false on failure
 *
 */
function hm_smtp_helo($smtp_stream, $hostname) {
    $str = $_SESSION['str'];
    if (isset($_SESSION['config']['smtp_auth']) && 
        ($_SESSION['config']['smtp_auth'] == 'cram-md5' ||
        $_SESSION['config']['smtp_auth'] == 'plain' || 
        $_SESSION['config']['smtp_auth'] == 'login')) {
        fputs($smtp_stream, "EHLO $hostname\r\n");
    }
    else {
        fputs($smtp_stream, "HELO $hostname\r\n");
    }
    if ($response = fgets($smtp_stream, 1024)) {
        if (substr($response, 0, 4) == '250 ') {
            return true;
        }
        elseif (substr($response, 0, 4) == '250-') {
            while (substr($response, 0, 4) == '250-') {
                $response = fgets($smtp_stream, 1024);
            }
            return true;
        }
        else {
            $_SESSION['notices'][] = $str[182].':<br> '.$response;
            return false;
        }
        
    }
    else {
        $_SESSION['notices'][] = $lost;
        return false;
    }
}

/**
 *  smtp authentication routine
 *
 *  @param  $smtp_stream    resource    connection to smtp server
 *  @param  $auth           string      authentication type
 *  @param  $lost           string      failure to connect message
 *  @param  $config_file    string      configuration file location
 *  @param  $hasty_key      string      encryption key
 *  @return                 boolean     true on success, false on failure
 *
 */
function hm_auth_smtp($smtp_stream, $auth, $lost, $config_file, $hasty_key) {
    $str = $_SESSION['str'];
    if ($auth != 'plain') {
        $command = 'AUTH '.strtoupper($auth)."\r\n";
    }
    else {
        $key = hmw_crypt($_SESSION['pass'], $config, 'decrypt', $hasty_key);
        $user = $_SESSION['user'];
        $command = 'AUTH PLAIN '.base64_encode($user."\0".$user."\0".$key)."\r\n";
    }
    fputs($smtp_stream, $command);
    if ($response = fgets($smtp_stream, 1024)) {
        if (substr($response, 0, 3) != '334' && $auth != 'plain') {
            $_SESSION['notices'][] = $str[234].':<br>'.$response;
            return false;
        }
        else {
            switch ($auth) {
                case 'cram-md5':
                    $challenge = base64_decode(substr($response, 4));
                    $user = $_SESSION['user'];
                    $key = hmw_crypt($_SESSION['pass'], $config, 'decrypt', $hasty_key); 
                    $key .= str_repeat(chr(0x00), (64-strlen($key)));
                    $ipad = str_repeat(chr(0x36), 64);
                    $opad = str_repeat(chr(0x5c), 64);
                    $digest = bin2hex(pack("H*", md5(($key ^ $opad).
                              pack("H*", md5(($key ^ $ipad).$challenge)))));
                    $challenge_response = base64_encode($user.' '.$digest);
                    fputs($smtp_stream, $challenge_response."\r\n");
                    $result = fgets($smtp_stream, 1024);
                    break;
                case 'login':
                    $key = hmw_crypt($_SESSION['pass'], $config, 'decrypt', $hasty_key);
                    $user = $_SESSION['user'];
                    fputs($smtp_stream, base64_encode($user)."\r\n");
                    $prompt = fgets($smtp_stream, 1024);
                    fputs($smtp_stream, base64_encode($key)."\r\n");
                    $result = fgets($smtp_stream, 1024);
                    break;
                case 'plain':
                    $result = $response;
                    break;
                default:
                   $result = $response;
                    break;
            }
            if (substr($result, 0, 3) != '235') {
                $_SESSION['notices'][] = $str[234].':<br> '.$result;
                return false;
            }
            else {
                return true;
            }
        }
    }
    else {
        $_SESSION['notices'][] = $lost;
        return false;
    }
}

/**
 *  performs the SMTP MAIL command
 *
 *  @param  $smtp_stream    resource    connection to the smtp server
 *  @param  $lost           string      failure to connect message
 *  @param  $from           string      the senders email address
 *  @return                 boolean     true on success, false on failure
 *
 */
function hm_smtp_mail ($smtp_stream, $lost, $from) {
    $str = $_SESSION['str'];
    fputs($smtp_stream, "MAIL FROM: <$from>\r\n");
    if ($response = fgets($smtp_stream, 1024)) {
        if (preg_match("/^250/", $response)) {
            return true;
        }
        else {
            $_SESSION['notices'][] = $str[183].':<br> '.$response;
            return false;
        }
    }
    else {
        $_SESSION['notices'][] = $lost;
        return false;
    }
}

/**
 *  performs the SMTP RCPT command
 *
 *  @param      $smtp_stream    resource    connection to the smtp server
 *  @param      $lost           string      failure to connect message
 *  @param      $to             string      a single to address
 *  @return                     boolean     true on success, false on failure
 *
 */
function hm_smtp_rcpt ($smtp_stream, $to, $lost) {
    $str = $_SESSION['str'];
    //$dmpart = strrev(strrev(strstr($to, '@')));
    fputs($smtp_stream, "RCPT TO: <$to>\r\n");
    if ($response = fgets($smtp_stream, 1024)) {
        if (preg_match("/^25/", $response)) {
            return true;
        }
        else {
            $_SESSION['notices'][] = $str[262].':<br>'.$response;
            return false;
        }
        
    }
    else {
        $_SESSION['notices'][] = $lost;
        return false;
    }
}

/**
 *  sends the SMTP data (headers and body)
 *
 *  @param      $smtp_stream    resource    connection to the smtp server
 *  @param      $lost           string      failure to connect message
 *  @param      $message        string      the message to send
 *  @param      $reply          boolean     true if the message is a reply
 *  @param      $multi          boolean     true if the message has attachments
 *  @param      $bndry          string      mime boundary
 *  @return                     boolean     true on success, false on failure
 *
 */
function hm_smtp_data ($smtp_stream, $message, $lost, $reply, $multi, $bndry) {
    $str = $_SESSION['str'];
    fputs($smtp_stream, "DATA\r\n");
    if ($response = fgets($smtp_stream, 1024)) {
        if (preg_match("/^354/", $response)) {
            if ($multi) {
                if(!hm_write_data($smtp_stream, $message, $bndry)) {
                    echo $lost; 
                    return false;
                }
                else {
                    fputs($smtp_stream, "\r\n.\r\n");
                }
            }
            else {
                    fputs($smtp_stream, $message."\r\n.\r\n");
            }
            if ($final_response = fgets($smtp_stream, 1024)) {
                if (preg_match("/^250/", $final_response)) {
                    if (!$reply) {
                        $_SESSION['notices'][] = $str[184];
                    }
                    return true;
                }
                else {
                    $_SESSION['notices'][] = $str[185].':<br>'.$final_response;
                    return false;
                }
            }
            else {
                $_SESSION['notices'][] = $lost; 
                return false;
            }
        }
        else {
            $_SESSION['notices'][] = $str[185].':<br>'.$response;
            return false;
        }
        
    }
    else {
        $_SESSION['notices'][] = $lost;
        return false;
    }
}

/**
 *  checks for non-ascii chars.
 *
 *  @param      $string     string      string to check
 *  @return                 boolean     true if a char over ASCII code 127 is found
 *
 */
function hm_special_char_check($string) {
    $result = false;
    $len = strlen($string);
    for ($i=0;$i<$len;$i++) {
        if (ord($string{$i}) > 127) {
            $result = true;
            break;
        }
    }
    return $result;
}
?>
