#! /bin/sh
# MVM Project (C) Marc SCHAEFER <schaefer@alphanet.ch>
# mvm_vgetty_script.sh -- Main script run from vgetty
# Creation: 14/02/98
# Update:   01/03/99
# V2.0 PV006
# DISCLAIMER
#    No warranty, either expressed or implied.
# COPYRIGHT
#    Fully protected under Berne Convention, use authorized except
#    if you make money from this use (contact me before).
# DESCRIPTION
#    This shell script controls interaction with the vgetty voice modem
#    control library (through a pipe). It replaces the voice_script.sh.
# BUGS
# TODO
# NOTES
# BASED-ON
#    voice_script.sh V1.3 PV004 of 08/02/98.
# MODIFICATION-HISTORY
#    14/02/98  schaefer  Created this file.
#    12/04/98  schaefer  better do_param_cmd
#    29/12/98  schaefer  Save the recorded message even if user hangups.
#                        Also support for no voicemail, no set language, etc.
#    15/01/99  schaefer  GO_FAX support from script.
#    01/03/99  schaefer  Fixed bug in ask_dtmf() which was causing crashes
#                        with some patterns of code entering, especially
#                        while playing a file. Simplified the routine.
#                        Added comments.
# $Id: mvm_vgetty_script.sh,v 1.10 1999/03/01 17:48:36 schaefer Exp $

env > /tmp/mvm_env

. /etc/mvm_definitions.sh
. $SCRIPTS_DIR/mvm_low_level_library.sh


# 0  not identified
# 1  identified
# 100 sub-operator
# NOTES
#    - Operator level is only for user 0 (CURRENT_USER==0)
#    - Cannot be set by do_login.
USER_LEVEL=0

# Sets the default area to none
CURRENT_AREA=

# Low level vgetty interface functions

function receive {
   read -r INPUT <&$VOICE_INPUT;
   echo "$INPUT";
}

function send {
   echo $1 >&$VOICE_OUTPUT;
   kill -PIPE $VOICE_PID
}

# SYNOPSIS
#    result = say_message(...)
# RESULT
#    - see playfile()
# DESCRIPTION
#    Generate numbers language independantly.
# LOCAL-VARIABLES
#    $TMPFILE.5
#    $RESULT
# NOTE
# BUGS
#    - cat should support multilanguage.
# TODO
function say_message {
   RESULT=`generic_numbers_to_audio_file $TMPFILE.6 $*`
   if [ "$RESULT" = "OK" ]; then
      RESULT=`play_file $TMPFILE.6`
      echo "$RESULT"
   else
      # Do not pass further.
      echo "READY"
   fi
   rm -f $TMPFILE.6
}

# SYNOPSIS
#    result = tell_id(id)
# RESULT
#    will be in the form 
#       - id
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function communicates an account ID
# BUGS
#    What about stderr (eg, rm)
# LOCAL-VARIABLES
#    The ANSWER and RESULT variables.
function tell_id {
   RESULT=`play_file $ACCOUNT_MADE_FILE`
   case $RESULT in
      READY) RESULT=`say_message --pre-formatted $YOUR_ACCOUNT_NUMBER_IS --say-number $1`
             case $RESULT in
                READY) RESULT=$1;; 
                *) ;;
             esac;;
      *) ;;
   esac
   echo "$RESULT"
}

# SYNOPSIS
#    result = beep()
# RESULT
#    will be in the form 
#       - READY
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function beeps.
# LOCAL-VARIABLES
#    The RESULT variable.

function beep {
   send "BEEP"
   RESULT=`receive`
   case $RESULT in
      BEEPING)       RESULT=`receive`
                     case $RESULT in
                        READY)         ;;
                        RECEIVED_DTMF) RESULT=`receive` # loop ?
                                       RESULT=ABORTED;;
                     esac;;
      *)             RESULT=END;; 
      RECEIVED_DTMF) RESULT=`receive` # loop ?
                     RESULT=ABORTED;;
      *) RESULT=END;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = edit_user()
# RESULT
#    will be in the form 
#       - READY
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function edits a specific user, which is asked
#    first. All operations are done here, except for
#    access level which must be done by calling procedure,
#    usually the edit_user menu item).
#    Current implemented items are:
#       1 show/change level
#       2 show/change name
#       3 show/change information
#       4 set new password
#       9 deletes the user
#       # terminates the editing (no fallback!)
# BUGS
#    - What about stderr (eg, rm)
# LOCAL-VARIABLES
#    The RESULT variable.
# NOTES
# TODO
#    - To implement

function edit_user {
   RESULT=`play_file $EDIT_USER_FILE`
   echo "$RESULT"
}

# SYNOPSIS
#    result = ask_item(banner)
# RESULT
#    will be in the form 
#       - either any DTMF (one single char)
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function plays the description, then
#    waits for a single DTMF. It then returns
#    it. 
# LOCAL-VARIABLES
#    The RESULT variable.
# NOTES
# BUGS
# TODO
function ask_item {
   RESULT=`play_file_and_ask_dtmf $1`
   echo "$RESULT"
}

# SYNOPSIS
#    result = ask_menu_type(banner)
# RESULT
#    will be in the form 
#       - [a-z]* for menu type
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function plays the description, then
#    waits for a DTMF code which is interpreted
#    as a menu item type.
# LOCAL-VARIABLES
#    The ANSWER, ANSWER2 and RESULT variables.
# NOTES
# BUGS
#    - no nonexistant item notification (returns ABORTED)
# TODO
function ask_menu_type {
   RESULT=`ask_code $1`
   case $RESULT in
      [0-9]*) RESULT=`grep "^$RESULT " $EDIT_MENU_FILE_TYPE`
              case $RESULT in
                 *_*) RESULT=`echo $RESULT | awk '{print $2}'`;;
                 *)   RESULT=ABORTED;;
              esac;;
   esac
   echo "$RESULT"   
}

# SYNOPSIS
#    result = edit_menu_search_item(menupath, item)
# RESULT
#    will be in the form 
#       - READY
#       - NOT_FOUND
# DESCRIPTION
#    This functions searches for a specific menu item in a menu.
# BUGS
# LOCAL-VARIABLES
# NOTES
# TODO
function edit_menu_search_item {
   grep "^$2 " $1/commands > /dev/null
   if [ $? = 0 ]; then
      echo "READY"
   else
      echo "NOT_FOUND"
   fi
}

# SYNOPSIS
#    result = edit_menu_new_item(menupath)
# RESULT
#    will be in the form 
#       - READY
#       - or END
# DESCRIPTION
#    This function creates a new menu item. It first asks for
#    the menu identifier, verifies that this item is not yet
#    used, and then asks the type and the level, and, if needed,
#    a secondary argument. It then adds the item to the menu
#    and returns. We don't do any user level check.
# BUGS
#    - no secondary argument.
# LOCAL-VARIABLES
#    The RESULT, ANSWER variable.
# NOTES
# TODO
function edit_menu_new_item {
   RESULT=`ask_item $EDIT_MENU_ITEM`
   case $RESULT in
      ?)      if [ "`edit_menu_search_item $1 $RESULT`" = "NOT_FOUND" ]; then
                 ANSWER=$RESULT
                 RESULT=`ask_menu_type $EDIT_MENU_ITEM_TYPE`
                 case $RESULT in
                    [a-z]*)  ANSWER="$ANSWER $RESULT"
                             RESULT=`ask_code $EDIT_MENU_ITEM_LEVEL`
                             case $RESULT in
                                [0-9]*)  echo "$ANSWER $RESULT" >> $1/commands
                                         RESULT=READY;;
                                ABORTED) RESULT=READY;;
                             esac;;
                    ABORTED) RESULT=READY;;
                 esac
	      else
		 RESULT=`play_file $EDIT_MENU_ITEM_EXISTS`
	      fi;;
       END)   ;;
       *)     RESULT=ABORTED;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = edit_menu_kill_item(menupath)
# RESULT
#    will be in the form 
#       - READY
#       - or END
# DESCRIPTION
#    This function deletes an existing menu (which is introduced by
#    the user) item if it finds it.
#    We don't do any user level check.
# BUGS
#    - missing confirmation
# LOCAL-VARIABLES
#    The RESULT variable.
# NOTES
# TODO
function edit_menu_kill_item {
   RESULT=`ask_item $EDIT_MENU_ITEM`
   case $RESULT in
         ?) if [ "`edit_menu_search_item $1 $RESULT`" = "READY" ]; then
		 grep -v "^$RESULT " < $1/commands > $1/commands.new
		 mv $1/commands.new $1/commands
	      else
		 RESULT=`play_file $EDIT_MENU_NO_SUCH_ITEM`
	      fi;;
       END)   ;;
       *)     RESULT=ABORTED;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = edit_menu(menu_path)
# RESULT
#    will be in the form 
#       - READY
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function edits a specific menu. It is not
#    itself implemented yet as a menu (see BUGS).
#    It will not check itself the access level (must
#    be done by calling procedure, usually the edit_menu
#    menu item).
#    Current implemented items are:
#       1 adds an item (can be a goto_menu, then asks owner,
#         or any type asked from a list)
#       2 removes an item (can be a goto_menu, the system will then
#         ask if removing the branch is asked)
#       3 tests the menu
#       4 edit the welcome file
#       * cancel changes
#       # accept changes and back
# BUGS
#    - What about stderr (eg, rm)
#    - What about implementing this also as a menu and
#      use menu items checking the level each ? This
#      would mean maintaining a LAST_MENU variable.
#      Editing the edit_menu would also be possible.
#    - It is possible to remove the edit menu function
#      from a menu. Oups.
#    - Moving menus and some operations can only be done
#      at the command line level.
# LOCAL-VARIABLES
#    The RESULT and TMPMENU variable.
# NOTES
#    - It does allow to add an item to a menu which is not allowed
#    to use by the current user. Checks will be done at run-time.
#    - Menu cannot be deleted from within the edit_menu. This means
#    that an arborescence starting from one user cannot be deleted
#    else by the user. (think of it). Thus we do not need a back_menu
#    result. And as menus are not cached, this is also OK.
#    - Now supports hotkeying.
# TODO
#    - 4. not implemented (edit welcome message)
function edit_menu {
   TMPMENU=/tmp/tmpmenu_$$
   mkdir $TMPMENU
   if [ -f $1/commands ]; then
      cp $1/* $TMPMENU
   else
      # default settings
      mkdir $1 # creates dir, will NOT be deleted
               # on abort --- BUG
      cat > $TMPMENU/commands <<FINGLOP
9 edit_menu 100
0 goodbye 0
# message_to_operator 0
+ back_menu 0
FINGLOP
   fi

   while :
   do
      RESULT=`play_file_and_ask_dtmf $EDIT_MENU_FILE`
      case $RESULT in
         1) RESULT=`edit_menu_new_item $TMPMENU`
            case $RESULT in
               READY) ;;
               *)     break;;
            esac;;	 
         2) RESULT=`edit_menu_kill_item $TMPMENU`
            case $RESULT in
               READY) ;;
               *)      break;;
            esac;;
         3) RESULT=`do_voice_mail $TMPMENU`
            case $RESULT in
               READY|goodbye) ;;
               *)             break;;
            esac;;
         4) RESULT=`record_to_file $RECORD_TO_FILE $TMPMENU/welcome`
            case $RESULT in
               READY) ;;
               ABORTED) RESULT=READY;;
               *)     break;;
            esac;;
         +)  RESULT=READY
             break;;
         \#) RESULT=READY
             cp $TMPMENU/* $1
             break;;
         *)  break;;
      esac
   done
   rm -rf $TMPMENU
   echo "$RESULT"
}

# SYNOPSIS
#    result = do_login()
# RESULT
#    will be in the form 
#       - account ID
#       - or ABORTED
#       - or END
#       - or NONE if account could not be accessed
# DESCRIPTION
#    This function logins to an existant account.
# BUGS
#    What about stderr (eg, rm)
# LOCAL-VARIABLES
#    The ANSWER, ANSWER2 and RESULT variables.
# TODO
function do_login {
   RESULT=`ask_code $ASK_CODE_FILE`
   case $RESULT in
      [0-9]*) ANSWER=`ask_code $ASK_PASSWORD_FILE`
              case $ANSWER in
                 [0-9]*) ANSWER2=$RESULT
                         RESULT=`check_password $RESULT $ANSWER`
                         case $RESULT in
                            INEXISTANT|WRONG) RESULT=NONE;;
                            OK)               RESULT=$ANSWER2;; 
                         esac;;
                 *)      RESULT=$ANSWER;;
              esac;;
       END)   ;;
       *)     RESULT=ABORTED;; # We do not accept empty case.
   esac

   # we had a hang bug here. I enable it again to check it again.
   # seems to work now.
   if [ "1" = "1" ]; then
   case $RESULT in
      NONE)    ANSWER=`play_file $PASSWORD_ERROR_FILE`
               case $ANSWER in
                  READY) ;;
                  *)     RESULT=$ANSWER;;
               esac;;
      [0-9]*)  ANSWER=`play_file $PASSWORD_OK_FILE`
               case $ANSWER in
                  READY) ;;
                  *)     RESULT=$ANSWER;;
               esac;; 
   esac
   fi

   echo "$RESULT"
}

# SYNOPSIS
#    result = send_mail(toaccountID, fromaccountID, askconfirmation)
# RESULT
#    will be in the form 
#       - READY
#       - INEXISTANT
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function allows to send a voice mail message.
# BUGS
#    - What about stderr (eg, rm)
#    - risk of collision with send_mail(1) ? -> use send_mail instead
# LOCAL-VARIABLES
#    The RESULT and IGNORE variable.
# TODO
function send_mail {
   case $3 in
      YES) if [ -e $MAILBOX_DIR/$1/.name ]; then
	      RESULT=`play_file $MAILBOX_DIR/$1/.name`
	      case $RESULT in
		 READY) RESULT=`confirm`
			case $RESULT in
			   YES) RESULT=READY;;
                           NO)  RESULT=ABORTED;;
			esac;;
	      esac
	   else
	      RESULT=INEXISTANT
	   fi;;
       *)  RESULT=READY;;
   esac
   case $RESULT in
      READY) RESULT=$TMPFILE.sm
             RESULT=`record_to_file $LEAVE_MESSAGE_FILE $RESULT`
             # Always send the mail, even if it's partial.
             if [ -f $TMPFILE.sm ]; then
                IGNORE=`send_voice_mail $1 $2 $TMPFILE.sm NO`
                rm -f $TMPFILE.sm
             fi;;
   esac
   echo "$RESULT"
}

# SYNOPSIS
#    result = do_send_mail()
# RESULT
#    will be in the form 
#       - READY
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function allows to send a voice mail message
#    by asking for destinator, and doing all checks.
# BUGS
#    - What about stderr (eg, rm)
#    - What if CURRENT_USER is nothing ?
# TODO
# LOCAL-VARIABLES
#    The ANSWER, and RESULT variables.
function dosend_mail {
   RESULT=`ask_code $WHICH_RECIPIENT_FILE`
   case $RESULT in
      [0-9]*) ;;
      *)      RESULT=0;; # operator (note that [0-9]* does not accept empty)
   esac
   case $RESULT in
      [0-9]*) RESULT=`send_mail $RESULT $CURRENT_USER YES`
              case $RESULT in
                 INEXISTANT) RESULT=`play_file $NO_SUCH_RECIPIENT_FILE`;;
              esac ;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = read_mail(accountID)
# RESULT
#    will be in the form 
#       - READY
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function allows to browse through voice messages
# BUGS
#    - What about stderr (eg, rm)
#    - FORWARD not implemented.
# TODO
#    - We could add the spoken name of the sender, ability
#      to reply, forward, and so on, remove capabilities, etc.
# LOCAL-VARIABLES
#    The ANSWER, ANSWER2, NUMBERM, and RESULT variables.
function read_mail {
   NUMBERM=`ls -1 $MAILBOX_DIR/$1/ | wc -l`
   NUMBERM=`echo $NUMBERM`
   case $NUMBERM in
      0) RESULT=`play_file $NO_MESSAGES_FILE`;;
      1) RESULT=`play_file $ONE_MESSAGE_FILE`;;
      *) RESULT=`say_message --pre-formatted $YOU_HAVE_SOUND --say-number $NUMBERM --pre-formatted $YOU_HAVE_MESSAGES`;;
   esac
   case $RESULT in
      READY) case $NUMBERM in
             0) ;;
             *) sleep 1
                for ANSWER2 in $MAILBOX_DIR/$1/*
                do
                   while :
                   do
		      NUMBERM=`echo $ANSWER2 | awk -F_ '{print $4}'` # recycle vars !
		      ANSWER=`message_date $ANSWER2`
                      ANSWER=`english_text_to_audio_file "$ANSWER" $TMPFILE.0`
                      RESULT=`say_message --pre-formatted $NEXT_MESSAGE_FROM --say-number $NUMBERM --pre-formatted $WAS_RECORDED_ON --pre-formatted $TMPFILE.0`
		      rm -f $TMPFILE.0
		      case $RESULT in
			 READY) sleep 1
				RESULT=`play_file $ANSWER2`
				case $RESULT in
				   READY) sleep 1
                                          RESULT=`message_command`
                                          case $RESULT in
                                             NEXT)   RESULT=READY; break;;
                                             AGAIN|FORWARD)  RESULT=READY; continue;;
                                             DELETE) RESULT=READY
                                                     rm -f $ANSWER2
                                                     break;;
                                             REPLY) RESULT=`send_mail $NUMBERM $CURRENT_USER NO`
                                                    case $RESULT in
                                                       READY) ;;
                                                       INEXISTANT) RESULT=`play_file $NO_SUCH_RECIPIENT_FILE`
                                                                   case $RESULT in
                                                                      READY) ;;
                                                                      *)     break;;
                                                                   esac;;
                                                       *)     break;;
                                                    esac
                                                    ;;
                                             *) break;;
                                          esac;;
				   *) break;;
				esac;;
			 *) break;;
		      esac
                   done
                   case $RESULT in
                      READY) ;;
                      *)     break;;
                   esac
                done
                case $RESULT in
                   READY) RESULT=`play_file $END_OF_MESSAGES_FILE`;;
                esac;;
             esac;;
   esac
   echo "$RESULT"
}   

# SYNOPSIS
#    result = create_account()
# RESULT
#    will be in the form 
#       - account ID
#       - or ABORTED
#       - or END
#       - or NONE if account could not be created
# DESCRIPTION
#    This function creates a new account.
# BUGS
#    - What about stderr (eg, rm)
#    - [0-9]* matches a non-empty sequence. Thus, an
#      empty sequence will make this function return nothing
#      which will be handled correctly by the main loop, but
#      this is a bug.
#    - use the confirm function instead of doing it by hand.
# NOTES
#    - Now supports hot-keying.
# LOCAL-VARIABLES
#    The ANSWER and RESULT variables.
#    The SELECTED_PASSWORD, USER_NAME_TMP, INFO_PERS_TMP variables.
function create_account {
   RESULT=`play_file_and_ask_dtmf $MAKE_ACCOUNT_FILE`
   case $RESULT in
      1) RESULT=`ask_code $CHOOSE_PASSWORD_FILE`
         case $RESULT in
            [0-9]*) SELECTED_PASSWORD=$RESULT
                    RESULT=`ask_code $VERIFY_PASSWORD_FILE`
                    case $RESULT in
                       [0-9]*) if [ "$RESULT" = "$SELECTED_PASSWORD" ]; then
                                  USER_NAME_TMP=/tmp/usern_$$
                                  RESULT=`record_to_file $USER_NAME_FILE $USER_NAME_TMP`
                                  case $RESULT in
                                     READY) INFO_PERS_TMP=/tmp/infopers_$$
                                            RESULT=`record_to_file $PERSONAL_INFO_FILE $INFO_PERS_TMP`
                                            echo "mvm: create_account: info_pers result $RESULT" >&2
                                            case $RESULT in
                                               READY) RESULT=`play_file $FINDING_ID_FILE`
                                                      case $RESULT in
                                                         READY) RESULT=`make_account $SELECTED_PASSWORD $USER_NAME_TMP $INFO_PERS_TMP`
                                                                echo "mvm: create_account: make_account result $RESULT" >&2
                                                                rm -f $USER_NAME_TMP $INFO_PERS_TMP
                                                                case $RESULT in
                                                                   NONE) ;;
                                                                   *) RESULT=`tell_id $RESULT`
                                                                      echo "mvm: create_account: tell_id result $RESULT" >&2
                                                                      case $RESULT in
                                                                         [0-9]*) ;;
                                                                         *) # kill account
                                                                            ;;
                                                                      esac;;
                                                                esac;;
                                                         *) rm -f $USER_NAME_TMP $INFO_PERS_TMP;;
                                                      esac;;
                                               *) rm -f $USER_NAME_TMP;;
                                            esac;;
                                  esac
                               else
                                  RESULT=`play_file $PASSWORDS_DO_NOT_MATCH_FILE`
                                  case $RESULT in
                                     READY) RESULT=NONE;;
                                  esac
                               fi;;
                    esac;;
                *)       RESULT=ABORTED;;
         esac;;
      [0-9+\#]) RESULT=ABORTED;;
   esac
   echo "$RESULT"
}

# SYNOPSIS
#    result = play_file_and_ask_dtmf(file)
# RESULT
#    will be in the form 
#       - either [0-9+\#]
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function returns ONE DTMF digit.
#    Abort is when the input delays terminates.
# LOCAL-VARIABLES
#    The RESULT variable.
# NOTES
#    - star is transformed into +
#    - We support one hot-key during the play.
function play_file_and_ask_dtmf {
   RESULT=`play_file_and_dtmf $1`
   case $RESULT in
      READY) RESULT=`ask_dtmf`;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = ask_dtmf()
# RESULT
#    will be in the form 
#       - either [0-9+\#]
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function returns ONE DTMF digit.
#    Abort is when the input delays terminates.
# LOCAL-VARIABLES
#    The ANSWER, and RESULT variables.
# NOTES
#    - star is transformed into +
function ask_dtmf {
   RESULT=

   send "WAIT 20"
   ANSWER=`receive`
   echo "mvm: ask_dtmf: received $ANSWER" >&2
   case $ANSWER in
      WAITING|READY) ANSWER=`receive`
                     echo "mvm: ask_dtmf: received S2 $ANSWER" >&2
                     # No need to stop since in AUTOSTOP mode.

                     case $ANSWER in
                        RECEIVED_DTMF) ANSWER=`receive`
                                       echo "mvm: ask_dtmf: received S3 $ANSWER" >&2
                                       RESULT=$ANSWER
                                       ANSWER=`receive`
                                       echo "mvm: ask_dtmf: received S4 $ANSWER" >&2;;
                        READY|NO_VOICE_ENERGY|SILENCE_DETECTED) RESULT=ABORTED
                                       # Timeout
                                       ;;
                        *)             RESULT=END;;
                    esac;;
      RECEIVED_DTMF) ANSWER=`receive`
                     echo "mvm: ask_dtmf: received S5 $ANSWER" >&2
                     RESULT=ABORTED;;
      *)             RESULT=END;;
   esac

   if [ "$RESULT" = "*" ]; then
      RESULT=+
   fi

   echo "mvm: ask_dtmf: result $RESULT" >&2
   echo "$RESULT"
}

# SYNOPSIS
#    result = record_to_file(banner, tofile)
# RESULT
#       - READY
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This functions plays the description, then
#    records a file until any DTMF is received.
# LOCAL-VARIABLES
#    The ANSWER, RESULT variables.
#    $TMPFILE.4
# BUGS
#    - Does not support multiple DTMF
#    - stderr is used (rm)
#    - beep missing
function record_to_file {
   RESULT=`play_file $1`
   case $RESULT in
      READY) RESULT=`beep`
             case $RESULT in
                READY) send "RECORD $TMPFILE.4"
		       ANSWER=`receive`
		       case $ANSWER in
			  RECORDING)     RESULT=`receive`
					 case $RESULT in
					    READY)         RESULT=`beep`;;
					    RECEIVED_DTMF) ANSWER=`receive` # eat key
							   send "STOP"
							   RESULT=`receive` # may need an eat_dtmf loop here
                                                           RESULT=`beep`
							   ;;
					    *)             RESULT=END
                                                           rm -f $2;;              
					 esac;;
			  RECEIVED_DTMF) ANSWER=`receive` # eat key
					 send "STOP"
					 RESULT=`receive` # may need an eat_dtmf loop here;;
					 case $RESULT in
					    READY) RESULT=ABORTED;;
					 esac;;
                          *)             RESULT=END;;
		       esac
                       if [ "$ENABLE_1496_RECORD_BUG_FIX" != "" ]; then
                          if [ -e $TMPFILE.4 ]; then
                             $PVF_TOOLS_DIR/rmdtopvf < $TMPFILE.4 | $PVF_TOOLS_DIR/$ENABLE_1496_RECORD_BUG_FIX | $PVF_TOOLS_DIR/$PVF_MODEM_FILTER > $2
                             rm $TMPFILE.4
                          fi
                       fi

                       if [ -e $TMPFILE.4 ]; then
                          mv $TMPFILE.4 $2
                       fi;;
                *) ;;
             esac;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = do_param_cmd(prompt, script, confirm, error)
# RESULT
#       - READY
#       - or ABORTED
#       - or END
#       - or GO_FAX
# DESCRIPTION
#    This functions plays the prompt, then retrieves a DTMF
#    string (* is used as abort, # as finish), passes it to
#    the named script in the appropriate script directory (just
#    the numbers, A-D are disallowed), and then if the script
#    returns 0, plays the confirm, else plays an error.
#    If the script returns 3 or 0, returns GO_FAX.
# LOCAL-VARIABLES
#    The ANSWER, RESULT variables.
# BUGS
function do_param_cmd {
   RESULT=`ask_code $VOICEMAIL/messages/$1`
   if [ "$RESULT" = "" ]; then
      RESULT=+
   fi

   case $RESULT in
      [0-9]*) $SCRIPTS_DIR/$2 "$RESULT" "$CURRENT_USER"
              case $? in
                 0) RESULT=`play_file $VOICEMAIL/messages/$3`
                    if [ "$RESULT" = "READY" ]; then
                       RESULT=GO_FAX
                    fi;;
                 3) RESULT=GO_FAX;;
                 *) RESULT=`play_file $VOICEMAIL/messages/$4`;;
              esac;;
      +)      RESULT=ABORTED
              ;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = confirm()
# RESULT
#    will be in the form 
#       - YES
#       - NO
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This functions plays a standard confirm message,
#    and ask for confirmation (1), any other key returns NO.
# LOCAL-VARIABLES
#    The RESULT variable.
function confirm {
   RESULT=`play_file $CONFIRM_FILE`
   case $RESULT in
      READY) RESULT=`ask_dtmf`
             case $RESULT in
                END|ABORTED) ;;
                *)           if [ "$RESULT" = "1" ]; then
                                RESULT=YES
                             else
                                RESULT=NO
                             fi;;
             esac;;
   esac

   echo "$RESULT"
}

# SYNOPSIS
#    result = message_command()
# RESULT
#    will be in the form 
#       - NEXT
#       - DELETE
#       - AGAIN
#       - REPLY
#       - FORWARD
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function is used for private message
#    to control what is done with a received
#    message.
#       # next message
#       * quit
#       1 play again
#       2 forward message
#       3 reply
#       4 delete and next message
# BUGS
# LOCAL-VARIABLES
#    The RESULT variable.
function message_command {
  while :
  do
     RESULT=`play_file_and_ask_dtmf $CONTROL_MESSAGE_FILE`
     case $RESULT in
        \#) RESULT=NEXT; break;;
	+) RESULT=ABORTED; break;;
	1)  RESULT=AGAIN; break;;
	2)  RESULT=FORWARD; break;;
	3)  RESULT=REPLY; break;;
	4)  RESULT=DELETE; break;;
        END|ABORTED) break;;
     esac
  done
  echo "$RESULT"
}

# SYNOPSIS
#    result = set_language()
# RESULT
#    will be in the form 
#       - a code
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function sets the new current language.
# BUGS
#    This does not handle the empty code case. See also do_login
# LOCAL-VARIABLES
#    The RESULT variable.
#    The SELECTED_LANGUAGE variable.
function set_language {
   RESULT=`ask_code $SELECT_LANGUAGE_FILE`
   if [ "$RESULT" = "" ]; then
      RESULT=+
   fi
   case $RESULT in
      [0-9]*) ;;
      +)      RESULT=ABORTED
              ;;
   esac
   echo "mvm: set_language: result $RESULT" >&2
   echo "$RESULT"
}

# SYNOPSIS
#    result = ask_code(banner)
# RESULT
#    will be in the form 
#       - either [0-9]*
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function plays the description, then
#    waits for a sequence of numbers terminated
#    by a # (which is not kept). It then returns
#    it. Aborting can be done with *, or by a
#    DTMF during the play. An empty
#    code is possible.
# LOCAL-VARIABLES
#    The ANSWER, ANSWER2 and RESULT variables.
# NOTES
#    - Now has support for one hot-key during message
#      play.
#    - And even multi-hotkeys.
function ask_code {
   RESULT=`play_file_and_dtmf $1 MULTI`
   echo "mvm: ask_code: multi $RESULT" >&2
   case $RESULT in
      *+*) RESULT=ABORTED;;
      *\#*) RESULT=`echo "$RESULT" | sed 's/#.*//'`
            echo "mvm: ask_code: corrected $RESULT" >&2;;
      *) case $RESULT in
	    +*) RESULT=ABORTED;;
	    \#*) RESULT=;;
	    READY|[0-9]*) if [ "$RESULT" = "READY" ]; then
				   RESULT=
				fi
				while :
				do
				   ANSWER=`ask_dtmf`
				   case $ANSWER in
				      END)     RESULT=END
					       break;;
				      ABORTED) RESULT=ABORTED
					       break;;
				      *)       if [ "$ANSWER" = "#" ]; then
						  break
					       fi
					       if [ "$ANSWER" = "+" ]; then
						  RESULT=ABORTED
						  break
					       fi
					       RESULT="${RESULT}${ANSWER}";;
				   esac
				done;;
	 esac;;
   esac
	
   echo "mvm: ask_code: result $RESULT" >&2
   echo "$RESULT"
}

# SYNOPSIS
#    result = play_file_and_dtmf(file)
# RESULT
#    will be in the form 
#       - either READY (normal case)
#       - or END
#       - or a DTMF [0-9#+] (hotkey)
# DESCRIPTION
#    This functions plays file, then returns.
#    If a DTMF has been played, it is returned
#    to the caller.
#    It sets END if something wrong happened.
# LOCAL-VARIABLES
#    The ANSWER and RESULT variables.
# NOTES
#    - Support hotkey.
#    - Changes * into +
#    - If argc is 2, can return multi-codes.
# BUGS
#    No way to say if the file was non existant.
#    It is perhaps understood as END anyway.
function play_file_and_dtmf {
   RESULT=READY

   if [ "$SELECTED_LANGUAGE" != "" ]; then
      ANSWER=$1.${SELECTED_LANGUAGE}
      if [ ! -e $ANSWER ]; then
         ANSWER=$1
      fi
   else
      ANSWER=$1
   fi

   send "PLAY $ANSWER"
   ANSWER=`receive`
   echo "mvm: play_file_and_dtmf($1): mode $#: after play: $ANSWER" >&2
   case $ANSWER in
      PLAYING) ANSWER=`receive`
               echo "mvm: play_file_and_dtmf: after playing: $ANSWER" >&2
               case $ANSWER in
                  READY)         ;;
                  *)             case $ANSWER in
                                    RECEIVED_DTMF) ANSWER=`receive`
						   if [ "$ANSWER" = "*" ]; then
						      ANSWER=+
						   fi
                                                   echo "mvm: play_file_and_dtmf: received $ANSWER" >&2
                                                   case $ANSWER in
                                                      +|\#|[0-9]) RESULT=$ANSWER;;
                                                      *)          RESULT=END;;
                                                   esac

                                                   if [ "$RESULT" != "END" ]; then
						      ANSWER=`get_all_remaining_dtmfs`
						      case $ANSWER in
                                                         READY) ;;
							 *+*|*\#*|*[0-9]*) if [ $# = 2 ]; then
										 RESULT=${RESULT}${ANSWER}
									      fi;;
							 *)                   RESULT=END;;
						      esac
                                                   fi
                                                   # This is normally READY.
                                                   ;;
                                    *)             RESULT=END;;
                                 esac
               esac
               ;;
      *)
               case $ANSWER in
                  RECEIVED_DTMF) ANSWER=`receive`
                                 echo "mvm: play_file_and_dtmf: other: $ANSWER" >&2
                                 case $ANSWER in
                                    +|\#|[0-9]) RESULT=$ANSWER;;
                                    *) RESULT=END;;
                                 esac;;
                  *)             RESULT=END;;
               esac

               # No need to STOP because we haven't started yet.

               ;;
   esac

   echo "mvm: play_file_and_dtmf: result $RESULT" >&2
   echo "$RESULT"
}

# SYNOPSIS
#    result = get_all_remaining_dtmfs()
# RESULT
#    will be in the form 
#       - either READY (normal case)
#       - or a suite of +\*[0-9]
#       - or END if error.
# DESCRIPTION
#    This functions eats all remaining DTMFs in the
#    stream in the form 'RECEIVED_DTMF' DTMF ... READY
# NOTES
#    - This can only be used when the caller knows that
#      the suite of DTMF will be stopped by a READY, e.g.
#      after a play interrupted by a DTMF.
# LOCAL-VARIABLES
#    The ANSWER and RESULT variables.
# BUGS
function get_all_remaining_dtmfs {
   RESULT=

   while :
   do
      ANSWER=`receive`
      echo "mvm: get_all_remaining_dtmfs: received L1 $ANSWER" >&2
      case $ANSWER in
         RECEIVED_DTMF) ANSWER=`receive`
                        echo "mvm: get_all_remaining_dtmfs: received L2 $ANSWER" >&2
                        if [ "$ANSWER" = "*" ]; then
                           ANSWER=+
                        fi
                        case $ANSWER in
                           +|\#|[0-9]) RESULT=${RESULT}${ANSWER};;
                           *) RESULT=END
                              break;;
                        esac;;
         READY) break;;
         *)     RESULT=END
                break;;
      esac
   done

   if [ "$RESULT" = "" ]; then
      RESULT=READY
   fi

   echo "mvm: get_all_remaining_dtmfs: result $RESULT" >&2
   echo "$RESULT"
}

# SYNOPSIS
#    result = play_file(file)
# RESULT
#    will be in the form 
#       - either READY (normal case)
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This functions plays file, then returns.
#    It sets ABORTED if a DTMF * has been played,
#    stops the message but returns READY if #
#    is used, any other key is used.
#    It sets END if something wrong happened.
# LOCAL-VARIABLES
#    The ANSWER and RESULT variables.
# BUGS
#    No way to say if the file was non existant.
#    It is perhaps understood as END anyway.
function play_file {
   RESULT=READY

   if [ "$SELECTED_LANGUAGE" != "" ]; then
      ANSWER=$1.${SELECTED_LANGUAGE}
      if [ ! -e "$ANSWER" ]; then
         ANSWER=$1
      fi
   else
      ANSWER=$1
   fi

   send "PLAY $ANSWER"
   ANSWER=`receive`
   echo "mvm: play_file($1): after play $ANSWER" >&2
   case $ANSWER in
      PLAYING) ANSWER=`receive`
               echo "mvm: play_file: after playing $ANSWER" >&2
               case $ANSWER in
                  READY)         ;;
                  *)             case $ANSWER in
                                    RECEIVED_DTMF) ANSWER=`receive`
                                                   if [ "$ANSWER" = "*" ]; then
                                                      ANSWER=+
                                                   fi
                                                   echo "mvm: play_file: key $ANSWER" >&2
                                                   case $ANSWER in
                                                      \#|+|[0-9]) RESULT=ABORTED;;
                                                      *)  RESULT=END;;
                                                   esac

                                                   if [ "$RESULT" != "END" ]; then
						      ANSWER=`get_all_remaining_dtmfs`
                                                      case $ANSWER in
                                                         READY) ;;
                                                         *+*|*\#*|*[0-9]*) ;;
                                                         *) RESULT=END;;
                                                      esac
                                                   fi
                                                   # This is normally READY.
                                                   ;;
                                    *)             RESULT=END;;
                                 esac;;

               esac
               ;;
      *)
               case $ANSWER in
                  RECEIVED_DTMF) ANSWER=`receive`
                                 echo "mvm: play_file: keying 2 $ANSWER" >&2
                                 RESULT=ABORTED;;
                  *)             RESULT=END;;
               esac

               # No need to STOP because we haven't started yet.

               ;;
   esac
   echo "mvm: play_file: result $RESULT" >&2
   echo "$RESULT"
}

# SYNOPSIS
#    result = area_exists(areaname)
# RESULT
#    will be in the form
#       - READY
#       - NONE 
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function checks if a designated area exists,
#    if the current user has the access permission, and
#    if yes returns READY. Else returns NONE. The
#    calling function must then update accordingly
#    the CURRENT_AREA variable.
# NOTES
#    - Area name is a number.
# LOCAL-VARIABLES
#    The RESULT variable.
function area_exists {
   echo "$RESULT"
}

# SYNOPSIS
#    result = area_select()
# RESULT
#    will be in the form
#       - NONE
#       - or ABORTED
#       - or END
#       - or [0-9]+ (area name)
# DESCRIPTION
#    This function proposes a list of areas, then calls
#    area_exists() and returns its result. The caller
#    should update CURRENT_AREA.
# NOTES
#    - Area name is a number.
# LOCAL-VARIABLES
#    The RESULT variable.
function area_select {
   echo "$RESULT"
}

# SYNOPSIS
#    result = area_read(areaname)
# RESULT
#    will be in the form
#       - READY 
#       - or END
# DESCRIPTION
#    This function allows to browse and read voice messages in
#    an area which may or may not be the current area. Read
#    level is enforced.
# TODO
#    - Next implementation, along with the general message editor,
#      could allow more flexibility.
# LOCAL-VARIABLES
#    The RESULT variable.
function area_read {
   echo "$RESULT"
}

# SYNOPSIS
#    result = area_write(areaname)
# RESULT
#    will be in the form
#       - READY 
#       - or END
# DESCRIPTION
#    This function allows to create voice messages in
#    an area which may or may not be the current area. Write
#    level is enforced.
# LOCAL-VARIABLES
#    The RESULT variable.
function area_write {
   echo "$RESULT"
}

# SYNOPSIS
#    result = area_edit(areaname)
# RESULT
#    will be in the form
#       - READY 
#       - or END
# DESCRIPTION
#    This function allows to administer completely or
#    partly the named area.
#    Write level and owner/operator level are enforced.
# LOCAL-VARIABLES
#    The RESULT variable.
function area_edit {
   echo "$RESULT"
}

# SYNOPSIS
#    result = area_create()
# RESULT
#    will be in the form
#       - READY 
#       - or END
# DESCRIPTION
#    This function asks for an area name, description,
#    and welcome message. It then creates it.
# LOCAL-VARIABLES
#    The RESULT variable.
function area_create {
   echo "$RESULT"
}

# SYNOPSIS
#    result = do_voice_mail(menudir)
# RESULT
#    will be in the form
#       - READY 
#       - goodbye
#       - or ABORTED
#       - or END
# DESCRIPTION
#    This function handles a submenu.
# LOCAL-VARIABLES
#    The RESULT variable.
# NOTES
#    - Now has some support for one hotkey while playing.
function do_voice_mail {
   CURRENT_MENU=$1

   echo "mvm: do_voice_mail($CURRENT_MENU)" >&2

   while :
   do
      while :
      do

         echo "mvm: do_voice_mail: user $CURRENT_USER level $USER_LEVEL" >&2

         if [ -f $CURRENT_MENU/direct_command_exec ]; then
            ANSWER=`cat $CURRENT_MENU/direct_command_exec`
	    echo "mvm: do_voice_mail: direct_command_exec $ANSWER" >&2
            set - $ANSWER
            ANSWER=$1
         else
	    if [ "$SELECTED_LANGUAGE" != "" ]; then
	       ANSWER=$CURRENT_MENU/welcome.${SELECTED_LANGUAGE}
	       if [ ! -e "$ANSWER" ]; then
		  ANSWER=$CURRENT_MENU/welcome
	       fi
	    else
	       ANSWER=$CURRENT_MENU/welcome
	    fi

	    # Play this menu welcome message.
	    if [ -e "$ANSWER" ]; then
	       ANSWER=`play_file_and_dtmf $ANSWER`

	    else
	       ANSWER=READY
	    fi

	    case $ANSWER in
	       END)     break;;
	       READY)   ANSWER=`ask_dtmf`;;
	       *)       ;;
	    esac

	    echo "mvm: do_voice_mail: DTMF $ANSWER" >&2

	    case $ANSWER in
	       ABORTED) ;;
	       END)     break;;
	       *)       # Analyze the command
			# DTMF command_name min_level args ... args
			if [ -f $CURRENT_MENU/commands ]; then
			   echo "cmd: $ANSWER" >> /tmp/mvm_commands
			   ANSWER=`grep "^$ANSWER " $CURRENT_MENU/commands`
			   if [ $? != 0 ]; then
			      continue
			   fi

			   echo "LOW: $ANSWER" >> /tmp/mvm_commands
			   set - $ANSWER
			   shift
			   # Aie, aie, no error check
			   # command_name min_level args ... args
			   ANSWER=$1
			   shift
			   # min_level args ... args
			   if [ $USER_LEVEL -lt $1 ]; then
			      # User level too low.
			      if [ "$CURRENT_USER" != "0" ]; then
				 echo "nok level $1 > $USER_LEVEL" >> /tmp/mvm_commands
				 continue
			      else
				 echo "root level" >> /tmp/mvm_commands
			      fi
			   else
			      echo "ok level $1 <= $USER_LEVEL" >> /tmp/mvm_commands
			   fi
			   shift # forget the required level
			   # remaining: args ... args
			else
			   # default menu
			   case $ANSWER in
			      0)  ANSWER=goodbye;;
			      1)  ANSWER=read_mail;;
			      2)  ANSWER=send_mail;;
			      3)  ANSWER=create_account;;
			      4)  ANSWER=cookie;;
			      5)  ANSWER=random_sample;;
			      9)  ANSWER=edit_menu;;
			      +) ANSWER=identify;;
			      \#) ANSWER=message_to_operator;;
			   esac
			fi;;
	    esac
         fi

         case $ANSWER in
            ABORTED) ;;
            END) break;;
            *)
                 echo "HIGH: command $ANSWER arguments $*" >> /tmp/mvm_commands
		 
                 case $ANSWER in                                 
                    goodbye)   # goodbye
                               break;;
                    play)      # plays a file
                               if [ $# = 1 ]; then
                                  ANSWER=`play_file $1`
                                  case $ANSWER in
                                     ABORT) ;;
                                     END)   break;;
                                  esac
                               fi;;
                    record_to) # record to a file
                               if [ $# = 1 ]; then
                                  ANSWER=`record_to_file $RECORD_TO_FILE $1`
                                  case $ANSWER in
                                     ABORT) ;;
                                     END)   break;;
                                  esac
                               fi;;
                    record_and_exec) # record to a file and execute
                               if [ $# = 1 ]; then
                                  ANSWER=`record_to_file $RECORD_TO_FILE $TMPFILE`
                                  case $ANSWER in
                                     ABORT) ;;
                                     END)   break;;
                                     *) $1 $TMPFILE; rm $TMPFILE;;
                                  esac
                               fi;;
                    param_cmd) # get a DTMF string and pass it to a
                               # script
                               if [ $# = 1 ]; then
                                  set - `cat $1`
                                  if [ $# = 4 ]; then
                                     ANSWER=`do_param_cmd $1 $2 $3 $4`
                                     case $ANSWER in
                                        ABORT) ;;
                                        GO_FAX) echo "mvm: account $CURRENT_USER level $USER_LEVEL now going fax reception and aborting voice script." >&2
                                                break;;
                                        END)   break;;
                                     esac
                                  else
                                     echo "mvm: account $CURRENT_USER level $USER_LEVEL not enough parameters in do_param_cmd description file." >&2
                                  fi
                               fi;;
		 
                    read_mail) # read your private mail
                               if [ "$CURRENT_USER" != "" ]; then
                                  ANSWER=`read_mail $CURRENT_USER`
                                  case $ANSWER in
                                     ABORTED) ;;
                                     END)     break;;
                                  esac
                               fi
                               ;;
                    send_mail) # send private mail
#                              if [ "$CURRENT_USER" != "" ]; then
                                  ANSWER=`dosend_mail`
                                  case $ANSWER in
                                     ABORTED) ;;
                                     END)     break;;
                                  esac
#                              fi
                               ;;
                    create_account)
                               # create an account
                               if [ "$CURRENT_USER" = "" ]; then
                                  ANSWER=`create_account`
                                  case $ANSWER in
                                     ABORTED) ;;
                                     END)     break;;
                                     [0-9]*)  CURRENT_USER=$ANSWER
                                              USER_LEVEL=1
                                              echo "mvm: account $CURRENT_USER level $USER_LEVEL created/logon." >&2
                                              ;;
                                  esac
                               fi
                               ;;
                    cookie)   # receive a cookie
                              IGNORE="`/usr/games/fortune -s`"
                              IGNORE=`english_text_to_audio_file "Here is a cookie for you. $IGNORE" $TMPFILE`
                              ANSWER=`play_file $TMPFILE`
                              rm -f $TMPFILE
                              case $ANSWER in
                                 ABORTED) ;;
                                 END)     break;;
                              esac
                              ;;
                    random_sample)
                              # select a random file from the library
                              # and play it
                              IGNORE=`random_file $LIBRARY_DIR`
                              TMPFILE=/tmp/gener_$$
                              (case $IGNORE in
                                 *.gz) gzip -d < $IGNORE;;
                                 *) cat $IGNORE;;
                              esac) | $PVF_TOOLS_DIR/autopvf | $PVF_CUT | $PVF_SPEED | $PVF_TOOLS_DIR/$PVF_MODEM_FILTER  > $TMPFILE
                              ANSWER=`play_file /tmp/gener_$$`
                              rm /tmp/gener_$$
                              case $ANSWER in
                                 ABORTED) ;;
                                 END)     break;;
                              esac
                              ;;
                    edit_menu)
                              # menu editing for the owner or the 0
                              # for the first version can be hardcoded,
                              # then it should be a normal menu with
                              # items (who control the access each time).
                              if [ "$CURRENT_USER" != "" ]; then
                                 if [ -f $CURRENT_MENU/.owner ]; then
                                    ANSWER=`cat $CURRENT_MENU/.owner`
                                 else
                                    ANSWER=0
                                 fi
                                 if [ "$ANSWER" = "$CURRENT_USER" ]; then
                                    ANSWER=`edit_menu $CURRENT_MENU`
                                    case $ANSWER in
                                       READY)   ;;
                                       ABORTED) ;;
                                       *)       ;;
                                    esac    
                                 fi
                              fi
                              ;;
                    edit_user)
                              # user editing for 0 only --- not implemented
                              if [ "$CURRENT_USER" = "0" ]; then
                                 ANSWER=`edit_user`
                                 case $ANSWER in
                                    END)     break;;
                                    ABORTED) ;;
                                    *)       ;;
                                 esac    
                              fi
                              ;;
                    message_to_operator)
                              # leave message to operator
                              if [ "$CURRENT_USER" = "" ]; then
                                 ANSWER=`send_mail 0 -1 NO`
                              else
                                 ANSWER=`send_mail 0 $CURRENT_USER NO`
                              fi
                              case $ANSWER in
                                 ABORT) ;;
                                 END)   break;;
                              esac
                              ;;
                    identify) # identify user
                              ANSWER=READY
                              if [ "$CURRENT_USER" != "" ]; then
                                 ANSWER=`play_file $ALREADY_LOGGED_FILE`
                              fi
                              case $ANSWER in
                                 ABORT) ;;
                                 READY) ANSWER=`do_login`
                                        case $ANSWER in
                                           [0-9]*)  if [ -f $MAILBOX_DIR/$ANSWER/.level ]; then
                                                       USER_LEVEL=`cat $MAILBOX_DIR/$ANSWER/.level`
                                                    else
                                                       USER_LEVEL=1 # identified.
                                                    fi
                                                    CURRENT_USER=$ANSWER
                                                    echo "mvm: $CURRENT_USER/$USER_LEVEL logged on" >&2
                                                    ANSWER=`read_mail $CURRENT_USER`
                                                    case $ANSWER in
                                                       ABORTED) ;;
                                                       END)     break;;
                                                    esac;;
                                           NONE)    ;;
                                           ABORTED) ;;
                                           END)     break;;
                                        esac;;
                                 *)     break;;
                              esac
                              ;;
                    back_menu) break;;
                    goto_menu) if [ $# = 1 ]; then
                                  ANSWER=`do_voice_mail $1`
                                  case $ANSWER in
                                     END|goodbye|goto_main|GO_FAX) break;;
                                  esac
                               fi;;
                    goto_main) ANSWER=goto_main
                               break;;
                    goto_area) # Goto to a specified area
                               if [ $# = 1 ]; then
                                  if [ "$CURRENT_USER" != "" ]; then
                                     ANSWER=`area_exists $1`
                                     case $ANSWER in
                                        READY) CURRENT_AREA=$1;;
                                        NONE)  ANSWER=READY;;
                                        END)   break;;
                                     esac
                                  fi
                               fi
                               ;;
                    edit_area)   # Edit the current area
                                 if [ "$CURRENT_USER" != "" ]; then
                                    ANSWER=`area_edit $CURRENT_AREA`
                                    case $ANSWER in
                                       NONE)  ANSWER=READY;;
                                       END)   break;;
                                    esac
                                 fi;;
                    create_area)  # Create an area
                                 if [ "$CURRENT_USER" != "" ]; then
                                    ANSWER=`area_create`
                                    case $ANSWER in
                                       [0-9]*) CURRENT_AREA=$ANSWER;;
                                       NONE)  ANSWER=READY;;
                                       END)   break;;
                                    esac
                                 fi;;
                    select_area) # Select an area from the list
                                 if [ "$CURRENT_USER" != "" ]; then
                                    ANSWER=`area_select`
                                    case $ANSWER in
                                       [0-9]*) CURRENT_AREA=$ANSWER;;
                                       NONE)  ANSWER=READY;;
                                       END)   break;;
                                    esac
                                 fi
                                 ;;
                    read_public_messages) # Read the public messages
                              if [ "$CURRENT_AREA" != "" ]; then
                                 if [ "$CURRENT_USER" != "" ]; then
                                    ANSWER=`area_read $CURRENT_AREA`
                                    case $ANSWER in
                                       END)   break;;
                                    esac
                                 fi
                              fi
                                 ;;
                    record_public_message) # Record a public message
                              if [ "$CURRENT_AREA" != "" ]; then
                                 if [ "$CURRENT_USER" != "" ]; then
                                    ANSWER=`area_write $CURRENT_AREA`
                                    case $ANSWER in
                                       END)   break;;
                                    esac
                                 fi
                              fi
                                           ;;
                    message_to_user) if [ $# = 1 ]; then
                                        # leave message to specified user
                                        if [ "$CURRENT_USER" = "" ]; then
                                           ANSWER=`send_mail $1 -1 NO`
                                        else
                                           ANSWER=`send_mail $1 $CURRENT_USER NO`
                                        fi
                                        case $ANSWER in
                                           ABORT) ;;
                                           END)   break;;
                                        esac
                                     fi;;
                    *) IGNORE=`english_text_to_audio_file "Your sensitive mind just found an anomaly in the fabric of blue space. Unknown command was $ANSWER with arguments $*" $TMPFILE`
		 
                       echo "mvm: ANOMALY $ANSWER/$*" >&2
                       ANSWER=`play_file $TMPFILE`
                       rm -f $TMPFILE
                       case $ANSWER in
                          ABORTED) ;;
                          END)     break;;
                       esac
                       ;;
                 esac;;
	 esac

         if [ -f $CURRENT_MENU/direct_command_exec ]; then
            if [ "$ANSWER" = "READY" ]; then
               ANSWER=goodbye
            fi
            break
         fi
      done
      case $ANSWER in
         END|goodbye|GO_FAX) break;;
         goto_main)   break;;
         backmenu)    ANSWER=READY
                      break;;
      esac
   done
   echo "$ANSWER"
}

# Redirect stderr (idea of gert@greenie.muc.de (Gert Doering))
date >> /tmp/mvm-errorlog
exec 2>> /tmp/mvm-errorlog

#
# Let's see if the voice library is talking to us
#

ANSWER=`receive`
if [ "$ANSWER" != "HELLO SHELL" ]; then
     echo "$0: voice library not answering" >&2
     exit 1
fi

#
# Let's answer the message
#

send "HELLO VOICE PROGRAM"

#
# Let's see if it worked
#

ANSWER=`receive`
if [ "$ANSWER" != "READY" ]; then
     echo "$0: initialization failed" >&2
     exit 1
fi


#
# Enable events
#

send "ENABLE EVENTS"
ANSWER=`receive`
if [ "$ANSWER" != "READY" ]; then
     echo "$0: could not enable events" >&2
     exit 1
fi

# FROM NOW ON, TRY TO HANDLE ALL POSSIBLE CASES CLEVERLY

if [ "$VOICE_MODE" = "normal" ]; then
# Let's wait for two seconds

send "WAIT 2"
ANSWER=`receive`
# BUG: what if we receive the DTMF event here and after the WAITING ?
if [ "$ANSWER" != "WAITING" ]; then
     echo "$0: could not start waiting" >&2
fi

while :
do
     ANSWER=`receive`

     case $ANSWER in
        DATA_CALLING_TONE|FAX_CALLING_TONE) MODEM=true
                                            break;;
        RECEIVED_DTMF) MODEM=false
                       break;;
        0|READY) MODEM=true # Timeout
               break;;
        # BUGS: if we receive a hangup event, we will ignore it and
        # timeout. The ATA will then fail and recover after some time.
     esac
done

if [ "$MODEM" = "true" ]; then
   send "STOP"
   ANSWER=`receive`
   # BUG: what if we receive a DTMF here ?
   if [ "$ANSWER" != "READY" ]; then
     echo "$0: could not stop waiting" >&2
   fi

   send "GOODBYE" # ENTER_DATA_FAX_MODE

   ANSWER=`receive`

   if [ "$ANSWER" != "GOODBYE SHELL" ]; then
     echo "$0: could not say goodbye to the voice library" >&2
   fi

   exit 3 # special value
else
   # Get the received DTMF

   ANSWER=`receive`

   send "STOP"
   ANSWER=`receive`
   # BUG: what if we receive another DTMF ?
   if [ "$ANSWER" != "READY" ]; then
     echo "$0: could not stop waiting" >&2
   fi
fi

fi

#send "DEVICE DIALUP_LINE"
send "AUTOSTOP ON"
ANSWER=`receive`
# no check =

echo "mvm: sign on (set language)" >&2

if [ "$SELECTED_LANGUAGE" = "NO_ASK" ]; then
   ANSWER=ABORTED
else
   ANSWER=`set_language`
fi
case $ANSWER in
 [0-9]*|ABORTED) case $ANSWER in
                    [0-9]*) SELECTED_LANGUAGE=$ANSWER
                            if [ "$SELECTED_LANGUAGE" = "0" ]; then
                               SELECTED_LANGUAGE=
                            fi;;
                 esac
                 
  case `play_file $INTRODUCTION_FILE` in
   READY|ABORTED)

   # The voicemail thing

   echo "mvm: language set to $SELECTED_LANGUAGE" >&2

   while :
   do
      ANSWER=`do_voice_mail $MAIN_MENU_DIR`
      case $ANSWER in
         READY|goto_main) ;;  # falling back_menu
         *)     break;;
      esac
   done

   case $ANSWER in
      END)       # This doesn't really work if the protocol between us
                 # and vgetty is not anymore synchronized.
                 IGNORE=`play_file $CRASH_FILE`;;
      GO_FAX)    ;;
      *)         IGNORE=`play_file $GOODBYE_FILE`;;
   esac;;
 esac;;
esac

echo "mvm: finishing" >&2

# Let's say goodbye

send "GOODBYE"
IGNORE=`receive`

if [ "$ANSWER" = "GO_FAX" ]; then
   echo "mvm: GO_FAX" >&2
   exit 3
else
   exit 0
fi



