/* ext2.h
 * David Lutz
 *
 * This file contains important file system information like
 * the superblock, inode and group structures and a few global
 * variables.
 */

#ifndef _EXT2_H_
#define _EXT2_H_

/* This is screwy because under Linux (at least on Jason't system) 
 * sys/stat.h calls sys/types.h which calls Linux/types.h which is 
 * asm/types.h which already has __xx.  Get that?  So I made it _xx here
 * to avoid conflicts.
 * --Jason
 */

typedef signed char _s8;
typedef unsigned char _u8;
typedef unsigned char byte;

typedef signed short _s16;
typedef unsigned short _u16;

typedef signed long _s32;
typedef unsigned long _u32;

#define EXT2_SUPER_MAGIC 0xEF53
#define EXT2_MIN_BLOCK  1024
#define EXT2_MIN_FRAG   1024

#define	EXT2_NDIR_BLOCKS 12
#define	EXT2_IND_BLOCK  EXT2_NDIR_BLOCKS
#define	EXT2_DIND_BLOCK	(EXT2_IND_BLOCK + 1)
#define	EXT2_TIND_BLOCK	(EXT2_DIND_BLOCK + 1)
#define	EXT2_N_BLOCKS	(EXT2_TIND_BLOCK + 1)


#undef S_IFMT
#undef S_IFSOCK
#undef S_IFLNK
#undef S_IFFIL
#undef S_IFBLK
#undef S_IFDIR
#undef S_IFCHR
#undef S_IFIFO

#undef S_ISSOCK
#undef S_ISLNK
#undef S_ISFIL
#undef S_ISBLK
#undef S_ISDIR
#undef S_ISCHR
#undef S_ISFIFO

#define S_IFMT   0x0F000					/*017 0000 */

#define S_IFSOCK 0x0C000					/*014 0000 */
#define S_IFLNK  0x0A000					/*012 0000 */
#define S_IFFIL  0x08000					/*010 0000 */
#define S_IFBLK  0x06000					/*006 0000 */
#define S_IFDIR  0x04000					/*004 0000 */
#define S_IFCHR  0x02000					/*002 0000 */
#define S_IFIFO  0x01000					/*001 0000 */

#define S_ISSOCK(m)     (((m) & S_IFMT) == S_IFSOCK)
#define S_ISLNK(m)      (((m) & S_IFMT) == S_IFLNK)
#define S_ISFIL(m)      (((m) & S_IFMT) == S_IFFIL)
#define S_ISBLK(m)      (((m) & S_IFMT) == S_IFBLK)
#define S_ISDIR(m)      (((m) & S_IFMT) == S_IFDIR)
#define S_ISCHR(m)      (((m) & S_IFMT) == S_IFCHR)
#define S_ISFIFO(m)     (((m) & S_IFMT) == S_IFIFO)

/*not yet supported: */
#define S_IFUID  0x00800					/*000 4000 */
#define S_IFGID  0x00400					/*000 2000 */
#define S_IFVTX  0x00200					/*000 1000 */

/*
 * Super Block Structure
 */
typedef struct
{
    _u32 s_inodes_count;					/* Inodes count */
    _u32 s_blocks_count;					/* Blocks count */
    _u32 s_r_blocks_count;					/* Reserved blocks count */
    _u32 s_free_blocks_count;					/* Free blocks count */
    _u32 s_free_inodes_count;					/* Free inodes count */
    _u32 s_first_data_block;					/* First Data Block */
    _u32 s_log_block_size;					/* Block size */
    _s32 s_log_frag_size;					/* Fragment size */
    _u32 s_blocks_per_group;					/* # Blocks per group */
    _u32 s_frags_per_group;					/* # Fragments per group */
    _u32 s_inodes_per_group;					/* # Inodes per group */
    _u32 s_mtime;						/* Mount time */
    _u32 s_wtime;						/* Write time */
    _u16 s_mnt_count;						/* Mount count */
    _s16 s_max_mnt_count;					/* Maximal mount count */
    _u16 s_magic;						/* Magic signature */
    _u16 s_state;						/* File system state */
    _u16 s_errors;						/* Behaviour when detecting errors */
    _u16 s_pad;
    _u32 s_lastcheck;						/* time of last check */
    _u32 s_checkinterval;					/* max. time between checks */
    _u32 s_creator_os;						/* OS */
    _u32 s_rev_level;						/* Revision level */
    _u16 s_def_resuid;						/* Default uid for reserved blocks */
    _u16 s_def_resgid;						/* Default gid for reserved blocks */
    _u32 s_reserved[235];					/* Padding to the end of the block */
}
super_block;

/*
 * Group Descriptor Structure
 */
typedef struct
{
    _u32 bg_block_bitmap;					/* Blocks bitmap block */
    _u32 bg_inode_bitmap;					/* Inodes bitmap block */
    _u32 bg_inode_table;					/* Inodes table block */
    _u16 bg_free_blocks_count;					/* Free blocks count */
    _u16 bg_free_inodes_count;					/* Free inodes count */
    _u16 bg_used_dirs_count;					/* Directories count */
    _u16 bg_pad;
    _u32 bg_reserved[3];
}
group_desc;

/*
 * Inode Structure
 */
typedef struct
{
    _u16 i_mode;						/* File mode */
    _u16 i_uid;							/* Owner Uid */
    _u32 i_size;						/* Size in bytes */
    _u32 i_atime;						/* Access time */
    _u32 i_ctime;						/* Creation time */
    _u32 i_mtime;						/* Modification time */
    _u32 i_dtime;						/* Deletion Time */
    _u16 i_gid;							/* Group Id */
    _u16 i_links_count;						/* Links count */
    _u32 i_blocks;						/* Blocks count */
    _u32 i_flags;						/* File flags */
    _u32 i_reserved1;						/* Reserved 1 */
    _u32 i_block[15];						/* Pointers to blocks */
    _u32 i_version;						/* File version (for NFS) */
    _u32 i_file_acl;						/* File ACL */
    _u32 i_dir_acl;						/* Directory ACL */
    _u32 i_faddr;						/* Fragment address */
    _u8 i_frag;							/* Fragment number */
    _u8 i_fsize;						/* Fragment size */
    _u32 i_reserved2[2];					/* Reserved 2 */
}
inode;

/*
 * Directory Structure
 */
struct dir
{
    _u32 inode_num;
    _u16 rec_len;
    _u16 name_len;
    char name[256];						/* between 0 and 256 chars */
};

/*
 * Partition Table Entry
 */
typedef struct
{
    _u8 bootIndicator;
    _u8 startHead;
    _u8 startSector;
    _u8 startCylinder;
    _u8 system;
    _u8 endHead;
    _u8 endSector;
    _u8 endCylinder;
    _u32 leadSectors;
    _u32 numSectors;
}
partitionTableEntry;

#endif								/* _EXT2_H_ */
