/* fuji_mb86606.c -- Fujitsu MB86606 Linux driver implementation
 * Marc SCHAEFER <schaefer@dilog.ch>
 * Creation: 28/09/97
 * Update:   11/12/97
 * V1.0 PV010
 * DESCRIPTION
 *    This module implements a very simple (stupid)
 *    Fujitsu MB86606 Linux kernel SCSI module.
 * NOTES
 *    - Seems that Linux only sends 1024 bytes (fs size) commands
 *      if scatter/gather is not implemented. At least with 2.0.30.
 *    NEGOCIATION
 *       - Synchronous parameter negociation is done as follows:
 *            if we initiate:
 *               step 1
 *                  - We send a SDTR message to the target (this is the
 *                    maximum we can *receive*). We set the IN_NEGOCIATION
 *                    to show we have initiated.
 *               step 2
 *                  - If !IN_NEGOCIATION goto step 3
 *                  - The target replies. If the value are acceptable, we
 *                    set them. If the values are the same we had configured
 *                    before, we do not reply. We clear IN_NEGOCIATION and
 *                    we set DID_NEGOCIATE.
 *                  - If the values are not acceptable, we send the most
 *                    we can achieve (this should not happen)
 *            if the target initiates:
 *               step 3
 *                  - The target sends the message. If the values are
 *                    acceptable, we set them, and reply the same, and set
 *                    DID_NEGOCIATE (IN_NEGOCIATION is not set if here)
 *                    If those are unacceptable, we set the most we can
 *                    achieve, and we set IN_NEGOCIATION, and we send our
 *                    values.
 * BUGS
 *    - /sbin/mke2fs had a few hangs.
 *    - If !FUJI_ENABLE_SYNC, does not send the command. Bizarre, since
 *      it does send when negociating is done.
 *    - Ultra and 10 MHz is ok until big read/writes (see next).
 *      Async is always ok, except crashes Linux after some time.
 *    - It seems that in a write (and sometimes in read?) of big
 *      sizes, with e.g. xfer=64k, scatter=1024 bytes (check), we do not
 *      have REQ_ASS/SEND_CMD/TERMINATED loop but sometimes REQ_ASS
 *      before TERMINATED. Only happens in sync mode. If we ignore that
 *      REQ_ASS interrupt if in DATA_S, TERMINATE comes normally, and
 *      then REQ_ASS as expected. This is the current work-around, just
 *      asked Fujitsu about it.
 *    - We should tell the upper layer when no status came, and when
 *      data was short, etc.
 *    - Sometime it is not very clear if we shall call scsi_done() especially
 *      when no queue_command() happened (e.g. abort)
 *    - Linux seems to hang after in queuecommand() on rmmod/insmod again
 *      if init succeeds. Seems no longer at working-7.
 *    - many error checking, recovery/messaging/etc. left out.
 *    - What happens in case of PCI parity error ?
 *    - I do not like the types used for variables. Missing typedefs, etc.
 *    - dd if=/dev/zero of=/dev/sda bs=512 count=10000 gives a GPF under
 *      2.0.27. Also reading.
 *    - No error handling for PCI side.
 * TODO
 *    - Private definitions to fuji_mb86606_private.h
 *    - Merge some defines with pci.h defines.
 *    - Check negociation is not infinite.
 *    - Ensure that all states are reachable: do not separate
 *      msgin when select/not cmd.
 *    - Use only commands which can be used in the user program.
 *    - Send SDTR at end of WDTR. 
 *    - If reset called again while set, do a software reset twice and
 *      reconfig too. Or only when flags say so.
 * BASED-ON
 *    Took some structuration ideas, debugging and PCI detection from
 *       qlogicisp.c of Linux 2.0.30
 * MODIFICATION-HISTORY
 *    11/12/97  schaefer  A few bugfixes
 * $Id: fuji_mb86606.c,v 1.70 1997/12/11 15:40:32 schaefer Exp $
 */

#include <linux/blk.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/bios32.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <linux/unistd.h>
#include <asm/io.h>
#include <asm/irq.h>

#if LINUX_VERSION_CODE < 0x20100
#define ioremap vremap
#define iounmap vfree
#endif

#define FUJI_ENABLE_PROC_SUPPORT /* /proc */
#ifdef FUJI_ENABLE_PROC_SUPPORT
#define PROC_SCSI_FUJI_MB86606 PROC_SCSI_AHA1542 /* temporary */
#endif

#include "sd.h"
#include "hosts.h"
#include "fuji_mb86606.h"
#include "fuji_mb86606_private.h"

static char vcid[] = "$Id: fuji_mb86606.c,v 1.70 1997/12/11 15:40:32 schaefer Exp $";

/* Configuration section */

#define DEBUG_FUJI_MB86606       0
#define DEBUG_FUJI_MB86606_INTR  0
#define DEBUG_FUJI_MB86606_INTR_DATA 0
#define DEBUG_FUJI_MB86606_SETUP 1
#define TRACE_FUJI_MB86606       0

/* End Configuration section */

#include <linux/module.h>

/* Type and macro definitions */

typedef signed char BOOL;

#if TRACE_FUJI_MB86606

#define TRACE_BUF_LEN  (32*1024)

struct {
   u_long next;
   struct {
      u_long time;
      u_int index;
      u_int addr;
      u_char * name;
   } buf[TRACE_BUF_LEN];
} trace;

#define TRACE(w, i, a)                                          \
{                                                               \
        unsigned long flags;                                    \
                                                                \
        save_flags(flags);                                      \
        cli();                                                  \
        trace.buf[trace.next].name  = (w);                      \
        trace.buf[trace.next].time  = jiffies;                  \
        trace.buf[trace.next].index = (i);                      \
        trace.buf[trace.next].addr  = (long) (a);               \
        trace.next = (trace.next + 1) & (TRACE_BUF_LEN - 1);    \
        restore_flags(flags);                                   \
}

#else
#define TRACE(w, i, a)
#endif

#if DEBUG_FUJI_MB86606
#define ENTER(x)        printk("fuji_mb86606: entering %s()\n", x);
#define LEAVE(x)        printk("fuji_mb86606: leaving %s()\n", x);
#define DEBUG(x)        x
#else
#define ENTER(x)
#define LEAVE(x)
#define DEBUG(x)
#endif /* DEBUG_FUJI_MB86606 */

#if DEBUG_FUJI_MB86606_INTR
#define ENTER_INTR(x)   printk("fuji_mb86606: entering %s()\n", x);
#define LEAVE_INTR(x)   printk("fuji_mb86606: leaving %s()\n", x);
#define DEBUG_INTR(x)   x
#else
#define ENTER_INTR(x)
#define LEAVE_INTR(x)
#define DEBUG_INTR(x)
#endif /* DEBUG FUJI_MB86606_INTR */

#if DEBUG_FUJI_MB86606_INTR_DATA
#define DEBUG_INTR_DATA(x)   x
#else
#define DEBUG_INTR_DATA(x)
#endif

/* PCI Interrupt register
 *    - All interrupt types allowed.
 *    - Terminator enabled.
 *    - LED disabled.
 */
#define LED_OFF(host) \
   WRITE_TO_REGISTER(host, SPC_REG_PCI_INT_ENABLE, \
                     SPC_REG_PCI_INT_ENABLE_DTTPE \
                     | SPC_REG_PCI_INT_ENABLE_SSE \
                     | SPC_REG_PCI_INT_ENABLE_RMA \
                     | SPC_REG_PCI_INT_ENABLE_RTA \
                     | SPC_REG_PCI_INT_ENABLE_STA \
                     | SPC_REG_PCI_INT_ENABLE_DAP \
                     | SPC_REG_PCI_INT_ENABLE_PO0 /* LED DIS */);

#define LED_ON(host) \
   WRITE_TO_REGISTER(host, SPC_REG_PCI_INT_ENABLE, \
                     SPC_REG_PCI_INT_ENABLE_DTTPE \
                     | SPC_REG_PCI_INT_ENABLE_SSE \
                     | SPC_REG_PCI_INT_ENABLE_RMA \
                     | SPC_REG_PCI_INT_ENABLE_RTA \
                     | SPC_REG_PCI_INT_ENABLE_STA \
                     | SPC_REG_PCI_INT_ENABLE_DAP);

struct dataseg {
   void *d_base;
   u_long d_count;
};

/* NOTES
 *    - This structure is currently used in the driver. When
 *      user program is implemented, a simplified version should
 *      be made available to the user program data space, and
 *      commands should be fully handled by that program.
 */
struct Command_Entry {
   Scsi_Cmnd *handle;
   u_char target_lun;
   u_char target_id;
   u_short cdb_length;
   u_short segment_cnt;
   u_short current_segment;
   u_short flags;
   u_char cdb[FUJI_MB86606_SG_MAX_CDB_SIZE];
   struct dataseg dataseg[FUJI_MB86606_SG_TABLE_SIZE + 1];
   u_char status; /* SCSI STATUS, if _DONE_STATUS */
};

#define COMMAND_ENTRY_FLAGS_NONE                0
#define COMMAND_ENTRY_FLAGS_VALID          0x8000 /* If set, there is a
                                                   * command in this slot.
                                                   */

#define COMMAND_ENTRY_FLAGS_DONE_SELECT_S  0x0400 /* Selection is sent */
#define COMMAND_ENTRY_FLAGS_DONE_SELECTED  0x0200 /* Selection is complete */
#define COMMAND_ENTRY_FLAGS_DONE_DATA_S    0x0100 /* In data xfer */
#define COMMAND_ENTRY_FLAGS_DONE_DATA      0x0080 /* Data xfer done */
#define COMMAND_ENTRY_FLAGS_DONE_STATUS_S  0x0040 /* Done status cmd send */
#define COMMAND_ENTRY_FLAGS_DONE_STATUS    0x0020 /* Done status */
#define COMMAND_ENTRY_FLAGS_DONE_MOUT_S    0x0010 /* Done msg out cmd send */
#define COMMAND_ENTRY_FLAGS_DONE_MIN_S     0x0008 /* Done msg in cmd send,
                                                   * next intr is a message.
                                                   */
#define COMMAND_ENTRY_FLAGS_DONE_FINAL_M   0x0004 /* Final msg complete,
                                                   * expecting DISCONNECT.
                                                   */
#define COMMAND_ENTRY_FLAGS_DONE_CMD_S     0x0002 /* Cmd being sent */
#define COMMAND_ENTRY_FLAGS_DONE_CMD       0x0001 /* Done sending cmd */
#define COMMAND_ENTRY_FLAGS_MASK_DONE      0x07FF

/*
 * Device Flags:
 *
 * Bit  Name
 * ---------
 *  7   Disconnect Privilege
 *  6   Parity Checking
 *  5   Wide Data Transfers
 *  4   Synchronous Data Transfers
 *  3   Tagged Queuing
 *  2   Automatic Request Sense
 *  1   Stop Queue on Check Condition
 *  0   Renegotiate on Error
 */

struct target_info {
   u_short flags;
};

#define TARGET_INFO_FLAGS_DID_WIDE_NEGOCIATION 1 /* A wide negociation has
                                                  * taken place. Results are
                                                  * stored in the chip.
                                                  * NOTES
                                                  *    - When you set this,
                                                  *      reset sync neg to
                                                  *      async, and reset
                                                  *      the three other bits.
                                                  */
#define TARGET_INFO_FLAGS_DID_SYNC_NEGOCIATION 2 /* A sync negociation has
                                                  * taken place. Results are
                                                  * stored in the chip.
                                                  * NOTES
                                                  *    - When you set this,
                                                  *      reset in sync bit.
                                                  */

#define TARGET_INFO_FLAGS_IN_WIDE_NEGOCIATION  4 /* A wide negociation is
                                                  * taking place. All values
                                                  * have been reset. If we
                                                  * receive a WDTR message,
                                                  * acknowledging our values,
                                                  * we set it.
                                                  * NOTES
                                                  *    - Setting this bit
                                                  *      sets wide/sync to
                                                  *      zero.
                                                  */
                               
#define TARGET_INFO_FLAGS_IN_SYNC_NEGOCIATION  8 /* A sync negociation is
                                                  * taking place. When we set
                                                  * that bit we reset DID_SYNC,
                                                  * and we await the answer
                                                  * from the target to commit
                                                  * the values.
                                                  */

#if 0
#define FUJI_MAX_SYNC_OFFSET 0x0 /* ASYNC */
#define FUJI_MIN_SYNC_PERIOD 0x32
#else
#define FUJI_MAX_SYNC_OFFSET 0x8 /* Upto 0xFE is possible. */
#define FUJI_MIN_SYNC_PERIOD 0x0c /* Ultra SCSI */
#endif
#define FUJI_ENABLE_WIDE
#define FUJI_ENABLE_SYNC

#define FUJI_MAX_MESSAGE_LENGTH 6 /* Identify + SDTR */

/* NOTES
 *    - If we receive a WDTR or SDTR message while not in the _IN_, we
 *      *do* answer the target EVEN if the values are the same.
 *    - Messaging retry is not needed for parity errors (see spec).
 */

#define MAX_TARGETS 16
#define QUEUE_LENGTH (FUJI_MB86606_REQ_QUEUE_LEN)

struct fuji_mb86606_hostdata {
        u_char bus;
        u_char revision;
        u_char device_fn;
        struct target_info targets[MAX_TARGETS];

        char message_buffer[FUJI_MAX_MESSAGE_LENGTH];
        u_char message_count;

        u_short flags;
        u_long active_commands;

        struct Command_Entry commands[QUEUE_LENGTH];
        struct Command_Entry *busy_with; /* The command in processing
                                          * if flags & HOST_DATA_FLAGS_BUSY.
                                          */
        Scsi_Cmnd *reset_cmd;            /* No longer needed when Scsi_Cmnd
                                          * has a pointer to Cmd.
                                          */
};

#define HOST_DATA_FLAGS_BUSY 0x8000 /* Waiting for an interrupt, not idle */
#define HOST_DATA_FLAGS_RSET 0x4000 /* Waiting for RESET */

/* Global variables */
struct Scsi_Host *irq2host[NR_IRQS];
static char info_buf[100]; /* long enough. Bad if // init. buggy. sorry. */

/* The User Program */
#define FUJI_MAX_USER_PROGRAM_LENGTH 1023 /* not really */
#define FUJI_USER_PROGRAM_OFFSET 0x800
#define FUJI_MEMORY_SIZE 4096
static u_char user_program[] /* Must be a multiple of 4 bytes */
   = { 0x09, 0x19, 0x30, 0xD1,
       0x10, 0x18, 0x31, 0xD1,
       0x10, 0x1B, 0x40, 0xD1,
       0x10, 0x91, 0xE0, 0x00,
       0x0B, 0xC6, 0x8F, 0xD1,
       0x20, 0x1A, 0x50, 0x0D,
       0x91, 0xE2, 0xC6, 0x8B,
       0xD1, 0x28, 0x1C, 0x70,
       0x91, 0xE3, 0x91, 0xE4
       /* Address 0x30 to 0x70 are used to store data */
     };

#ifdef FUJI_ENABLE_PROC_SUPPORT
static struct proc_dir_entry proc_scsi_fuji_mb86606 = {
        PROC_SCSI_FUJI_MB86606, 7, "fuji_mb86606",
        S_IFDIR | S_IRUGO | S_IXUGO, 2
};
#endif

/* Private function definitions */

/* NAME
 *    fuji_mb86606_set_user_program
 * DESCRIPTION
 *    Downloads the user program to the SPC.
 * RESULT
 *    0 if everything ok, else error.
 * NOTES
 * BUGS
 * TODO
 */
static int fuji_mb86606_set_user_program(struct Scsi_Host *host,
                                         u_char *user_program,
                                         u_int user_program_length);

/* NAME
 *    fuji_mb86606_enable_irqs
 * DESCRIPTION
 *    Enables the IRQ line from the SPC.
 * RESULT
 *    NONE
 * NOTES
 * BUGS
 * TODO
 */
static void fuji_mb86606_enable_irqs(struct Scsi_Host *);

/* NAME
 *    fuji_mb86606_disable_irqs
 * DESCRIPTION
 *    Disables the IRQ line from the SPC.
 * RESULT
 *    NONE
 * NOTES
 * BUGS
 * TODO
 */
static void fuji_mb86606_disable_irqs(struct Scsi_Host *host);

/* NAME
 *    fuji_mb86606_init
 * DESCRIPTION
 *    General PCI detection and initialization of
 *    the Fujistsu MB86606 board.
 * RESULT
 *    0 if ok, else error.
 * NOTES
 * BUGS
 * TODO
 */
static int fuji_mb86606_init(struct Scsi_Host *host);

/* NAME
 *    fuji_mb86606_reset_hardware
 * DESCRIPTION
 *    Initializes the Fujitsu MB86606 hardware.
 * RESULT
 *    0 if ok, else error.
 * NOTES
 * BUGS
 * TODO
 */
static int fuji_mb86606_reset_hardware(struct Scsi_Host *host);

/* NAME
 *    fuji_mb86606_intr_handler
 * DESCRIPTION
 *    Interrupt handler for all PCI boards.
 * RESULT
 *    NONE
 * NOTES
 * BUGS
 * TODO
 */
static void fuji_mb86606_intr_handler(int irq,
                                      void *dev_id,
                                      struct pt_regs *regs);

/* NAME
 *    fuji_mb86606_state_to_result
 * DESCRIPTION
 *    Converts SPC status into SCSI status in
 *    case of error.
 * RESULT
 *    scsi result.
 * NOTES
 * BUGS
 * TODO
 */
static u_long fuji_mb86606_state_to_result(struct Command_Entry *cmd,
	        			   u_char isr_reg,
					   u_char step_reg,
					   u_char status_reg);

/* NAME
 *    fuji_mb86606_had_reset
 * DESCRIPTION
 *    Ensure that all parameters and all flags are set so that
 *    WDTR/SDTR conventions are zeroed.
 * RESULT
 *    NONE
 * NOTES
 * BUGS
 * TODO
 */
static void fuji_mb86606_had_reset(struct Scsi_Host *host);

/* NAME
 *    fuji_configure_param
 * DESCRIPTION
 *    This function configures the SCSI parameters for
 *    the named chip and the named target. If an acknowledge
 *    is needed (because we did not initiate the negociation,
 *    or because the values have been changed by the target
 *    and thus we need to confirm), or if the proposed values
 *    are inacceptable (case of target initiated negociation).
 * RESULT
 *    TRUE if no acknowledgment nor correction needed.
 * NOTES
 *    - If we did not re-send an acknowledge message when the
 *      values have changed, this would mean the target can
 *      send data to us faster than from us to target, and this
 *      is not efficient nor useful.
 *    - Some of the arguments can be NULL, then they are ignored.
 *      Note that either wide or (offset, period) must be set.
 * BUGS
 * TODO
 * SEE-ALSO
 *    - Comments about negociation in the beginning of this file.
 */
static BOOL fuji_configure_param(u_char *wide,
				 u_char *period,
				 u_char *offset,
				 struct Scsi_Host *host,
                                 struct fuji_mb86606_hostdata *hostdata,
				 u_char target_id);

/* NAME
 *    internal_done
 * DESCRIPTION
 *    This function is called by queue_command() on behalf of command()
 *    for command() implementation faking. This is just for older kernels.
 * RESULT
 *    NONE
 * NOTES
 * BUGS
 * TODO
 */
static void internal_done(Scsi_Cmnd *Cmnd);

#if DEBUG_FUJI_MB86606
static void fuji_mb86606_print_scsi_cmd(Scsi_Cmnd *Cmnd);
#endif

/* Private inline functions implementation */
static inline void fuji_mb86606_enable_irqs(struct Scsi_Host *host) {
   /* BUGS
    *    - Probably inefficient.
    */
   WRITE_TO_REGISTER(host, SPC_REG_INT_ENABLE,
                     (READ_REGISTER(host, SPC_REG_INT_ENABLE)
                      | SPC_REG_INT_ENABLE_ENABLE_INT));
}

static inline void fuji_mb86606_disable_irqs(struct Scsi_Host *host) {
   WRITE_TO_REGISTER(host, SPC_REG_INT_ENABLE,
                     (READ_REGISTER(host, SPC_REG_INT_ENABLE)
                      & ~(SPC_REG_INT_ENABLE_ENABLE_INT)));
}

/* Function implementation */
int fuji_mb86606_detect(Scsi_Host_Template *tmpt) {
   int hosts = 0;
   u_short index;
   u_char bus, device_fn;
   struct Scsi_Host *host;
   struct fuji_mb86606_hostdata *hostdata;
   u_char i;

   ENTER("fuji_mb86606_detect");

   printk("fuji_mb86606: %s\n", vcid);

#ifdef FUJI_ENABLE_PROC_SUPPORT
   tmpt->proc_dir = &proc_scsi_fuji_mb86606;
#endif

   if (pcibios_present() == 0) {
      printk("fuji_mb86606: PCI bios not present\n");
      return 0;
   }
   
   memset(irq2host, 0, sizeof(irq2host));
   
   for (index = 0; pcibios_find_device(PCI_VENDOR_ID_FUJITSU,
                                       PCI_DEVICE_ID_FUJI_MB86606,
                                       index, &bus, &device_fn)
                   == PCIBIOS_SUCCESSFUL;
        index++) {

      host = scsi_register(tmpt, sizeof(struct fuji_mb86606_hostdata));
      if (!host) {
         printk("fuji_mb86606: where is my host zone %d ?.\n", index);
         continue;
      }
      hostdata = (struct fuji_mb86606_hostdata *) host->hostdata;
      
      memset(hostdata, 0, sizeof(struct fuji_mb86606_hostdata));
      hostdata->bus = bus;
      hostdata->device_fn = device_fn;
      hostdata->active_commands = 0;
      hostdata->flags = HOST_DATA_FLAGS_BUSY;
      hostdata->busy_with = NULL;

      for (i = 0; i < QUEUE_LENGTH; i++) {
         hostdata->commands[i].flags = 0;
      }
      
      if (fuji_mb86606_init(host)
          || fuji_mb86606_reset_hardware(host)) {
         scsi_unregister(host);
         continue;
      }
      
      host->this_id = SPC_OWN_ID;

      irq2host[host->irq] = host; /* This must be done before, in case of
                                   * spurious interrupts.
                                   */
      
      if (request_irq(host->irq,
                      fuji_mb86606_intr_handler,
                      SA_INTERRUPT,
                      "fuji_mb86606",
                      NULL)) {
         printk("fuji_mb86606: interrupt %d already in use\n",
                host->irq);
         iounmap(host->base);
         scsi_unregister(host);
         continue;
      }
      
      if (check_region(host->io_port, 0xff)) {
         printk("fuji_mb86606: i/o region 0x%04x-0x%04x already in use\n",
                host->io_port,
                host->io_port + 0xff);
         free_irq(host->irq, NULL);
         iounmap(host->base);
         scsi_unregister(host);
         continue;
      }

      request_region(host->io_port, 0xff, "fuji_mb86606");

      hostdata->flags = 0;
      fuji_mb86606_enable_irqs(host);
      
      hosts++;


      fuji_mb86606_had_reset(host); /* before */
      printk("fuji_mb86606: Resetting SCSI bus.\n");
      SEND_COMMAND(host, SPC_CMD_SCSI_RESET); /* SCSI reset */
   }
   
   LEAVE("fuji_mb86606_detect");
   
   return hosts;
}

int fuji_mb86606_release(struct Scsi_Host *host) {
   struct fuji_mb86606_hostdata *hostdata;
   
   ENTER("fuji_mb86606_release");
   
   hostdata = (struct fuji_mb86606_hostdata *) host->hostdata;
   
   free_irq(host->irq, NULL);
   
   release_region(host->io_port, 0xff);

   iounmap(host->base);

   /* TODO
    *    - Check if SCSI freeing for us scsi_unregister() ?
    */
   
   LEAVE("fuji_mb86606_release");
   
   return 0;
}

const char *fuji_mb86606_info(struct Scsi_Host *host) {
   struct fuji_mb86606_hostdata *hostdata;
   
   ENTER("fuji_mb86606_info");
 
   /* ASSUMPTION
    *    - sizeof(info_buf) is enough.
    */  
   hostdata = (struct fuji_mb86606_hostdata *) host->hostdata;
   sprintf(info_buf,
           "Fujitsu MB86606/PCI %d dev %d irq %ld I/O 0x%x mem 0x%lx",
           hostdata->bus,
           (hostdata->device_fn & 0xf8) >> 3,
           (u_long) host->irq,
           host->io_port,
           (u_long) host->base);
   
   LEAVE("fuji_mb86606_info");
   
   return info_buf;
}

static volatile int internal_done_flag = 0;
static volatile int internal_done_errcode = 0;

int fuji_mb86606_command(Scsi_Cmnd *Cmnd) {
   fuji_mb86606_queuecommand(Cmnd, internal_done);

   while (!internal_done_flag) { ; }

   internal_done_flag = 0;
   return internal_done_errcode;
}

/*
 * The middle SCSI layer ensures that queuecommand never gets invoked
 * concurrently with itself or the interrupt handler (though the
 * interrupt handler may call this routine as part of
 * request-completion handling).
 */
int fuji_mb86606_queuecommand(Scsi_Cmnd *Cmnd, void (*done)(Scsi_Cmnd *)) {
   struct Scsi_Host *host;
   struct Command_Entry *cmd;
   struct fuji_mb86606_hostdata *hostdata;
   int sg_count;
   struct scatterlist *sg;
   struct dataseg *ds;
   u_long flags;

   save_flags(flags);
   cli();
   
   ENTER("fuji_mb86606_queuecommand");

   if (!Cmnd) {
      printk("fuji_mb86606: NULL.\n");
   }

   host = Cmnd->host;
   hostdata = (struct fuji_mb86606_hostdata *) host->hostdata;
   Cmnd->scsi_done = done;
   
   DEBUG(fuji_mb86606_print_scsi_cmd(Cmnd));

   /* BUGS
    *    - Should check that no other command is active.
    *      Infact, later this should be ``room in the cmd queue'', e.g.
    *      use as a round-robin table (without using the holes if any).
    *      Or through flags !VALID if we have the time.
    * NOTES
    *    - Hope the following is atomic.
    */

   if (hostdata->active_commands >= QUEUE_LENGTH) {
      printk("fuji_mb86606: command table overflow.\n");
      restore_flags(flags);
      return 1;
   }
   else if (Cmnd->cmd_len >  FUJI_MB86606_SG_MAX_CDB_SIZE) {
      printk("fuji_mb86606: command size %d overflow.\n", Cmnd->cmd_len);
      restore_flags(flags);
      return 1;
   }
   else {
      cmd = hostdata->commands; /* first and only */

      memset(cmd, 0, sizeof(struct Command_Entry));

      cmd->handle = Cmnd;
      cmd->target_lun = Cmnd->lun;
      cmd->target_id = Cmnd->target;
      cmd->cdb_length = Cmnd->cmd_len;

      memcpy(cmd->cdb, Cmnd->cmnd, Cmnd->cmd_len);

      if (Cmnd->use_sg) {
         sg = (struct scatterlist *) Cmnd->request_buffer;
         ds = cmd->dataseg;
         cmd->segment_cnt = Cmnd->use_sg;

         for (sg_count = 0; sg_count < Cmnd->use_sg; sg_count++) {
            ds->d_base = (void *) virt_to_bus(sg->address);
            ds->d_count = sg->length;
            sg++;
            ds++;
         }
      }
      else {
	 cmd->dataseg[0].d_base = (void *) virt_to_bus(Cmnd->request_buffer);
	 cmd->dataseg[0].d_count = Cmnd->request_bufflen;
	 cmd->segment_cnt = 1;
      }

      cmd->current_segment = 1; /* 0 is invalid */
      
      hostdata->active_commands++;

      if (hostdata->flags & HOST_DATA_FLAGS_BUSY) {
         DEBUG(printk("fuji_mb86606: host busy; queuing.\n");)
         cmd->flags = COMMAND_ENTRY_FLAGS_VALID;
      }
      /* BUGS
       *    - When disconnect is enabled and multi-commands are allowed, or
       *      multiple-targets, the following is probably illegal, unless
       *      we disable interruptions, , send the cmd, and the interrupt
       *      routine is able to requeue the command somehow.
       */
      else if (READ_REGISTER(host, SPC_REG_STATUS) & SPC_REG_STATUS_BUSY) {
         printk("fuji_mb86606: host !busy; but status says busy.\n");
         restore_flags(flags);
         return 1;
      }
      else {
         DEBUG(printk("fuji_mb86606: host idle; sending.\n");)
         cmd->flags = COMMAND_ENTRY_FLAGS_VALID
                      | COMMAND_ENTRY_FLAGS_DONE_SELECT_S;
         hostdata->flags |= HOST_DATA_FLAGS_BUSY;
         hostdata->busy_with = cmd;
         hostdata->message_count = 1;

         hostdata->targets[cmd->target_id].flags
            &= ~(TARGET_INFO_FLAGS_IN_WIDE_NEGOCIATION
                 | TARGET_INFO_FLAGS_IN_SYNC_NEGOCIATION);

         hostdata->message_buffer[0] = IDENTIFY(FALSE, cmd->target_lun);

         /* Target ID */
         WRITE_TO_REGISTER(host, SPC_REG_SEL_RESEL_ID,
                                 cmd->target_id & SPC_REG_SEL_RESEL_ID_MASK);

#ifdef FUJI_ENABLE_WIDE
         if (!(hostdata->targets[cmd->target_id].flags
               & TARGET_INFO_FLAGS_DID_WIDE_NEGOCIATION)) {
            hostdata->message_buffer[1] = EXTENDED_MESSAGE;
            hostdata->message_buffer[2] = 2;
            hostdata->message_buffer[3] = EXTENDED_WDTR;
            hostdata->message_buffer[4] = 1; /* 16 bits */
            hostdata->message_count += 4;
            hostdata->targets[cmd->target_id].flags
               |= TARGET_INFO_FLAGS_IN_WIDE_NEGOCIATION;
            hostdata->targets[cmd->target_id].flags
               &= ~(TARGET_INFO_FLAGS_DID_SYNC_NEGOCIATION
                    | TARGET_INFO_FLAGS_IN_SYNC_NEGOCIATION);

            /* Kill settings */
            WRITE_TO_REGISTER(host, SPC_REG_SYNC_PERIOD, 0);
            WRITE_TO_REGISTER(host, SPC_REG_SYNC_OFFSET, 0);

         }
         else
#endif
#ifdef FUJI_ENABLE_SYNC
         if (!(hostdata->targets[cmd->target_id].flags
               & TARGET_INFO_FLAGS_DID_SYNC_NEGOCIATION)) {
            hostdata->message_buffer[1] = EXTENDED_MESSAGE;
            hostdata->message_buffer[2] = 3;
            hostdata->message_buffer[3] = EXTENDED_SDTR;
            hostdata->message_buffer[4] = FUJI_MIN_SYNC_PERIOD;
            hostdata->message_buffer[5] = FUJI_MAX_SYNC_OFFSET;
            hostdata->message_count += 5;
            hostdata->targets[cmd->target_id].flags
               |= TARGET_INFO_FLAGS_IN_SYNC_NEGOCIATION;

            /* Kill, but keep wide if any */
            WRITE_TO_REGISTER(host, SPC_REG_SYNC_PERIOD,
                              READ_REGISTER(host, SPC_REG_SYNC_PERIOD)
                              & SPC_REG_SYNC_PERIOD_WIDE_EN);
            WRITE_TO_REGISTER(host, SPC_REG_SYNC_OFFSET, 0);
         }
#endif


         /* NOTES
          *    - _IN_ shows that we did initiate the negociation. This
          *      means that when the target replies, we will set the
          *      proposed value and not reply. If the value is not
          *      supported, we will issue a new SDTR message.
          *    - In case the target does the negociation, if we do not
          *      agree, we send a message and perform as above.
          * BUGS
          *    - Next version should put the cmd structure array into SPC.
          */

         SEND_COMMAND(host, SPC_CMD_SELECT_WITH_ATN);
      }
   }

   LEAVE("fuji_mb86606_queuecommand");

   restore_flags(flags);
   return 0;
}

/* NOTES
 *    - All the virt-to-bus and bus-to-virt are because there is
 *      an assumption that in the future most of this will run in
 *      the Fujitsu part as a user program.
 */
void fuji_mb86606_intr_handler(int irq, void *dev_id, struct pt_regs *regs) {
   struct Scsi_Host *host;
   struct fuji_mb86606_hostdata *hostdata;
   struct Command_Entry *cmd;
   u_char isr_reg;
   u_char status_reg;
   u_char step_reg;
   u_char scsi_ctl;
   Scsi_Cmnd *Cmnd;
   u_char had_error;
   struct dataseg *ds;
   u_short len;
   u_char offset;
   u_char period;
   u_char wide;
   BOOL ok;
   u_long i;

   ENTER_INTR("fuji_mb86606_intr_handler");
   
   host = irq2host[irq];
   if (!host) {
      printk("fuji_mb86606: unexpected interrupt on line %d\n", irq);
      return;
   }

   LED_ON(host);

   hostdata = (struct fuji_mb86606_hostdata *) host->hostdata;

   had_error = 0;
   
   /* NOTES
    *    - see constraints on read order page 3-43.
    *    - There are no constraints on SPC_REG_STATUS
    *    - isr/seq could be read in one inw.
    */
   status_reg = READ_REGISTER(host, SPC_REG_STATUS);
   isr_reg = READ_REGISTER(host, SPC_REG_INTERRUPT);
   step_reg = READ_REGISTER(host, SPC_REG_COMMAND_STEP);
   scsi_ctl = READ_REGISTER(host, SPC_REG_SCSI_CONTROL);
   
   DEBUG_INTR(printk("fuji_mb86606: interrupt on line %d\n", irq));
   DEBUG_INTR(printk("fuji_mb86606: REG: isr 0x%x step 0x%x stat 0x%x ctl 0x%x.\n",
   			isr_reg, step_reg, status_reg, scsi_ctl);)

   if (hostdata->flags & HOST_DATA_FLAGS_BUSY) {
      if ((cmd = hostdata->busy_with)) { /* Any active command */
         Cmnd = (Scsi_Cmnd *) cmd->handle;

         if (cmd->flags & COMMAND_ENTRY_FLAGS_VALID) {
	    /* NOTES
	     *    - If handled in reverse order, could be more efficient.
	     */
            if (cmd->flags & COMMAND_ENTRY_FLAGS_DONE_SELECTED) {
               /* After selection completes. */
               if (isr_reg == SPC_REG_INTERRUPT_TERMINATED_NORMALLY) {
                  /* The first types of the
                   * comparisons are for COMMAND COMPLETE type of
                   * interrupts.
                   */
                  if (cmd->flags & COMMAND_ENTRY_FLAGS_DONE_STATUS_S) {
                     cmd->status = READ_REGISTER(host, SPC_REG_MCS_BUFFER);
                     cmd->flags |= COMMAND_ENTRY_FLAGS_DONE_STATUS;
                     cmd->flags &= ~(COMMAND_ENTRY_FLAGS_DONE_STATUS_S);
                     DEBUG_INTR(printk("fuji_mb86606: got status %d.\n", cmd->status);)
                     SEND_COMMAND(host, SPC_CMD_RESET_ACK);
                  }
                  else if (cmd->flags & COMMAND_ENTRY_FLAGS_DONE_DATA_S) {
                     cmd->flags |= COMMAND_ENTRY_FLAGS_DONE_DATA;
                     cmd->flags &= ~(COMMAND_ENTRY_FLAGS_DONE_DATA_S);
                     SEND_COMMAND(host, SPC_CMD_RESET_ACK);
                     DEBUG_INTR_DATA(printk("fuji_mb86606: data xfer finished.\n");)
                  }
                  else if (cmd->flags
                           & COMMAND_ENTRY_FLAGS_DONE_CMD_S) {
                     cmd->flags |= COMMAND_ENTRY_FLAGS_DONE_CMD;
                     cmd->flags &= ~(COMMAND_ENTRY_FLAGS_DONE_CMD_S);
                     DEBUG_INTR(printk("fuji_mb86606: CMD xfer finished.\n");)
		     len
			= (READ_REGISTER(host, SPC_REG_MODIFIED_BYTE_L)
			   | (READ_REGISTER(host, SPC_REG_MODIFIED_BYTE_H) << 8));
		     cmd->flags |= COMMAND_ENTRY_FLAGS_DONE_SELECTED;
		     cmd->flags &= ~(COMMAND_ENTRY_FLAGS_DONE_SELECT_S);
		     if (len != cmd->cdb_length) {
			printk("fuji_mb86606: short cmd: %d.\n", len);
			had_error = TRUE;
		     }
                     SEND_COMMAND(host, SPC_CMD_RESET_ACK);
                  }
                  else if (cmd->flags
                           & COMMAND_ENTRY_FLAGS_DONE_MIN_S) {
	             len
			= (READ_REGISTER(host, SPC_REG_MODIFIED_BYTE_L)
			   | (READ_REGISTER(host, SPC_REG_MODIFIED_BYTE_H) << 8));
                     if (len <= sizeof(hostdata->message_buffer)) {
                        hostdata->message_count = 0;
                        /* NOTES
                         *    - Emptying the MCS buffer is always needed.
                         *      Else the chip gives an error.
                         */
                        while (len--) {
                           hostdata->message_buffer[hostdata->message_count]
                              = READ_REGISTER(host,
                                              SPC_REG_MCS_BUFFER
                                              + hostdata->message_count);
                           hostdata->message_count++;
                        }

                        len = 0;

                        /* ASSUMPTIONS
                         *    - We currently assume we always get the
                         *      complete message. If this assumption is
                         *      wrong, a special bit MIN_S_CONT should be
                         *      set (and reset in ANY case where message_count
                         *      is touched), and we would get the n needed
                         *      message bytes. We also assume that ACK is
                         *      still pending and must be reset manually
                         *      (no auto-reset). There is nothing after WDTR
                         *      or SDTR.
                         * NOTES
                         *    - Message Parity Error and Reject are to be
                         *      handled as negociation reject, if in
                         *      negociation. Else, we set the error flag.
			 * BUGS
			 *    - Does not support the target changing it's
			 *      mind, such as SYNC/ULTRA -> no, Fast 10 -> ok,
			 *      Fast 10 -> REJECT. Would set Fast 10 there.
                         *      This is not supposed to happen, and would need
                         *      much more hacking.
			 */

                        DEBUG_INTR(printk("fuji_mb86606: @@ %d.\n",
                                          hostdata->message_count);)

                        ok = TRUE;
                        while (hostdata->message_count && ok) {
                           switch (hostdata->message_buffer[len]) {
                              case COMMAND_COMPLETE:
  		                 DEBUG_INTR(printk("fuji_mb86606: finish.\n");)
			         cmd->flags |= COMMAND_ENTRY_FLAGS_DONE_FINAL_M;
                                 /* Now expecting disconnect */
                                 break;
			      case MSG_PARITY_ERROR:
				 printk("fuji_mb86606: PARITY msg.\n");
				 /* no break intended */
			      case MESSAGE_REJECT:
				 if (hostdata->targets[cmd->target_id].flags
				     & TARGET_INFO_FLAGS_IN_WIDE_NEGOCIATION) {
				    hostdata->targets[cmd->target_id].flags
				     |= TARGET_INFO_FLAGS_DID_WIDE_NEGOCIATION;
				    DEBUG_INTR(printk("fuji_mb86606: refused WDTR; no more attempts.\n");)
                                    /* Because not yet confirmed, not yet
                                     * saved. We reset it anyway. Note that
                                     * if this is target-driven, IN_WIDE will
                                     * not be set and thus the following is
                                     * only called in the host-driven case.
                                     */
                                    wide = 0;
                                    fuji_configure_param(&wide,
                                                         NULL,
                                                         NULL,
                                                         host,
                                                         hostdata,
                                                         cmd->target_id);
                                    /* result ignored */
				 }
				 else if (hostdata->targets[cmd->target_id].flags
				     & TARGET_INFO_FLAGS_IN_SYNC_NEGOCIATION) {
				    hostdata->targets[cmd->target_id].flags
				       |= TARGET_INFO_FLAGS_DID_SYNC_NEGOCIATION;
				    DEBUG_INTR(printk("fuji_mb86606: refused sync; no more attempts.\n");)
                                    offset = 0;
                                    period = 0;
                                    fuji_configure_param(NULL,
                                                         &period,
                                                         &offset,
                                                         host,
                                                         hostdata,
                                                         cmd->target_id);
                                    /* result ignored */
				 }
				 else {
				    /* BUGS
				     *    - If MSG_PARITY_ERROR, should be
                                     *      handled.
				     */
				    printk("fuji_mb86606: REJECT/PAR msg 0x%x.\n",
					   hostdata->message_buffer[len]);
				    had_error = TRUE;
                                    ok = FALSE;
				    hostdata->message_count = 0;
				 }
				 break;
                              case EXTENDED_MESSAGE:
                                 if (((hostdata->message_count - 2)
                                      == hostdata->message_buffer[len + 1])
                                     && hostdata->message_count) {
#ifdef FUJI_ENABLE_WIDE
                                    if ((hostdata->message_buffer[len + 2]
                                         == EXTENDED_WDTR)
                                        && (hostdata->message_count == 4)) {
                                       wide
                                          = hostdata->message_buffer[len + 3];
                                       offset = 0;
                                       period = FUJI_MIN_SYNC_PERIOD;

                                       if (!fuji_configure_param(&wide,
                                                                 &period,
                                                                 &offset,
                                                                 host,
                                                                 hostdata,
                                                                 cmd->
                                                                  target_id)) {
				          hostdata->message_buffer[0]
				             = EXTENDED_MESSAGE;
					  hostdata->message_buffer[1]
					   = 2;
					  hostdata->message_buffer[2]
					   = EXTENDED_WDTR;
					  hostdata->message_buffer[3]
					   = wide;
					  hostdata->message_count = 4;
					  WRITE_TO_REGISTER(host,
                                                            SPC_REG_CONTROL,
						       SPC_REG_CONTROL_ATN_ON);
                                       } /* else we could send SDTR if
                                          * needed, and set flags, etc.
                                          */
                                       ok = FALSE; /* Finish here */
                                    }
                                    else
#endif
#ifdef FUJI_ENABLE_SYNC
                                    if ((hostdata->message_buffer[len + 2]
                                         == EXTENDED_SDTR)
                                        && (hostdata->message_count == 5)) {
                                       period
                                          = hostdata->message_buffer[len + 3];
                                       offset
                                          = hostdata->message_buffer[len + 4];

                                       if (!fuji_configure_param(NULL,
                                                                 &period,
                                                                 &offset,
                                                                 host,
                                                                 hostdata,
                                                                 cmd->target_id)) {
				          hostdata->message_buffer[0]
				             = EXTENDED_MESSAGE;
					  hostdata->message_buffer[1]
					   = 3;
					  hostdata->message_buffer[2]
					   = EXTENDED_SDTR;
					  hostdata->message_buffer[3]
					   = period;
					  hostdata->message_buffer[4]
					   = offset;
					  hostdata->message_count = 5;
					  WRITE_TO_REGISTER(host,
                                                            SPC_REG_CONTROL,
						       SPC_REG_CONTROL_ATN_ON);
                                       }
                                       ok = FALSE; /* Finish here */
                                    }
                                    else {
#endif
				       printk("fuji_mb86606: REJ %d/%d.\n",
                                              hostdata->message_buffer
                                                           [len + 2],
					      hostdata->message_count);
				       hostdata->message_buffer[0]
					  = MESSAGE_REJECT;
				       hostdata->message_count = 1;
				       ok = FALSE;
				       WRITE_TO_REGISTER(host, SPC_REG_CONTROL,
						       SPC_REG_CONTROL_ATN_ON);
#ifdef FUJI_ENABLE_SYNC
                                    }
#endif
                                 }
                                 else {
                                    printk("fuji_mb86606: EXT mism %d/%d.\n",
                                           hostdata->message_count - 2,
                                           hostdata->message_buffer[len + 1]);
				    had_error = TRUE;
                                    ok = FALSE;
				    hostdata->message_count = 0;
                                 }
                                 break;
                              default:
				 printk("fuji_mb86606: REJECTED message %d.\n",
					hostdata->message_buffer[len]);
				 hostdata->message_buffer[0]
				    = MESSAGE_REJECT;
				 hostdata->message_count = 1;
                                 ok = FALSE;
				 WRITE_TO_REGISTER(host, SPC_REG_CONTROL,
						   SPC_REG_CONTROL_ATN_ON);
				 break;
                           }

                           if (ok) {
                              len++;
                              hostdata->message_count--;
                           }
                        }
                     }
                     else {
                        printk("fuji_mb86606: long message %d error.\n",
                               len);
                        had_error = TRUE;
                     }

                     SEND_COMMAND(host, SPC_CMD_RESET_ACK);
                     cmd->flags &= ~(COMMAND_ENTRY_FLAGS_DONE_MIN_S);
                  }
                  else if (cmd->flags
                           & COMMAND_ENTRY_FLAGS_DONE_MOUT_S) {
                     len
			= (READ_REGISTER(host, SPC_REG_MODIFIED_BYTE_L)
			   | (READ_REGISTER(host, SPC_REG_MODIFIED_BYTE_H) << 8));

                     if (len != hostdata->message_count) {
                        printk("fuji_mb86606: msg OUT count incorrect %d/%d.\n",
                               len, hostdata->message_count);
                        had_error = TRUE;
                     }
                     hostdata->message_count = 0; /* sent successfully */
                     SEND_COMMAND(host, SPC_CMD_RESET_ACK);
                     cmd->flags &= ~(COMMAND_ENTRY_FLAGS_DONE_MOUT_S);
                  }
                  else {
                     printk("fuji_mb86606: unknown flags 0x%x in N\n",
                            cmd->flags);
                     had_error = TRUE;
                  }
               }
               else if (isr_reg == SPC_REG_INTERRUPT_REQ_ASSERTED) {
phase_change_handling:
                  /* This is for the ``phase change'' (REQ) interrupts. */
                  if (scsi_ctl & SPC_REG_SCSI_CONTROL_MSG) {
                     if (scsi_ctl & SPC_REG_SCSI_CONTROL_IO) {
                        DEBUG_INTR(printk("fuji_mb86606: PHASE msgin.\n");)
                        SEND_COMMAND(host, SPC_CMD_RECEIVE_MSG);
                        cmd->flags |= COMMAND_ENTRY_FLAGS_DONE_MIN_S;
                     }
                     else { /* MSG OUT */
                        /* NOTES
                         *    - message_count is reset at each data, cmd,
                         *      It is also reset at message in reception.
                         */
                        if (!hostdata->message_count) {
                           hostdata->message_buffer[0] = NOP;
                           hostdata->message_count = 1;
                        }

			for (i = 0; i < hostdata->message_count; i++) {
			   WRITE_TO_REGISTER(host, SPC_REG_MCS_BUFFER + i,
				             hostdata->message_buffer[i]);
			}                        

			/* Byte count */
			WRITE_TO_REGISTER(host, SPC_REG_MC_BYTE_L,
					  hostdata->message_count);
			WRITE_TO_REGISTER(host, SPC_REG_MC_BYTE_M,
				          0);
			WRITE_TO_REGISTER(host, SPC_REG_MC_BYTE_H,
					  0);

                        WRITE_TO_REGISTER(host, SPC_REG_CONTROL, 0);
                        SEND_COMMAND(host, SPC_CMD_SEND_N_BYTE_MSG);
                        DEBUG_INTR(printk("fuji_mb86606: PHASE msgout.\n");)
                        cmd->flags |= COMMAND_ENTRY_FLAGS_DONE_MOUT_S;
                     }
                  }
                  else { /* ! MSG */
                     /* Target did not react */
		     hostdata->targets[cmd->target_id].flags
			&= ~(TARGET_INFO_FLAGS_IN_WIDE_NEGOCIATION
			     | TARGET_INFO_FLAGS_IN_SYNC_NEGOCIATION);
                     hostdata->message_count = 0;
                     if (scsi_ctl & SPC_REG_SCSI_CONTROL_CD) {
                        if (scsi_ctl & SPC_REG_SCSI_CONTROL_IO) {
                           DEBUG_INTR(printk("fuji_mb86606: PHASE status.\n");)
                           SEND_COMMAND(host, SPC_CMD_RECEIVE_STATUS);
                           cmd->flags |= COMMAND_ENTRY_FLAGS_DONE_STATUS_S;
                        }
			else {
                           DEBUG_INTR(printk("fuji_mb86606: PHASE CMD.\n");)
			   for (i = 0; i < cmd->cdb_length; i++) {
			      WRITE_TO_REGISTER(host, SPC_REG_MCS_BUFFER + i,
						      cmd->cdb[i]);
			   }
			   /* Byte count, not really needed */
			   WRITE_TO_REGISTER(host, SPC_REG_MC_BYTE_L,
					     cmd->cdb_length);
			   WRITE_TO_REGISTER(host, SPC_REG_MC_BYTE_M,
				             0);
			   WRITE_TO_REGISTER(host, SPC_REG_MC_BYTE_H,
					     0);
			   SEND_COMMAND(host, SPC_CMD_SEND_CMD);
                           cmd->flags |= COMMAND_ENTRY_FLAGS_DONE_CMD_S;
			}
		     }
		     else {
                        if (cmd->flags & COMMAND_ENTRY_FLAGS_DONE_DATA_S) {
                           /* BUGS
                            *    - See BUGS at top of file.
                            */
                           DEBUG_INTR(printk("fuji_mb86606: stacked data phases, interrupt ignored.\n");)
                        }
			else
                        if (cmd->current_segment <= cmd->segment_cnt) {
			   ds = &(cmd->dataseg[cmd->current_segment - 1]);
			   cmd->current_segment++;

			   DEBUG_INTR_DATA(printk("fuji_mb86606: DMAbus @ 0x%lx len %lu seg %d/%d.\n", (u_long) ds->d_base, ds->d_count, cmd->current_segment - 1, cmd->segment_cnt);)
			   WRITE_TO_REGISTER(host, SPC_REG_DMA_XFER_VL,
				   	     (u_long) ds->d_base);
			   WRITE_TO_REGISTER(host, SPC_REG_DMA_XFER_L,
					     (u_long) ds->d_base >> 8);
			   WRITE_TO_REGISTER(host, SPC_REG_DMA_XFER_M,
					     (u_long) ds->d_base >> 16);
			   WRITE_TO_REGISTER(host, SPC_REG_DMA_XFER_H,
					     (u_long) ds->d_base >> 24);

			   WRITE_TO_REGISTER(host, SPC_REG_MC_BYTE_L,
					     ds->d_count);
			   WRITE_TO_REGISTER(host, SPC_REG_MC_BYTE_M,
					     ds->d_count >> 8);
			   WRITE_TO_REGISTER(host, SPC_REG_MC_BYTE_H,
					     ds->d_count >> 16);

                           if (ds->d_count > 0xFFFFFF) {
                              printk("fuji_mb86606: len bug.\n");
                           }

			   WRITE_TO_REGISTER(host, SPC_REG_DATA_BLOCK_L,
					     1);
			   WRITE_TO_REGISTER(host, SPC_REG_DATA_BLOCK_H,
					     0);

			   /* BUGS
			    *    - Some opt could be done above.
			    *    - Max is 16 MB
			    *    - Should verify direction based on something.
			    *    - Should also verify there IS a data xfer.
			    */
			   if (scsi_ctl & SPC_REG_SCSI_CONTROL_IO) {
			      DEBUG_INTR(printk("fuji_mb86606: PHASE datain.\n");)
			      cmd->flags |= COMMAND_ENTRY_FLAGS_DONE_DATA_S;
			      SEND_COMMAND(host, SPC_CMD_RECEIVE_DATA_TO_DMA);
			   }
			   else {
			      DEBUG_INTR(printk("fuji_mb86606: PHASE dataout.\n");)
 			      cmd->flags |= COMMAND_ENTRY_FLAGS_DONE_DATA_S;
			      SEND_COMMAND(host, SPC_CMD_SEND_DATA_FROM_DMA);
			   }
		        }
		        else {
			   printk("fuji_mb86606: segment underflow.\n");
			   had_error = TRUE;
	                }
                     }
                  }
               }
             else if ((isr_reg == SPC_REG_INTERRUPT_DISCONNECT)
			&& (cmd->flags & COMMAND_ENTRY_FLAGS_DONE_FINAL_M)
			&& (cmd->flags & COMMAND_ENTRY_FLAGS_DONE_STATUS)) {
		  DEBUG_INTR(printk("fuji_mb86606: final disconnect.\n");)
		  if ((cmd->current_segment > 1) /* had one I/O */
                      && (cmd->current_segment <= cmd->segment_cnt)) {
                     printk("fuji_mb86606: data underflow.\n");
                  } 
	          /* Final disconnect */
		  cmd->flags = 0;
		  hostdata->busy_with = NULL;
		  hostdata->active_commands--;
		  hostdata->flags &= ~(HOST_DATA_FLAGS_BUSY);
		  isr_reg = SPC_REG_INTERRUPT_TERMINATED_NORMALLY;
		  Cmnd->result = fuji_mb86606_state_to_result(cmd,
							      isr_reg,
							      step_reg,
							      status_reg);
		  (*(Cmnd->scsi_done))(Cmnd);
		  /* TODO
		   *    - Reschedule a command if needed/possible.
		   */
	       }
	       else if (isr_reg == SPC_REG_INTERRUPT_XFER_PHASE_ERROR) {
		  /* TODO
		   *    - Update RCV count.
		   *    - Check if happens only in data; and asserts it.
		   */
		  printk("fuji_mb86606: short xfer.\n");
		  cmd->flags |= COMMAND_ENTRY_FLAGS_DONE_DATA;
		  cmd->flags &= ~(COMMAND_ENTRY_FLAGS_DONE_DATA_S);
		  goto phase_change_handling; /* sorry :-: */
	       }
	       else {
		  printk("fuji_mb86606: unknown interrupt code 0x%x.\n",
			 isr_reg);
		  had_error = TRUE;
	       }
	    }
            else if (cmd->flags & COMMAND_ENTRY_FLAGS_DONE_SELECT_S) {
	       /* before selection completed successfully */
	       if (isr_reg == SPC_REG_INTERRUPT_TERMINATED_NORMALLY) {
                  cmd->flags |= COMMAND_ENTRY_FLAGS_DONE_SELECTED;
                  cmd->flags &= ~(COMMAND_ENTRY_FLAGS_DONE_SELECT_S);
                  /* Selection has completed successfully */
	       }
               else if (isr_reg == SPC_REG_INTERRUPT_REQ_ASSERTED) {
                  goto phase_change_handling; /* sorry */
               }
	       else {
		  DEBUG_INTR(printk("fuji_mb86606: sel error.\n");)
		  had_error = TRUE;
	       }
	    }
            else {
               printk("fuji_mb86606: cmd not sent error.\n");
               had_error = TRUE;
            }

	    if (had_error) {
	       DEBUG_INTR(printk("fuji_mb86606: had_error.\n");)
               if (isr_reg == SPC_REG_INTERRUPT_TERMINATED_NORMALLY) {
                  Cmnd->result = (DID_ERROR << 16);
               }
               else {
   	          Cmnd->result = fuji_mb86606_state_to_result(cmd,
			  				      isr_reg,
							      step_reg,
							      status_reg);
               }
	       /* BUGS
		*    - If the host is still used, should not do the
		*      following.
		* TODO
		*    - Reschedule a command if needed/possible.
		*/
	       cmd->flags = 0;
	       hostdata->busy_with = NULL;
	       hostdata->active_commands--;
	       hostdata->flags &= ~(HOST_DATA_FLAGS_BUSY);
	       (*(Cmnd->scsi_done))(Cmnd);
	    }
	 } 
	 else {
	   printk("fuji_mb86606: inconsistency (!valid).\n");
	 }
      }
      else { /* Unsupported event (could be reselect, e.g.) */
         printk("fuji_mb86606: !valid\n");
      }
   }
   else {
      /* Can only be a RESET initiated by us or any other
       * strange event, OR A RESELECTION (not supported yet).
       */
      if ((hostdata->flags & HOST_DATA_FLAGS_RSET)
          && (isr_reg == SPC_REG_INTERRUPT_TERMINATED_NORMALLY)) {
         printk("fuji_mb86606: RESET confirmation.\n");
         /* Kill all active commands */
   
         for (i = 0; i < QUEUE_LENGTH; i++) {
            if (hostdata->commands[i].flags & COMMAND_ENTRY_FLAGS_VALID) {
               Cmnd = (Scsi_Cmnd *)
                         hostdata->commands[i].handle;
   
               hostdata->commands[i].flags = 0;
   
               Cmnd->result = DID_RESET << 16;
               if (((Scsi_Cmnd *) hostdata->reset_cmd)
                   == Cmnd) {
                  hostdata->reset_cmd = NULL;
               }
               (*(Cmnd->scsi_done))(Cmnd);
            }
         }
   
         hostdata->flags = 0;
         hostdata->active_commands = 0;
         hostdata->busy_with = NULL;            
         if (hostdata->reset_cmd) {
            Cmnd = (Scsi_Cmnd *) hostdata->reset_cmd;
            Cmnd->result = (DID_RESET) << 16;
            (*(Cmnd->scsi_done))(Cmnd);
            hostdata->reset_cmd = NULL;
         }

         fuji_mb86606_had_reset(host);
      }
      else {
         printk("fuji_mb86606: !BUSY interrupt on line %d\n", irq);
      }
   }

   LED_OFF(host);
   LEAVE_INTR("fuji_mb86606_intr_handler");
}

/* NOTES
 *    - Can be called with a non valid cmd (yet non submitted Cmnd)
 */
int fuji_mb86606_abort(Scsi_Cmnd *Cmnd) {
#if 1
   printk("fuji_mb86606: abort calling reset.\n");
   return fuji_mb86606_reset(Cmnd, 0);
#else
   struct Scsi_Host *host;
   struct fuji_mb86606_hostdata *hostdata;
   int return_status = SCSI_ABORT_SUCCESS;

   ENTER("fuji_mb86606_abort");
   
   host = Cmnd->host;
   hostdata = (struct fuji_mb86606_hostdata *) host->hostdata;
   
   printk("fuji_mb86606: no abort.\n");

   /* ... Missing implementation ... */
   
   LEAVE("fuji_mb86606_abort");
   
   return return_status;
#endif
}

int fuji_mb86606_reset(Scsi_Cmnd *Cmnd, unsigned int reset_flags) {
   struct Scsi_Host *host;
   struct fuji_mb86606_hostdata *hostdata;
   int return_status = SCSI_RESET_SUCCESS;
   u_long flags;

   save_flags(flags);
   cli();
   
   ENTER("fuji_mb86606_reset");
   
   host = Cmnd->host;
   hostdata = (struct fuji_mb86606_hostdata *) host->hostdata;

   if (hostdata->flags & HOST_DATA_FLAGS_RSET) {
      printk("fuji_mb86606: reset in reset; doing SOFTWARE RESET this time.\n");
      fuji_mb86606_disable_irqs(host);
      if (fuji_mb86606_reset_hardware(host)) {
         fuji_mb86606_reset_hardware(host); /* result ignored */
      }
      hostdata->flags = 0;
      fuji_mb86606_enable_irqs(host);
      fuji_mb86606_had_reset(host); /* before */

      printk("fuji_mb86606: Resetting SCSI bus.\n");
      SEND_COMMAND(host, SPC_CMD_SCSI_RESET); /* SCSI reset */
      (*(Cmnd->scsi_done))(Cmnd);
   }
   else {
      printk("fuji_mb86606: resetting SCSI bus.\n");
      SEND_COMMAND(host, SPC_CMD_SCSI_RESET); /* SCSI reset */

      hostdata->active_commands = 0;
      hostdata->busy_with = NULL;
      hostdata->flags = HOST_DATA_FLAGS_RSET;

      /* BUGS
       *    - If Cmnd had some data pointing to cmd,
       *      would be much nicer. Don't forget to reset
       *      this association as soon as the cmd is finished.
       */
      hostdata->reset_cmd = Cmnd;

      fuji_mb86606_enable_irqs(host);

      LEAVE("fuji_mb86606_reset");
   }

   restore_flags(flags);
   
   return return_status;
}

int fuji_mb86606_biosparam(Disk *disk, kdev_t n, int ip[]) {
   int size = disk->capacity;
   
   ENTER("fuji_mb86606_biosparam");
   
   ip[0] = 64;
   ip[1] = 32;
   ip[2] = size >> 11;
   if (ip[2] > 1024) {
      ip[0] = 255;
      ip[1] = 63;
      ip[2] = size / (ip[0] * ip[1]);
      if (ip[2] > 1023) {
         ip[2] = 1023;
      }
   }
   
   LEAVE("fuji_mb86606_biosparam");
   
   return 0;
}

/* Private function implementation */


#define MAX_LOOP 10000
#define MAX_RESET_RETRIES 2
static int fuji_mb86606_reset_hardware(struct Scsi_Host *host) {
   u_short loop_count;
   
   ENTER("fuji_mb86606_reset_hardware");

   /* NOTES
    *    - The documentation says that if a hardware reset occured recently,
    *      we need to do ONE software reset to unstuck. It also says that
    *      if ONE software reset is sent with no hardware reset, we need
    *      to send a second one to unstuck. That is not very clear, and
    *      we have to consider a multi-insertion life-span (ie, with no
    *      resets). Thus this little bizarre approach.
    *    - That command does not cause an interrupt.
    * BUGS
    *    - Missing the 4 T delay.
    * TODO
    *    - Check if there is no way to send a hardware reset.
    */

   if (READ_REGISTER(host, SPC_REG_STATUS) & SPC_REG_STATUS_BUSY) {
      SEND_COMMAND(host, SPC_CMD_SOFTWARE_RESET); /* Unstuck hardware reset */
      loop_count = MAX_LOOP;
      while (loop_count--
             && (READ_REGISTER(host, SPC_REG_STATUS)
                 & SPC_REG_STATUS_BUSY)) { ; }
   }

   if (READ_REGISTER(host, SPC_REG_STATUS) & SPC_REG_STATUS_BUSY) {
      printk("fuji_mb86606: failed -- reset chip failed 1.\n");
      return 1;
   }

   SEND_COMMAND(host, SPC_CMD_SOFTWARE_RESET); /* Real reset */

   /* BUGS
    *    - Non accurate delay 4T.
    */
   loop_count = MAX_LOOP;
   while (loop_count--) { ; }

   if (!(READ_REGISTER(host, SPC_REG_STATUS) & SPC_REG_STATUS_BUSY)) {
      printk("fuji_mb86606: failed -- reset chip failed 2.\n");
      return 1;
   }

   /* Configure the Clock Conversion (this must be done here between the
    * two resets.
    */

   WRITE_TO_REGISTER(host, SPC_REG_CLOCK_CONV, SPC_CLOCK_CONVERSION);

   SEND_COMMAND(host, SPC_CMD_SOFTWARE_RESET);

   loop_count = MAX_LOOP;
   while (loop_count--
          && (READ_REGISTER(host, SPC_REG_STATUS)
              & SPC_REG_STATUS_BUSY)) { ; }

   if (READ_REGISTER(host, SPC_REG_STATUS) & SPC_REG_STATUS_BUSY) {
      printk("fuji_mb86606: failed -- reset chip failed 3.\n");
      return 1;
   }

   if (READ_REGISTER(host, SPC_REG_CLOCK_CONV) != SPC_CLOCK_CONVERSION) {
      printk("fuji_mb86606: can't read back SPC_REG_CLOCK_CONV.\n");
      return 1;
   }
   
   /* Initializing the chip */

   /* I/O control register
    *    - No manual data misalignment
    *    - No automatic data misalignment
    *    - INT PCI is open-drain
    *    - SCSI driver, both control and data lines are
    *      active negation (needed for FAST20).
    */
   WRITE_TO_REGISTER(host, SPC_REG_IO_CONTROL,
                     SPC_REG_IO_CONTROL_DC0
                     | SPC_REG_IO_CONTROL_DC1);

   /* SEL/RESEL ID register and xfer parameters for all
    * saved parameters. Not really needed
    * because of the SOFTWARE RESET.
    */

   /* PCI bursts and friends
    * NOTES
    *    - Fujitsu, in mail from hshenava@fmi.fujitsu.com on the
    *      Wed, 10 Dec 1997 10:21:42 -0800 tell there is a bug
    *      when using more than one burst word.
    */

   WRITE_TO_REGISTER(host, SPC_REG_PCI_CONTROL,
                     0); /* 1 long word burst */

   /* Clock Conversion has been done, see above */

   /* Own ID setting */

   WRITE_TO_REGISTER(host, SPC_REG_OWN_ID, SPC_OWN_ID);
   if ((READ_REGISTER(host, SPC_REG_OWN_ID) & SPC_REG_OWN_ID_MASK)
       != SPC_OWN_ID) {
      printk("fuji_mb86606: can't read back SPC_REG_OWN_ID.\n");
      return 1;
   }

   /* Response Operation Mode (INITIATOR ONLY configuration).
    *    - Reselection response enabled.
    *    - Selection response disabled.
    *    - No Automatic reception/phase change.
    *    - Not single initiator.
    *    - Arbitration.
    */
   WRITE_TO_REGISTER(host, SPC_REG_RESPONSE_MODE,
                     SPC_REG_RESPONSE_MODE_RESEL_ENABLE);
   if (READ_REGISTER(host, SPC_REG_RESPONSE_MODE)
       != (SPC_REG_RESPONSE_MODE_RESEL_ENABLE)) {
      printk("fuji_mb86606: can't read back SPC_REG_RESPONSE_MODE.\n");
      return 1;
   }

   /* SEL/RESEL Mode
    */
   WRITE_TO_REGISTER(host, SPC_REG_SEL_RESEL_MODE,
                     SPC_REG_SEL_RESEL_MODE_RESEL_MSG);
   if (READ_REGISTER(host, SPC_REG_SEL_RESEL_MODE)
       != SPC_REG_SEL_RESEL_MODE_RESEL_MSG) {
      printk("fuji_mb86606: can't read back SPC_REG_SEL_RESEL_MODE.\n");
      return 1;
   }

   /* Retry after selection timeout */
   WRITE_TO_REGISTER(host, SPC_REG_SEL_RESEL_RETRY,
                     SPC_SEL_RESEL_RETRIES);
   if (READ_REGISTER(host, SPC_REG_SEL_RESEL_RETRY)
       != SPC_SEL_RESEL_RETRIES) {
      printk("fuji_mb86606: can't read back SPC_REG_SEL_RESEL_RETRY.\n");
      return 1;
   }

   /* Selection timeout */
   WRITE_TO_REGISTER(host, SPC_REG_SEL_RESEL_TIMEO,
                     SPC_SELECTION_TIMEOUT);
   if (READ_REGISTER(host, SPC_REG_SEL_RESEL_TIMEO)
       != SPC_SELECTION_TIMEOUT) {
      printk("fuji_mb86606: can't read back SPC_REG_SEL_RESEL_TIMEO.\n");
      return 1;
   }

   /* REQ/ACK timeout */

   WRITE_TO_REGISTER(host, SPC_REG_REQ_ACK_TIMEOUT,
                     SPC_REQ_ACK_TIMEOUT);
   if (READ_REGISTER(host, SPC_REG_REQ_ACK_TIMEOUT)
       != SPC_REQ_ACK_TIMEOUT) {
      printk("fuji_mb86606: can't read back SPC_REG_REQ_ACK_TIMEOUT.\n");
      return 1;
   }

   /* Async setup */
   WRITE_TO_REGISTER(host, SPC_REG_ASYNC_SETUP,
                     SPC_ASYNC_SETUP);
   if (READ_REGISTER(host, SPC_REG_ASYNC_SETUP)
       != SPC_ASYNC_SETUP) {
      printk("fuji_mb86606: can't read back SPC_REG_ASYNC_SETUP.\n");
      return 1;
   }

   /* Parity error detection. In PCI mode, must be 11X01001
    *    - Check parity from SCSI
    *    - Even parity for system DMA (check)
    *    - Do not check the parity for DMA
    *    - Generate parity for SCSI
    *    - Generate parity for DMA. (check)
    * NOTES
    *    - SCSI parity generate is specified as also generating for DATAIN,
    *      thus if we want to detect in DATAIN as initiator, do we need
    *      to reset that ? It seems (after mail), thus we have to set it
    *      depending on the xfer direction in the non PCI case.
    */
   WRITE_TO_REGISTER(host, SPC_REG_PARITY_DETECT,
                     SPC_REG_PARITY_DETECT_SCSI_CHECK
                     | SPC_REG_PARITY_DETECT_SCSI_GEN
                     | SPC_REG_PARITY_DETECT_DMA_GEN);
   if (READ_REGISTER(host, SPC_REG_PARITY_DETECT)
       != (SPC_REG_PARITY_DETECT_SCSI_CHECK
           | SPC_REG_PARITY_DETECT_SCSI_GEN
           | SPC_REG_PARITY_DETECT_DMA_GEN)) {
      printk("fuji_mb86606: can't read back SPC_REG_PARITY_DETECT.\n");
      return 1;
   }

   /* Interrupt enable
    *    - Everything enabled, except interrupt itself, done
    *      in subroutines.
    */
   WRITE_TO_REGISTER(host, SPC_REG_INT_ENABLE,
                     SPC_REG_INT_ENABLE_SCSI_RESET
                     | SPC_REG_INT_ENABLE_REPORT
                     | SPC_REG_INT_ENABLE_TRANSFER_ERROR
                     | SPC_REG_INT_ENABLE_PHASE_ERROR
                     | SPC_REG_INT_ENABLE_AUTO_SEL_RESEL
                     | SPC_REG_INT_ENABLE_SEL_RESEL);
   if (READ_REGISTER(host, SPC_REG_INT_ENABLE)
       != (SPC_REG_INT_ENABLE_SCSI_RESET
           | SPC_REG_INT_ENABLE_REPORT
           | SPC_REG_INT_ENABLE_TRANSFER_ERROR
           | SPC_REG_INT_ENABLE_PHASE_ERROR
           | SPC_REG_INT_ENABLE_AUTO_SEL_RESEL
           | SPC_REG_INT_ENABLE_SEL_RESEL)) {
      printk("fuji_mb86606: can't read back SPC_REG_INT_ENABLE.\n");
      return 1;
   }

   /* Command group 6 and 7 SCSI length */
   WRITE_TO_REGISTER(host, SPC_REG_SPECIAL_CMD_LEN,
                     SPC_SPECIAL_CMD_LEN);
   if (READ_REGISTER(host, SPC_REG_SPECIAL_CMD_LEN)
       != SPC_SPECIAL_CMD_LEN) {
      printk("fuji_mb86606: can't read back SPC_REG_SPECIAL_CMD_LEN.\n");
      return 1;
   }

   /* Multi-node support */
   WRITE_TO_REGISTER(host, SPC_REG_RESPONSE_MAP_H,
                     SPC_RESPONSE_MAP_H);
   if (READ_REGISTER(host, SPC_REG_RESPONSE_MAP_H)
       != SPC_RESPONSE_MAP_H) {
      printk("fuji_mb86606: can't read back SPC_REG_RESPONSE_MAP_H.\n");
      return 1;
   }

   WRITE_TO_REGISTER(host, SPC_REG_RESPONSE_MAP_L,
                     SPC_RESPONSE_MAP_L);
   if (READ_REGISTER(host, SPC_REG_RESPONSE_MAP_L)
       != SPC_RESPONSE_MAP_L) {
      printk("fuji_mb86606: can't read back SPC_REG_RESPONSE_MAP_L.\n");
      return 1;
   }

   /* Operation reporting
    *    - Commands while interrupting are executed (?)
    *    - Hold the interrupt signal
    *    - Report command reject if BUSY
    *    - Do not report command reject if SEL
    *    - Do report REQ asserted
    *    - Do not report command complete for Set/Reset command.
    *    - Do not report level 2 interrupts (user program)
    *    - No Auto ACK reset.
    */
   WRITE_TO_REGISTER(host, SPC_REG_AUTO_OP_MODE,
                     SPC_REG_AUTO_OP_MODE_INT_HOLD
                     | SPC_REG_AUTO_OP_MODE_NO_COMPL_REPORT
                     | SPC_REG_AUTO_OP_MODE_NO_LEVEL_2);
   if (READ_REGISTER(host, SPC_REG_AUTO_OP_MODE)
       != (SPC_REG_AUTO_OP_MODE_INT_HOLD
           | SPC_REG_AUTO_OP_MODE_NO_COMPL_REPORT
           | SPC_REG_AUTO_OP_MODE_NO_LEVEL_2)) {
      printk("fuji_mb86606: can't read back SPC_REG_AUTO_OP_MODE.\n");
      return 1;
   }

   if ((READ_REGISTER(host, SPC_REG_REVISION_ID)
        & SPC_REG_REVISION_ID_MASK_CODE) != SPC_REVISION_ID_CODE) {
      printk("fuji_mb86606: revision ID is not %d", SPC_REVISION_ID_CODE); 
      return 1;
   }

   DEBUG(printk("fuji_mb86606: chip revision %d.\n",
         READ_REGISTER(host, SPC_REG_REVISION_ID)
         & SPC_REG_REVISION_ID_MASK_REVISION));

   /* Timeout setting */

   WRITE_TO_REGISTER(host, SPC_REG_SPC_TIMEOUT, SPC_SPC_TIMEOUT);

   LED_OFF(host); /* see definition, also sets PCI things */

   LEAVE("fuji_mb86606_reset_hardware");
   
#if 0
   return fuji_mb86606_set_user_program(host,
                                        user_program,
                                        sizeof(user_program));
#else
   return 0;
#endif
}

static int fuji_mb86606_init(struct Scsi_Host *sh) {
   u_int io_base;
   u_int page;
   u_int memory_base;
   struct fuji_mb86606_hostdata *hostdata;
   u_char bus, device_fn, revision, irq;
   u_short vendor_id, device_id, command;
   
   ENTER("fuji_mb86606_init");
   
   hostdata = (struct fuji_mb86606_hostdata *) sh->hostdata;
   bus = hostdata->bus;
   device_fn = hostdata->device_fn;
   
   if (pcibios_read_config_word(bus, device_fn, PCI_VENDOR_ID, &vendor_id)
       || pcibios_read_config_word(bus, device_fn, PCI_DEVICE_ID, &device_id)
       || pcibios_read_config_word(bus, device_fn, PCI_COMMAND, &command)
       || pcibios_read_config_dword(bus,
                                    device_fn,
                                    PCI_BASE_ADDRESS_0,
                                    &io_base)
       || pcibios_read_config_dword(bus,
                                    device_fn,
                                    PCI_BASE_ADDRESS_1,
                                    (u_int *) &memory_base)
       || pcibios_read_config_byte(bus,
                                   device_fn,
                                   PCI_CLASS_REVISION,
                                   &revision)
       || pcibios_read_config_byte(bus, device_fn, PCI_INTERRUPT_LINE, &irq)) {
      printk("fuji_mb86606: error reading PCI configuration\n");
      return 1;
   }
   
   if (vendor_id != PCI_VENDOR_ID_FUJITSU) {
      printk("fuji_mb86606: 0x%04x is not Fujitsu vendor ID\n",
             vendor_id);
      return 1;
   }
   
   if (device_id != PCI_DEVICE_ID_FUJI_MB86606) {
      printk("fuji_mb86606: 0x%04x does not match FUJI_MB86606 device id\n",
             device_id);
      return 1;
   }

   if (command & PCI_COMMAND_IO && (io_base & 3) == 1) {
      io_base &= PCI_BASE_ADDRESS_IO_MASK;
   }
   else {
      printk("fuji_mb86606: i/o mapping is disabled\n");
      return 1;
   }
   
   if (!(command & PCI_COMMAND_MASTER)) {
      printk("fuji_mb86606: bus mastering is disabled\n");
      return 1;
   }
   
   if (revision != FUJI_MB86606_REV_ID) {
      printk("fuji_mb86606: new FUJI_MB86606 revision ID (%d)\n", revision);
   }
   
   hostdata->revision = revision;

   printk("fuji: cmd 0x%x.\n", command);
   
   sh->irq = irq;
   sh->io_port = io_base;

   memory_base &= PCI_BASE_ADDRESS_MEM_MASK;
#if defined(__i386__)
   page = memory_base & PAGE_MASK;

   /* BUGS
    *    - no check
    */
   sh->base = (unsigned char *) (u_int) ioremap(page,
                                (memory_base - page) + FUJI_MEMORY_SIZE)
              + memory_base - page;
#else
   sh->base = memory_base;
#endif
   
   LEAVE("fuji_mb86606_init");
   
   return 0;
}

#if DEBUG_FUJI_MB86606
static void fuji_mb86606_print_scsi_cmd(Scsi_Cmnd *cmd) {
   int i;
   
   printk("fuji_mb86606: target = 0x%02x, lun = 0x%02x, cmd_len = 0x%02x\n",
          cmd->target, cmd->lun, cmd->cmd_len);
   printk("fuji_mb86606: command = ");
   for (i = 0; i < cmd->cmd_len; i++) {
      printk("0x%02x ", cmd->cmnd[i]);
   }
   printk("\n");
}

#endif /* DEBUG_FUJI_MB86606 */

static u_long fuji_mb86606_state_to_result(struct Command_Entry *cmd,
					   u_char isr_reg,
					   u_char step_reg,
					   u_char status_reg) {
   u_char host_status;
   u_long result;

   if (isr_reg == SPC_REG_INTERRUPT_TERMINATED_NORMALLY) {
      host_status = DID_OK;
   }
   else if (isr_reg == SPC_REG_INTERRUPT_TIMEOUT) {
      host_status = (cmd->flags & COMMAND_ENTRY_FLAGS_DONE_SELECTED)
                    ? DID_TIME_OUT
                    : DID_NO_CONNECT;
   }
   else {
      host_status = DID_ERROR;
   }

   result = (cmd->status & STATUS_MASK) | (host_status << 16);

   DEBUG_INTR(printk("fuji_mb86606: result is 0x%lx.\n", result);)
   return result;
}

static void fuji_mb86606_had_reset(struct Scsi_Host *host) {
   struct fuji_mb86606_hostdata *hostdata
      = (struct fuji_mb86606_hostdata *) host->hostdata;
   u_char i;

   for (i = 0; i < MAX_TARGETS; i++) {
      hostdata->targets[i].flags = 0;
   }

   for (i = 0; i <= 15; i++) {
      WRITE_TO_REGISTER(host, SPC_REG_SEL_RESEL_ID_MASK,
                        i & SPC_REG_SEL_RESEL_ID_MASK);
      /* Async; non wide */
      WRITE_TO_REGISTER(host, SPC_REG_SYNC_PERIOD, 0);
      WRITE_TO_REGISTER(host, SPC_REG_SYNC_OFFSET, 0);
      
   }  
}

static BOOL fuji_configure_param(u_char *wide,
				 u_char *period,
				 u_char *offset,
				 struct Scsi_Host *host,
                                 struct fuji_mb86606_hostdata *hostdata,
				 u_char target_id) {
   BOOL accepted = TRUE;
   BOOL do_confirm = FALSE;
   u_char reg1 = READ_REGISTER(host, SPC_REG_SYNC_PERIOD);
   u_char reg2 = READ_REGISTER(host, SPC_REG_SYNC_OFFSET);
   u_char future_reg1 = reg1;
   u_char future_reg2 = reg2;

   DEBUG_INTR(printk("fuji_mb86606: called with [%d,%d,%d].\n",
                     (wide ? *wide : -1), (period ? *period : -1),
                     (offset ? *offset : -1));)

   /* NOTES
    *    - If wide is disabled, currently by undefining FUJI_ENABLE_WIDE,
    *      we won't be called.
    */

   if (wide) {
      if (*wide > 1) {
         accepted = FALSE;
         *wide = 1; /* No need to set reg_x */
      }
      else if (*wide == 1) {
         future_reg1 |= SPC_REG_SYNC_PERIOD_WIDE_EN;
      }
      else {
         future_reg1 &= ~(SPC_REG_SYNC_PERIOD_WIDE_EN);
      }
   }

   if (offset && period) { /* BOTH must be set, or none */
      if (*offset > FUJI_MAX_SYNC_OFFSET) {
         accepted = FALSE;
         DEBUG_INTR(printk("fuji_mb86606: offset %d too big, reset to %d.\n",
                           *offset,
                           FUJI_MAX_SYNC_OFFSET);)
         *offset = FUJI_MAX_SYNC_OFFSET;
      }
      else {
         future_reg2 = *offset;
      }

      if (future_reg2 == 0) {
         future_reg1 &= ~(SPC_REG_SYNC_PERIOD_SYNC_EN);
      }
      else {
         future_reg1 |= SPC_REG_SYNC_PERIOD_SYNC_EN;
      }

      if (*period < FUJI_MIN_SYNC_PERIOD) {
         *period = FUJI_MIN_SYNC_PERIOD;
         DEBUG_INTR(printk("fuji_mb86606: period too small, reset to %d.\n",
                           FUJI_MIN_SYNC_PERIOD);)
         accepted = FALSE;
      }
      else {
         u_char tmp_value;

         /* NOTES
          *    - Formula is  wanted_speed x value == input clock.
          *      e.g.        20 MHz         2        40 MHz
          *    - For simplicity we only accept a few values.
          */


         if (*period == 0xC) { /* special case */
            tmp_value = 2;
         }
         else {
            if (*period <= (255 / 4)) {
               tmp_value = (*period * 4) / 0x19;
	    }
	    else {
	       tmp_value = (*period / 0x19) * 4;
	    }

	    if (tmp_value < 2) {
	       tmp_value = 2;
	    }
	    if (tmp_value > 31) {
	       *period = 0;
               *offset = 0;
	       DEBUG_INTR(printk("fuji_mb86606: rounding period to async.\n");)
               accepted = FALSE; /* regx not used */
	    }

	    if (tmp_value <= (255 / 0x19)) {
	       accepted = (*period == ((tmp_value * 0x19) / 4)) && accepted;
               *period = (tmp_value * 0x19) / 4;
	    }
	    else {
	       accepted = (*period == ((tmp_value / 4) * 0x19)) && accepted;
               *period = (tmp_value / 4) * 0x19;
	    }
         }

         future_reg1 = (future_reg1 & ~(1|2|4|8|16)) | tmp_value;
      }
   }
   

   /* NOTES
    *    - We return TRUE if the values have been accepted, or if
    *      we did initiate and there is no reason to confirm.
    */

   do_confirm = !(hostdata->targets[target_id].flags
                  & (TARGET_INFO_FLAGS_IN_WIDE_NEGOCIATION
                   | TARGET_INFO_FLAGS_IN_SYNC_NEGOCIATION));

   if (accepted) {
      if (hostdata->targets[target_id].flags
          & TARGET_INFO_FLAGS_IN_WIDE_NEGOCIATION) {
         hostdata->targets[target_id].flags
            |= TARGET_INFO_FLAGS_DID_WIDE_NEGOCIATION;
      }
      if (hostdata->targets[target_id].flags
          & TARGET_INFO_FLAGS_IN_SYNC_NEGOCIATION) {
         hostdata->targets[target_id].flags
            |= TARGET_INFO_FLAGS_DID_SYNC_NEGOCIATION;
      }
      hostdata->targets[target_id].flags
         &= ~(TARGET_INFO_FLAGS_IN_WIDE_NEGOCIATION
              | TARGET_INFO_FLAGS_IN_SYNC_NEGOCIATION);
      WRITE_TO_REGISTER(host, SPC_REG_SYNC_PERIOD, future_reg1);
      WRITE_TO_REGISTER(host, SPC_REG_SYNC_OFFSET, future_reg2);      
      printk("fuji_mb86606: setting period/w 0x%x offset %d.\n",
             future_reg1, future_reg2);
      if (do_confirm) {
         DEBUG_INTR(printk("fuji_mb86606: sending msg.\n");)
      }
   }
   else {
      DEBUG_INTR(printk("fuji_mb86606: not accepted [%d,%d,%d].\n",
                        (wide ? *wide : -1), (period ? *period : -1),
                        (offset ? *offset : -1));)
   }

   return accepted && !do_confirm;
}

static int fuji_mb86606_set_user_program(struct Scsi_Host *host,
                                         u_char *user_program,
                                         u_int user_program_length) {
   if (user_program_length > FUJI_MAX_USER_PROGRAM_LENGTH) {
      printk("fuji_mb86606: user program too big (%d vs %d).\n",
             user_program_length,
             FUJI_MAX_USER_PROGRAM_LENGTH);
      return 1;
   }
   else if (user_program_length % 4) {
      printk("fuji_mb86606: user program length not longword multiple.\n");
      return 1;
   }
   else {
      unsigned long int i;
      unsigned long int *current_pos = (unsigned long int *) user_program;

      /* Configure */

      WRITE_TO_REGISTER(host,
			SPC_REG_UP_CONTROL,
			0); /* 2 KB x 1. Data can't be accessed when BUSY */

#if 0 /* through data port and I/O */
      DEBUG(printk("fuji_mb86606: downloading user program (size %d bytes).\n",
                    user_program_length);)

      /* Copy program
       * BUGS
       *    - None of the method work.
       * TODO
       *    - If the system is not intelligent, we might have to
       *      transfers words or longwords in PCI mode.
       */

      /* Set the starting address */
      WRITE_TO_REGISTER(host,
			SPC_REG_UP_SET_ADDR_L,
                        0);
      WRITE_TO_REGISTER(host,
			SPC_REG_UP_SET_ADDR_H,
                        0);

      /* Transfer (in bytes) */
      for (i = 0; i < user_program_length; i++) {
         WRITE_TO_REGISTER(host,
                           SPC_REG_UP_DATA_PORT_0,
                           user_program[i]);
      }

      /* Compare */

      /* Set the starting address */
      WRITE_TO_REGISTER(host,
			SPC_REG_UP_SET_ADDR_L,
                        0);
      WRITE_TO_REGISTER(host,
			SPC_REG_UP_SET_ADDR_H,
                        0);

      /* Verify (in bytes) */
      for (i = 0; i < user_program_length; i++) {
         if (user_program[i] != READ_REGISTER(host,
                                              SPC_REG_UP_DATA_PORT_0)) {
            printk("fuji_mb86606: can't verify user program (%d).\n",
                   i);
            return 1;
         }
      }
#else /* through memory PCI only, in words */
      /* Copy program */

      DEBUG(printk("fuji_mb86606: downloading user program (address 0x%lx size %d bytes).\n", host->base, user_program_length);)

      for (i = 0; i < (user_program_length >> 2); i++) { /* longword counter */
#if 0
         writel(*current_pos,
                (host->base + i + FUJI_USER_PROGRAM_OFFSET));
#else
         writel(*current_pos,
                (host->base + (i << 2) + FUJI_USER_PROGRAM_OFFSET));
#endif
         current_pos++;
      }

      /* Compare program */

      current_pos = (unsigned long int *) user_program;

      for (i = 0; i < (user_program_length >> 2); i++) { /* longword counter */
#if 0
         if (readl(host->base + i  + FUJI_USER_PROGRAM_OFFSET)
#else
         if (readl(host->base + (i << 2) + FUJI_USER_PROGRAM_OFFSET)
#endif
             != (*current_pos)) {
            printk("fuji_mb86606: can't verify user program (%ld).\n",
                   i);
            return 1;
         }
         current_pos++;
      }
#endif
   }

#if 1
   /* Set start address 0x000 */
   WRITE_TO_REGISTER(host,
                     SPC_REG_UP_START_L,
                     0);
   WRITE_TO_REGISTER(host,
                     SPC_REG_UP_START_H,
                     0);

   /* No auto resel support yet */

   printk("Launching up.\n");

   /* Launch user program */
   SEND_COMMAND(host, SPC_REG_CMD_LAUNCH_USER_PROGRAM);
   return 1;
#endif

   return 0;
}

static void internal_done(Scsi_Cmnd *Cmnd) {
   internal_done_errcode = Cmnd->result;
   ++internal_done_flag;
}

#ifdef MODULE
Scsi_Host_Template driver_template = FUJI_MB86606;

#include "scsi_module.c"
#endif /* MODULE */

