/* 
 * Runset Init: Stub "init" utility (Linux specific)
 *
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/kdaemon.h>
#include <sys/reboot.h>
#include <sys/syslog.h>
#include <sys/ioctl.h>

#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>

/* Process ID of the base runset */
static pid_t main_process;

/* Signal responsible for a shutdown.  It determines what happens when
 * shutdown is complete.
 */
static int reboot_signal;

/* Singleuser boot option
 *
 */
static int singleuser_flag = 0;


#ifndef NO_BDFLUSH

/* SIGALRM handler, which implements an "update daemon" for Linux 2.0.x
 * systems, without wasting an extra process-id slot.
 *
 * Linux 2.2 and further do not need an update daemon, and in fact
 * automatically kill any process that uses the bdflush(1,0) call.  Thus
 * this code has to be compiled out on recent kernels.
 */
void
update_handler (int dummy)
{
  bdflush (1, 0);
  alarm (5);
}

#endif /* !NO_BDFLUSH */

/* Shutdown signal handler
 * 
 * SIGTERM asks for halt
 * SIGHUP asks for reboot
 * SIGUSR1 asks for init to be re-execed in single user mode
 * SIGUSR2 asks for init to be re-execed in multiuser mode
 *
 * SIGINT also asks for reboot, but sends SIGINT instead of SIGTERM
 * to the main process.  If runset is the main process, this disables
 * handling of "tidy" actions, speeding the shutdown.
 *
 * This supports Linux's soft C-A-D, which sends SIGINT to process 1.
 * Also, shutdown can be neatly done with `kill 1'.
 */
static void
initiate_shutdown (int signumber)
{
  reboot_signal = signumber;
  switch (signumber)
    {
    case SIGHUP:
      syslog (LOG_NOTICE, "Received slow reboot signal. Shutting down.");
      break;
    case SIGINT:
      syslog (LOG_NOTICE, "Received reboot signal. Shutting down.");
      kill (main_process, SIGINT);
      return;
    case SIGTERM:
      syslog (LOG_NOTICE, "Received halt signal. Shutting down.");
      break;
    case SIGUSR1:
      syslog (LOG_NOTICE, "Received singleuser signal. Shutting down.");
      break;
    case SIGUSR2:
      syslog (LOG_NOTICE, "Received multiuser signal. Shutting down.");
      break;
    }
  kill (main_process, SIGTERM);
}


/* The entry point.
 * 
 * We set up our signal handlers and Linux's soft reboot feature. 
 *
 * Then function as an update daemon (no sense wasting a seperate pid 
 * slot for that.)  
 *
 */
int
main (int argc, char **argv)
{
  struct sigaction sa;
  pid_t process;
  int exit_status;
  const char *boot_tty;

  /* Idiot proofing... */
  if (getpid () != 1 || getuid () != 0)
    {
      printf ("This program is for booting only.\n");
      exit (EXIT_FAILURE);
    }

  /* Save the name of the boot terminal (usually /dev/tty1,
   * but not always. */
  boot_tty = ttyname (STDIN_FILENO);

  /* Check if single-user mode was requested by LILO (or equivalent).
   */
  while (*argv)
    {
      if (!strcmp ("single", *argv))
	singleuser_flag = 1;
      argv++;
    }

  openlog ("INIT", LOG_CONS, LOG_CRON);

  syslog (LOG_INFO, "Starting...");

  sigemptyset (&sa.sa_mask);
  sa.sa_flags = SA_RESTART;

#ifndef NO_BDFLUSH
  sa.sa_handler = &update_handler;
  sigaction (SIGALRM, &sa, 0);
  alarm (5);
#endif /* !NO_BDFLUSH */

  main_process = fork ();

  if (main_process == -1)
    {
      syslog (LOG_EMERG, "cannot fork base process (%m)");
      reboot (RB_HALT_SYSTEM);
    };

  if (!main_process)
    {
      if (!singleuser_flag)
	execl ("/etc/rs.d/init/multiuser", "multiuser", 0);

      execl ("/etc/rs.d/init/singleuser", "singleuser", 0);

      /* Try setting up a shell on our own */
      setsid ();
      ioctl (STDIN_FILENO, TIOCSCTTY, 1);
      execl ("/bin/sh", "sh", 0);

      _exit (128);
    }

  sa.sa_handler = &initiate_shutdown;
  sigaction (SIGTERM, &sa, 0);
  sigaction (SIGINT, &sa, 0);
  sigaction (SIGHUP, &sa, 0);
  sigaction (SIGUSR1, &sa, 0);
  sigaction (SIGUSR2, &sa, 0);
  reboot (RB_DISABLE_CAD);

  while ((process = waitpid (WAIT_ANY, &exit_status, 0)) != main_process)
    {
      /* As Process 1 we also take responsibilty for orphan tasks.
       * These are logged, but are otherwise ignored.
       *
       * Note I'm blithely assuming no possible errors. There are
       * three defined error codes for waitpid - EINTR, ECHILD,
       * and EINVAL.  None seem possible.
       */
      syslog (LOG_INFO, "orphan process (#%d) collected", process);
    }

  if (exit_status)
    syslog (LOG_EMERG, "Base process has failed.");
  else
    syslog (LOG_NOTICE, "Base process has exited.");

  /* Reopen the boot terminal.  We have to do this because the
   * original fds may have been "hung up". */
  if (boot_tty)
    {
      int fd;
      fd = open (boot_tty, O_RDWR);
      if (fd != -1)
	{
	  dup2 (fd, STDIN_FILENO);
	  dup2 (fd, STDOUT_FILENO);
	  dup2 (fd, STDERR_FILENO);
	  close (fd);
	}
    }

  switch (reboot_signal)
    {
    default:
      break;
    case SIGINT:
    case SIGHUP:
      kill (-1, SIGSTOP);
      sync ();
      reboot (RB_ENABLE_CAD);
      reboot (RB_AUTOBOOT);
      break;
    case SIGUSR1:
      execl ("/sbin/init", "init", "single", 0);
      break;
    case SIGUSR2:
      execl ("/sbin/init", "init", 0);
      break;
    }

  /* The system-halt case is the default and fallback */
  kill (-1, SIGSTOP);
  sync ();
  reboot (RB_ENABLE_CAD);
  reboot (RB_HALT_SYSTEM);
  return EXIT_FAILURE;
}
