/* 
 * Runset Init: Client to control `switch'
 *
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <errno.h>
#include <getopt.h>
#include <pwd.h>
#include <grp.h>
#include <unistd.h>

static void
err_abort (int number)
{
  fprintf (stderr, "%s: %s\n",
	   program_invocation_short_name, strerror (number));
  exit (EXIT_FAILURE);
}

static uid_t
interpret_user (const char *user)
{
  char *tail;
  uid_t id;

  errno = 0;

  id = strtol (user, &tail, 0);

  if (errno || *tail)
    /* If entire string doesn't make sense as number */
    {
      struct passwd *user_entry;

      user_entry = getpwnam (user);

      if (!user_entry)
	return -1;

      id = user_entry->pw_uid;
    }
  return id;
}

static int
user_in_group (uid_t user, const char *groupname)
{
  struct passwd *user_entry;
  struct group *group_entry;
  char *tail, **member_list;
  gid_t group_id;

  user_entry = getpwuid (user);

  errno = 0;

  group_id = strtol (groupname, &tail, 0);

  if (errno || *tail)
    {
      group_entry = getgrnam (groupname);
      if (user_entry && user_entry->pw_gid == group_entry->gr_gid)
	return 1;
    }
  else
    {
      if (user_entry && user_entry->pw_gid == group_id)
	return 1;
      group_entry = getgrgid (group_id);
    }

  if (!group_entry)
    return 0;

  member_list = group_entry->gr_mem;

  while (*member_list)
    {
      if (interpret_user (*member_list++) == user)
	return 1;
    }
  return 0;
}

static void
security_check (const char *runset, uid_t user, char mode)
{
  FILE *stream;
  char *buffer = 0;
  size_t size = 0;
  char *path;

  path = alloca (strlen (runset) + 18);
  strcpy (path, "/etc/rs.d/switch/");
  strcat (path, runset);

  stream = fopen (path, "r");

  if (!stream)
    {
      if (errno == ENOENT)
	fprintf (stderr, "%s: There is no such runset\n",
		 program_invocation_short_name);
      else
	fprintf (stderr, "%s: Can't open `%s' (%s)\n",
		 program_invocation_short_name, path, strerror (errno));
      exit (EXIT_FAILURE);
    }

  while (getline (&buffer, &size, stream) != -1)
    {
      buffer[strlen (buffer) - 1] = 0;	/* clip off newline */

      if (buffer[0] == '!')
	{
	  if (mode != '!' && !strncmp (buffer, "!user ", 6))
	    {
	      if (user == interpret_user (buffer + 6))
		{
		  fclose (stream);
		  return;
		}
	    }

	  if (mode == '!' && !strncmp (buffer, "!user/cancel ", 13))
	    {
	      if (user == interpret_user (buffer + 13))
		{
		  fclose (stream);
		  return;
		}
	    }
	  if (mode != '!' && !strncmp (buffer, "!group ", 7))
	    {
	      if (user_in_group (user, buffer + 7))
		{
		  fclose (stream);
		  return;
		}
	    }

	  if (mode == '!' && !strncmp (buffer, "!group/cancel ", 14))
	    {
	      if (user_in_group (user, buffer + 14))
		{
		  fclose (stream);
		  return;
		}
	    }
	}
    }

  free (buffer);

  if (ferror (stream))
    {
      fprintf (stderr, "%s: error reading runset definition (%s)\n",
	       program_invocation_short_name, strerror (errno));
      exit (EXIT_FAILURE);
    }

  fprintf (stderr, "%s: you aren't authorized to control this runset\n",
	   program_invocation_short_name);
  exit (EXIT_FAILURE);

  fclose (stream);
}

static void
send_command (FILE * connection, char command, const char *runset, uid_t user)
{
  switch (command)
    {
    case '*':
      fprintf (connection, "*\n");
      break;
    case '?':
    case '!':
      fprintf (connection, "%c%s\n", command, runset);
      break;
    case '+':
    case '-':
      fprintf (connection, "%c%s,%d\n", command, runset, user);
      break;
    }
}

static FILE *
connect_to_switch (void)
{
  struct sockaddr_un address = { AF_UNIX, "/dev/switch" };
  int csocket;
  FILE *stream;

  csocket = socket (PF_UNIX, SOCK_STREAM, 0);

  if (csocket == -1)
    {
      fprintf (stderr, "%s: Can't open socket (%s)\n",
	       program_invocation_short_name, strerror (errno));
      exit (EXIT_FAILURE);
    }

  if (connect (csocket, &address, sizeof (address)))
    {
      fprintf (stderr, "%s: Can't connect to /dev/switch (%s)\n",
	       program_invocation_short_name, strerror (errno));
      exit (EXIT_FAILURE);
    }

  stream = fdopen (csocket, "r+");

  if (!stream)
    {
      fprintf (stderr, "%s: cannot make high-level stream (%s)\n",
	       program_invocation_short_name, strerror (errno));
      exit (EXIT_FAILURE);
    }

  return stream;
}

static void
show_responses (FILE * connection)
{
  char *buffer = 0, *running, *word;
  size_t size = 0;

  while (getline (&buffer, &size, connection) != -1)
    {
      /* Clip off newline */
      buffer[strlen (buffer) - 1] = 0;

      running = buffer;

      /* get runset name */
      word = strsep (&running, " \n");

      printf ("%s", word);

      while ((word = strsep (&running, " \n")))
	{
	  struct passwd *user_entry;
	  uid_t userid;

	  userid = strtol (word, 0, 0);

	  printf (" %s", (user_entry = getpwuid (userid))
		  ? user_entry->pw_name : word);
	}
      printf ("\n");
    }

  free (buffer);

  /* If it was EOF, that's fine. But check for errors */
  if (ferror (connection))
    {
      fprintf (stderr, "%s: error reading response from daemon (%s)\n",
	       program_invocation_short_name, strerror (errno));
      exit (EXIT_FAILURE);
    }
}

int
main (int argc, char **argv)
{
  char mode = '+';
  int check = 1, show = 1;
  uid_t user;
  int x;
  FILE *connection;

  static const struct option lopts[] = {
    {"version", no_argument, 0, 256},
    {"help", no_argument, 0, 257},
    {"cancel", no_argument, 0, 'C'},
    {"force", no_argument, 0, 'f'},
    {"show", no_argument, 0, 'l'},
    {"remove", no_argument, 0, 'r'},
    {"silent", no_argument, 0, 'q'},
    {"quiet", no_argument, 0, 'q'},
    {"user", required_argument, 0, 'u'},
    {0, 0, 0, 0}
  };

  user = getuid ();

  while ((x = getopt_long (argc, argv, "Cflqru:", lopts, 0)) != -1)
    {
      switch (x)
	{
	case 'l':		/* list */
	  if (mode == '+')
	    {
	      mode = '?';
	      check = 0;	/* no security needed if just looking */
	    }
	  else
	    err_abort (EINVAL);
	  break;
	case 'r':		/* remove */
	  if (mode == '+')
	    mode = '-';
	  else
	    err_abort (EINVAL);
	  break;
	case 'f':		/* force */
	  if (!getuid ())
	    check = 0;
	  else
	    err_abort (EPERM);
	  break;
	case 'u':		/* as other user */
	  if (!getuid ())
	    {
	      user = interpret_user (optarg);
	      if (user == -1)
		{
		  fprintf (stderr, "%s: no such user `%s'\n",
			   program_invocation_short_name, optarg);
		  exit (EXIT_FAILURE);
		}
	    }
	  else
	    err_abort (EPERM);
	  break;
	case 'C':		/* cancel (force removal of all requests) */
	  if (mode == '+')
	    {
	      mode = '!';
	    }
	  else
	    err_abort (EINVAL);
	  break;
	case 'q':		/* quiet */
	  show = 0;
	  break;
	case 256:
	  printf ("request (" PACKAGE ") " VERSION "\n");
	  return EXIT_FAILURE;
	case 257:
	  printf ("Usage: %s [OPTION]... SCRIPT-NAME\n"
		  "       %s [OPTION]... -l [SCRIPT-NAME]\n\n"
		  "Send a request to the `switch' server"
		  " to activate a system script.  If\n"
		  "you have permission and the server is working,"
		  " this will cause the script\n"
		  "to run until you retract your request by"
		  " running the same command with the `-r' option.\n\n"
		  "If multiple users may request the same script,"
		  " then the script will not be\n"
		  "stopped until all users remove their requests.\n\n"
		  "  -f, --force                "
		  "ignore access controls [superuser only]\n"
		  "  -C, --cancel               "
		  "cancel all user's requests for a script\n"
		  "  -r, --remove               "
		  "remove your request for a script\n"
		  "  -u, --user=USER            "
		  "make request in USER's name [superuser only]\n"
		  "  -q, --quiet, --silent      "
		  "do not list users requesting this script\n"
		  "  -l, --show                 "
		  "list all users who requested a script\n"
		  "                             "
		  "  or all active scripts (if no name given)\n"
		  "      --help                 "
		  "display this help and exit\n"
		  "      --version              "
		  "output version information and exit\n\n"
		  "Report bugs to <michael@talamasca.ocis.net>\n",
		  argv[0], argv[0]);
	  return EXIT_FAILURE;
	}
    }

  argv += optind;
  argc -= optind;

  if (argc == 0 && mode == '?')
    mode = '*';
  else if (argc != 1)
    err_abort (EINVAL);

  /* Check for dangerous characters.
   * / would allow one to leave the /etc/runsets directory
   * , messes with our IPC with `switch'.
   */
  if (argc && strpbrk (argv[0], "/,"))
    err_abort (EINVAL);

  if (check)
    security_check (argv[0], user, mode);

  connection = connect_to_switch ();

  send_command (connection, mode, argv[0], user);
  if (show)
    show_responses (connection);

  fclose (connection);

  return EXIT_SUCCESS;
}
