/* 
 * Runset Init: Commands - calendar-based execution
 *
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include "common.h"
#include "dispatch.h"

#include <time.h>
#include <assert.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <sys/syslog.h>

/* Counters
 * 
 * We use a single-link list of matching time values in each category
 * (seconds-past-minutes,minutes-past-hour,hours,days,months).
 *
 * This list must be sorted from low value to high.
 */
struct counter
{
  int value;
  struct counter *next;
};

/* Convert the specification PARM into a chain of counters
 * Either a '*' or a comma-seperated list of integers is accepted.
 *
 * LIMIT is the highest number to attempt to match. It is 59 for 
 * seconds or minutes, 23 for hours, and so on.
 *
 * OFFSET is subtracted from the user-supplied numbers. It is zero for
 * units like seconds which are counted from zero, and 1 for units like
 * months, which are counted from one.
 */
static struct counter *
decode (const char *parm, int limit, int offset)
{
  char *temp, *word;
  struct counter *new, *tail = 0;

  do
    {
      int x = 0;

      if (!strcmp (parm, "*"))
	x = 1;
      else
	{
	  temp = strdupa (parm);

	  while ((word = strsep (&temp, ",")))
	    {
	      if ((strtol (word, 0, 0) - offset) == limit)
		x = 1;
	    }
	}

      if (x)
	{

	  new = malloc (sizeof (struct counter));
	  if (!new)
	    {
	      syslog (LOG_ERR, "Out of memory allocating counter structure");
	      return 0;
	    }

	  new->value = limit;
	  new->next = tail;

	  tail = new;
	}
    }
  while (limit--);

  if (!tail)
    /* A little vague... 
     * What we mean to say is, `of all the numbers you gave, none were in
     * the appropriate range, so I have a null set of times to launch the
     * program. This is pointless.'
     */
    syslog (LOG_ERR, "Time specification `%s' is invalid", parm);

  return tail;
}

/* Simple single-list deallocation of CHAIN */
static void
free_chain (struct counter *chain)
{
  if (chain)
    {
      free_chain (chain->next);
      free (chain);
    }
}

/* Increment a counter chain. Used to find the next time we must launch 
 * after we are synchronized.
 *
 * FIRST points to start of the chain, CURRENT points to a pointer to 
 * our location in the chain.
 *
 * Normally, we advance CURRENT one node, and return 0. However, if we 
 * reach the end of the chain, we reset (*CURRENT = FIRST) and return 1, 
 * signalling a carry. 
 */
static int
bump (struct counter **current, struct counter *first)
{
  *current = (*current)->next;

  if (*current)
    return 0;

  *current = first;

  return 1;
}

/* Skip 31st day of 30 day months 
 * 
 * CURRENT and FIRST are the pointers to all 5 counters. We bump the 
 * month counter and reset the day counter if the only days remaining on the 
 * day counter are out of range for the month
 *
 * YEAR is a pointer year of interest. It may be increased.
 * 
 */
static void
skip_31 (struct counter **current, struct counter **first, int *year)
{
  int skip;

  switch (current[4]->value)
    {
      /* Remember, months and day-of-month count from *zero */
    case 3:			/* April */
    case 5:			/* June */
    case 8:			/* September */
    case 10:			/* November */
      skip = current[3]->value == 30;
    case 1:			/* February */
      if (!((*year + 1900) % 400) || (*year % 100 && !(*year % 4)))
	skip = current[3]->value > 28;
      else
	skip = current[3]->value > 27;
    default:
      skip = 0;
    }

  if (skip)
    {
      current[3] = first[3];
      if (bump (&current[4], first[4]))
	(*year)++;
      skip_31 (current, first, year);
    }
  return;
}

/* Verify that time specification doesn't only match invalid dates */
static int
check_31 (struct counter *months, struct counter *days)
{
  int first_day;

  first_day = days->value;

  /* If we are told to wait for February 30th. (note: we are willing
   * to wait four years for the leap year) */
  if (first_day >= 29 && months->value == 2 && !months->next)
    {
      syslog (LOG_ERR, "Pattern only matches days > 29 in February");
      return 0;
    }

  if (first_day <= 31)
    return 1;

  while (months)
    {
      switch (months->value)
	{
	case 1:		/* February */
	case 3:		/* April */
	case 5:		/* June */
	case 8:		/* September */
	case 10:		/* November */
	  break;
	default:
	  return 1;
	}
      months = months->next;
    }


  syslog (LOG_ERR, "Pattern only matches the 31st in 30-day months");
  return 0;
}

/* Synchronize counters with an external time.
 * 
 * We set the counters so that they match the first time after or equal 
 * to the present time. (ie: the next time to launch a process 
 *
 * CURRENT and FIRST point to the arrays of counter chains, as in skip_31.
 *
 * VALUE contains an array of values that each counter should synchronize to.
 *
 * COUNT is which counter to synchronize. Outside callers specify the number
 * of the most-significant counter (months (4) at moment). The function 
 * will handle all lower counters recursively.
 *
 * The return value indicates a carry to the next higher counter (which 
 * for the outside would be the year.)
 */

static int
synchronize (struct counter **current,
	     struct counter **first, int *value, int count)
{
  int carry;

  current[count] = first[count];

  while (current[count] && (current[count]->value < value[count]))
    current[count] = current[count]->next;

  if (!current[count])
    {
      current[count] = first[count];
      carry = 1;
    }
  else if (current[count]->value == value[count])
    {
      if (!count || !synchronize (current, first, value, count - 1))
	return 0;
      carry = bump (&current[count], first[count]);
    }
  else
    {
      carry = 0;
    }

  while (count--)
    current[count] = first[count];

  return carry;
}

/* Convert the time stored in the counters into a unix time_t
 * 
 * COUNTERS is the array of counter chain pointers. YEAR is the current 
 * year.
 */
static time_t
counters_to_systime (struct counter *counters[5], int year)
{
  struct tm broken;

  broken.tm_year = year;
  broken.tm_mon = counters[4]->value;
  broken.tm_mday = counters[3]->value + 1;
  broken.tm_hour = counters[2]->value;
  broken.tm_min = counters[1]->value;
  broken.tm_sec = counters[0]->value;
  broken.tm_isdst = -1;

  return mktime (&broken);
}

/* Launch a process at regular intervals */
void
act_clock (hchain top, cvec vector, cenv env)
{
  time_t grace = 0, next;
  int grace_length, x[5], year;
  struct counter *first[5], *current[5];
  struct tm *broken_time;


  int handler (int shutdown)
  {
    time_t now;

    assert (!(*top) (0));

    if (shutdown)
      return graceful_shutdown (top, grace_length, &grace);
    else
      {
	time (&now);
	if (now >= next)
	  {
	    launch (top, vector + 7, env, STDIN_FILENO);

	    if (bump (&current[0], first[0]))
	      if (bump (&current[1], first[1]))
		if (bump (&current[2], first[2]))
		  if (bump (&current[3], first[3]))
		    if (bump (&current[4], first[4]))
		      year++;

	    skip_31 (current, first, &year);
	    next = counters_to_systime (current, year);
	  }
	setwake (next);
	return 0;
      }
  }

  verify_parameters (vector, 6, 0);

  /* TODO: Handle day-of-week */

  if (strcmp (vector[5], "*"))
    {
      syslog (LOG_ERR, "Day-of-week not implemented\n");
      return;
    }

  if (!((first[0] = decode (vector[1], 59, 0))
	&& (first[1] = decode (vector[2], 59, 0))
	&& (first[2] = decode (vector[3], 23, 0))
	&& (first[3] = decode (vector[4], 30, 1))
	&& (first[4] = decode (vector[6], 11, 1))
	&& check_31 (first[4], first[3])))
    return;

  grace_length = get_env_number (env, "grace", 10);

  time (&next);			/* temporarily using next to store current time */
  broken_time = localtime (&next);

  x[4] = broken_time->tm_mon;
  x[3] = broken_time->tm_mday - 1;
  x[2] = broken_time->tm_hour;
  x[1] = broken_time->tm_min;
  x[0] = broken_time->tm_sec;
  year = broken_time->tm_year;

  if (synchronize (current, first, x, 4))
    year++;			/* Bump year if we wrapped around to earlier month */

  skip_31 (current, first, &year);
  next = counters_to_systime (current, year);

  setwake (next);

  next_command (&handler);

  free_chain (first[0]);
  free_chain (first[1]);
  free_chain (first[2]);
  free_chain (first[3]);
  free_chain (first[4]);
}
