/* 
 * Runset Init: Command interpretation
 *
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include "common.h"

#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <getopt.h>
#include <fcntl.h>
#include <sys/syslog.h>


/* "global" environment. 
 * 
 * This environment pointer is the present base set of definitions. It
 * starts as NULL, but should environment definition(s) occur alone on a 
 * line, they are linked onto global. (a la' Bourne Shell).
 * 
 */
static cenv global = 0;

/* Current time */
time_t now;

/* Next time we should wake up for. 0 means we have no actions waiting 
 * for time. Only setwake(), and wait_on() should modify this. 
 */
static time_t wake = 0;

/* 
 * Wait for the last action to finish it's startup or shutdown.
 *
 * If FLAG is 0, the handler is called in startup mode, and the program 
 * waits for the handler to return 0, indicating that it is clear to 
 * start the next process.  If the handler is not finished, we 
 * sigsuspend() and try again.  When it does return 0, we return 1.
 *
 * If planned_shutdown is set (which is caused by a SIGINT or SIGTERM 
 * signal), we return 0 regardless. 
 *
 * If FLAG is 1, the handler is called in shutdown mode, and we also wait 
 * for it to return 0 as above. (planned_shutdown is ignored.)
 */
static int
wait_on (hchain handler, int flag)
{
  time (&now);

  if (!flag && planned_shutdown)
    return 0;

  if ((*handler) (flag))
    {
      if (wake)
	{
	  if (wake <= now)
	    {
	      wake = 0;
	      return wait_on (handler, flag);
	    }
	  else
	    alarm (wake - now);
	}
      else
	alarm (0);

      sigsuspend (&nonblocking);
      reap ();
      return wait_on (handler, flag);
    }

  return 1;
}

/* Set clock to wake up handler at a given absolute time.
 * Each handler that needs this should call it at every invocation, since
 * individual calls are not remembered - only the earliest time requested 
 * (from all handlers) is used.
 */
void
setwake (time_t new_wake)
{
  if (!wake || new_wake < wake)
    wake = new_wake;
}

/* Function that finds the function to implement a given command and
 * execute it.
 */
static void
dispatch_command (hchain top,
		  cvec vec, cenv env, const struct dtable *dispatch)
{
  if (dispatch->name)
    {
      if (!strcmp (vec[0], dispatch->name))
	(*dispatch->function) (top, vec, env);
      else
	dispatch_command (top, vec, env, dispatch + 1);
    }
  else
    syslog (LOG_ERR, "unrecognized command `%s'", vec[0]);
}

/* helper for next_command.
 * 
 * It processes environment definitions, and passes actual commands to 
 * dispatch_command().
 */
static void
parse_command (hchain top, cvec vector, cenv env)
{
  if (!vector)
    {
      if (feof (config_file))
	{
	  int final_handler (int shutdown)
	  {
	    assert (!(*top) (shutdown));
	    return 1;
	  }

	  fclose (config_file);
	  wait_on (&final_handler, 0);
	}
      else
	{
	  assert (ferror (config_file));
	  syslog (LOG_ERR, "problem reading configuration file (%m)");
	}
    }
  else if (!vector[0])
    {
      global = env;
      next_command (top);
    }
  else if (strchr (vector[0], '='))
    {
      struct environ new_node;

      new_node.prev = env;
      new_node.data = vector[0];

      parse_command (top, vector + 1, &new_node);

      return;
    }
  else
    dispatch_command (top, vector, env, dispatch_table);
}

/* Load the next command
 * 
 * Each command function calls next_command() after initializing itself.
 * next_command() does not return until runset shuts down.
 *  
 * TOP is a handler that should be called when a signal arrives. Each command
 * creates a handler (using GNU C nested functions/trampolines), which 
 * calls back to the previous TOP, and then passes this handler as TOP to 
 * it's own invocation of next_command(). 
 *
 * (Some commands do not care about signals, and pass TOP unmodified.)
 *
 */
void
next_command (hchain top)
{
  void thunk (cvec vec)
  {
    parse_command (top, vec, global);
  }

  if (wait_on (top, 0))
    read_vector (&thunk);

  wait_on (top, 1);
}

/* Get a string from the given environment 
 * 
 * DEF is a default, to be used if this item is not defined.
 */
const char *
get_env_string (cenv env, const char *tag, const char *def)
{
  const char *value;

  if (env)
    {
      value = strpbrk (env->data, "=");

      /* Nothing w/o an equal sign should be in the environment */
      assert (value);

      if (strncmp (env->data, tag, value - env->data))
	return get_env_string (env->prev, tag, def);

      if (value)
	return value + 1;	/* skip over equal sign */
    }
  return def;
}

/* Get a number from the environment
 * 
 * DEF is a default used if the definition is not present, or if
 * strtol() can't make sense of it.
 */
int
get_env_number (cenv env, const char *tag, int def)
{
  const char *string;
  char *tail;

  int num;

  string = get_env_string (env, tag, 0);

  errno = 0;

  if (!string)
    return def;

  num = strtol (string, &tail, 0);

  return (errno || *tail) ? def : num;
}


/* verify a command has sufficent parameters
 * 
 * Log an error and return 1 if:
 *   MIN is nonzero and there are less parameters than MIN in VECTOR
 *
 *   MAX is nonzero and there are more parameters than MAX in VECTOR 
 *
 * else return 0.
 *
 * This is intended to be used on the VECTOR parameter passed to an action
 * vector[0] is the name of the command, and is *not* counted for the 
 * minmax check (but is used in the error messages).
 */
int
verify_parameters (cvec vector, int min, int max)
{
  cvec pointer;
  int min1, max1;

  pointer = vector + 1;
  min1 = min;
  max1 = max;

  while (*pointer)
    {
      pointer++;
      min1--;
      max1--;
    }

  if (min1 > 0 || (max && max1 < 0))
    {
      if (min)
	{
	  if (max)
	    {
	      if (max == min)
		syslog (LOG_ERR,
			"Command `%s' requires %d parameters",
			vector[0], min);
	      else
		syslog (LOG_ERR,
			"Command `%s' requires %d--%d parameters",
			vector[0], min, max);
	    }
	  else
	    syslog (LOG_ERR,
		    "Command `%s' requires %d or more parameters",
		    vector[0], min);
	}
      else
	{
	  if (max)
	    syslog (LOG_ERR,
		    "Command `%s' allows no more than %d parameters",
		    vector[0], min);
	  else
	    assert (0);		/* How can we have a problem with no limits? */
	}
      return 1;
    }
  return 0;
}
