/* 
 * Runset Init: common header
 *
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <time.h>

/* Constant Vector 
 *
 * This is a typedef for an unmodifiable argv-type vector.
 */
typedef const char *const *cvec;

/* Handler
 *
 * This is a type for the handler-chain procedures. These are functions that
 * are called on a SIGIO, SIGCHLD, or SIGALRM.
 */
typedef int (*hchain) (int shutdown_flag);

/* Environment:
 *  These are linked structures that associate a tag with a value, and may
 *  be accessed with get_env_number() and get_env_string().  They are used
 *  to modify some commands.
 *
 *  The tag and value are stored as a single string seperated by an equal 
 *  sign.
 *
 *  Uppercase tags are dumped into the unix environment of child 
 *  processes.  Only lowercase tags should be used by runset itself.
 *
 *  Note that each structure has only one previous link, but multiple
 *  structures can link to the same previous one.
 *    
 *  0 for prev indicates no more definitions. 
 */
struct environ
{
  const struct environ *prev;
  const char *data;
};

/* Typedef for pointers to constant environment nodes, which appear a lot
 * in this code 
 */
typedef const struct environ *cenv;


/* Dispatch table for the different command types */
struct dtable
{
  const char *name;
  void (*function) (hchain top, cvec vector, cenv env);
};

/* Sort-of-dispatch table for the different socket protocols 
 * 
 * NAME is the name we refer to the protocol by.
 * FAMILY is the socket family used by that protocol (eg. PF_INET)
 * STYLE is the socket style used (eg. SOCK_STREAM)
 * NUMBER is the number passed to socket() for it (eg. IPPROTO_TCP).
 * BIND is a procedure that converts the string ADDRESS into a 
 *   suitable sockaddr format, and attempts to bind FD to that address.
 *   PROTOCOL is the name of the protocol used.  ENV is the environment.
 * 
 */
struct protocol
{
  const char *name;
  int family;
  int style;
  int number;
  int (*bind) (int fd, const char *protocol, char *address, cenv env);
};

/* main.c */
extern int planned_shutdown;
extern sigset_t nonblocking, blocking;
extern sigset_t received_signals;

/* command.c --- process commands */
extern time_t now;

void setwake (time_t new_wake);
void next_command (hchain top);
const char *get_env_string (cenv env, const char *tag, const char *def);
int get_env_number (cenv env, const char *tag, int def);
int verify_parameters (cvec vector, int min, int max);

/* lines.c --- break configuration file into argv-style vectors */

extern int config_line_number;
extern FILE *config_file;

int read_vector (void (*thunk) (cvec));

/* pid.c --- manage child processes */
int count (hchain cookie);
void killall (hchain cookie, int signumber);
int launch (hchain cookie, cvec vector, cenv env, int fd);
void reap (void);
int graceful_shutdown (hchain cookie, int grace_period, time_t * grace);

/* utmp.c */
void log_utmp (const char *id, pid_t pid, const char *user, int type);

/* dispach.c */
extern const struct dtable dispatch_table[];

/* sockets.c */
extern const struct protocol protocols_table[];
