/* 
 * Runset Init: Command-file line parser
 *
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include "common.h"

#include <stdlib.h>
#include <string.h>

/* Line number in config file
 * 
 * For enhancing error messages. Not used at the moment (TODO).
 */
int config_line_number = 0;

/* Configuration file stream */
FILE *config_file;

/*
 * Read a line. The line is collected using getline(), comments are
 * stripped, and lines are merged if they end in backlash.
 */
static char *
read_line (void)
{
  char *x = 0, *y, *z;
  size_t size = 0;

  if (getline (&x, &size, config_file) == -1)
    return 0;

  config_line_number++;

  /* Clip off comments 
   * 
   * ! indicates commands to `request', which are ignored by this
   * program.
   */
  y = strpbrk (x, "#;!");
  if (y)
    *y = 0;

  /* Merge lines */
  if (x[strlen (x) - 1] == '\\')
    {
      y = read_line ();
      if (y)
	{
	  z = malloc (strlen (x) + strlen (y) + 1);
	  if (z)
	    {
	      strcpy (z, x);
	      strcat (z, y);
	    }
	  free (x);
	  free (y);
	  x = z;
	}
      else
	x = 0;
    }
  return x;
}

/* Read a "vector" from a line. That is, convert a line into an argv-style
 * vector.
 *
 * THUNK is called with the result. This lets us use alloca for higher 
 * efficency
 */
int
read_vector (void (*thunk) (cvec vector))
{
  char *line;
  const char **buffer;
  int counter = 1;

  line = read_line ();

  if (!line)
    {
      (*thunk) (0);
      return 0;
    }

  {
    char *scan;

    scan = line;

    while (*scan)
      if (strchr ("\n\t ", *scan))
	scan++;
      else
	{
	  counter++;
	  scan = strpbrk (scan, "\n\t ");
	}
  }

  buffer = alloca (counter * sizeof (char *));

  {
    int counter2;

    counter2 = 0;

    if ((buffer[counter2++] = strtok (line, "\n\t ")))
      while ((buffer[counter2++] = strtok (0, "\n\t ")));
  }

  (*thunk) (buffer);
  return 1;
}
