/* 
 * Runset Init: Setup/Entry point
 *
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include "common.h"

#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <getopt.h>
#include <fcntl.h>
#include <sys/syslog.h>


/* Planned shutdown flag.
 *
 * This flag is set by SIGINT or SIGTERM, and causes us to unwind all 
 * actions and exit.
 *
 * If there is an error, we will exit without setting this flag. If this 
 * happens, we return EXIT_FAILURE as our exit code.
 *
 */
int planned_shutdown = 0;

/* Signal mask
 * 
 * BLOCKING is the set of signals we expect to recieve, and thus block
 * for most of the code. (It's not neccesarily the mask in effect during 
 * blocking.)
 *
 * NONBLOCKING is the normal signal mask. 
 */
sigset_t nonblocking, blocking;

/* Record of signals we have received */
sigset_t received_signals;

/* Base on which the handler chain is built. */
static int
base (int shutdown)
{
  sigdelset (&received_signals, SIGUSR1);
  return 0;
}

/* Our signal handler. This handles all signals */
static void
signal_handler (int signum)
{
  sigaddset (&received_signals, signum);

  switch (signum)
    {
    case SIGTERM:
    case SIGINT:
    case SIGHUP:
      planned_shutdown = 1;
      break;
    }
}

/* entry point
 * 
 * We check arguments, open the config file and syslog, and set the signals 
 * up. Then start the real code with next_command(). 
 */
int
main (int argc, char **argv)
{
  int x;
  int trap_hup;
  struct sigaction sa;

  static const struct option lopts[] = {
    {"version", no_argument, 0, 256},
    {"help", no_argument, 0, 257},
    {0, 0, 0, 0}
  };

  while ((x = getopt_long (argc, argv, "", lopts, 0)) != -1)
    {
      switch (x)
	{
	case 256:
	  printf (PACKAGE " " VERSION "\n");
	  return EXIT_FAILURE;
	case 257:
	  printf ("Usage: %s [OPTIONS] [SCRIPT]...\n\n"
		  "Manage a group of processes according to the provided"
		  " SCRIPT.  (Consult\nmanual for script syntax.)\n"
		  "      --help                 "
		  "display this help and exit\n"
		  "      --version              "
		  "output version information and exit\n\n"
		  "Report bugs to <michael@talamasca.ocis.net>\n", argv[0]);
	  return EXIT_FAILURE;
	}
    }

  argc -= optind;
  argv += optind;

  if (argc != 1)
    {
      fprintf (stderr, "Incorrect number of arguments\n");
      return EXIT_FAILURE;
    }

  openlog (program_invocation_short_name, LOG_PID | LOG_CONS, LOG_CRON);

  config_file = fopen (argv[0], "r");

  if (!config_file)
    {
      syslog (LOG_ERR, "can't load script file `%s' (%m)", argv[0]);
      return EXIT_FAILURE;
    }

  {
    /* prevent config file being passed to children */
    int fd;

    fd = fileno (config_file);
    fcntl (fd, F_SETFD, fcntl (fd, F_GETFD, 0) | FD_CLOEXEC);
  }

  sigemptyset (&received_signals);

  /* We'd like to handle SIGHUP gracefully.  Yet also we don't want to 
   * break the `nohup' utility.  So we sense if SIGHUP is already set to 
   * ignore, and leave it alone if so.
   */
  sigaction (SIGHUP, 0, &sa);
  trap_hup = sa.sa_handler != SIG_IGN;

  sigemptyset (&sa.sa_mask);
  sa.sa_flags = SA_RESTART;
  sa.sa_handler = &signal_handler;

  sigaction (SIGIO, &sa, 0);
  sigaction (SIGCHLD, &sa, 0);
  sigaction (SIGALRM, &sa, 0);
  sigaction (SIGTERM, &sa, 0);
  sigaction (SIGINT, &sa, 0);
  sigaction (SIGUSR1, &sa, 0);
  if (trap_hup)
    sigaction (SIGHUP, &sa, 0);

  sigemptyset (&blocking);
  sigaddset (&blocking, SIGIO);
  sigaddset (&blocking, SIGCHLD);
  sigaddset (&blocking, SIGALRM);
  sigaddset (&blocking, SIGTERM);
  sigaddset (&blocking, SIGINT);
  sigaddset (&blocking, SIGHUP);
  sigaddset (&blocking, SIGUSR1);
  sigprocmask (SIG_BLOCK, &blocking, &nonblocking);

  next_command (&base);
  return planned_shutdown ? EXIT_SUCCESS : EXIT_FAILURE;
}
