/* 
 * Runset Init: Commands - Simple filesystem mounting
 *
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include "common.h"
#include "dispatch.h"

#include <sys/mount.h>
#include <sys/swap.h>
#include <sys/syslog.h>
#include <string.h>
#include <errno.h>

static int
get_fs_flags (const char *param)
{
  int flags;
  char *buffer, *word;

  flags = MS_MGC_VAL;

  buffer = strdupa (param);

  while ((word = strsep (&buffer, ",")))
    {
      if (!strcmp (word, "ro"))
	flags |= MS_RDONLY;
      else if (!strcmp (word, "nosuid"))
	flags |= MS_NOSUID;
      else if (!strcmp (word, "nodev"))
	flags |= MS_NODEV;
      else if (!strcmp (word, "noexec"))
	flags |= MS_NOEXEC;
      else if (!strcmp (word, "sync"))
	flags |= MS_SYNCHRONOUS;
      else if (!strcmp (word, "remount"))
	flags |= MS_REMOUNT;
      else if (!strcmp (word, "mandlock"))
	flags |= MS_MANDLOCK;
      else if (!strcmp (word, "noatime"))
	flags |= MS_NOATIME;
    }
  return flags;
}


/* Mount and unmount a filesystem */
void
act_filesystem (hchain top, cvec vector, cenv env)
{
  int flags;
  const char *xopts;

  /* Shortcut
   * 
   * Run mount with appropriate arguments (taken from stack variables).
   * (only present to reduce clutter).
   * 
   * Return 1 iff mount fails.
   */
  int qmount (void)
  {
    return mount (vector[3], vector[2], vector[1], flags, xopts) == -1;
  }


  if (verify_parameters (vector, 2, 3))
    return;

  flags = get_fs_flags (get_env_string (env, "flags", ""));
  xopts = get_env_string (env, "options", "");

  if (qmount ())
    {
      if (errno == EROFS)
	{
	  flags |= MS_RDONLY;

	  if (qmount ())
	    {
	      syslog (LOG_ERR,
		      "Error on mounting `%s' (tried readonly) (%m)",
		      vector[2]);
	      return;
	    }
	  else
	    {
	      syslog (LOG_WARNING, "Had to mount `%s' readonly", vector[2]);
	    }
	}
      else
	{
	  syslog (LOG_ERR, "Error on mounting `%s' (%m)", vector[2]);
	  return;
	}
    }

  next_command (top);

  if (flags & MS_REMOUNT)
    {
      /* Special semantics for write-enabling and de-write-enabling
       * partitions. This is commonly done to root on Linux 
       */

      flags |= MS_RDONLY;

      if (qmount ())
	syslog (LOG_WARNING, "Couldn't remount `%s' readonly (%m)",
		vector[2]);
    }
  else if (umount (vector[2]) == -1)
    {
      if (!(flags & MS_RDONLY))
	{
	  flags |= MS_RDONLY | MS_REMOUNT;
	  if (qmount ())
	    syslog (LOG_WARNING,
		    "Couldn't unmount or remount readonly '%s' (%m)",
		    vector[2]);
	  else
	    syslog (LOG_WARNING,
		    "Couldn't unmount '%s'. Remounted readonly (%m)",
		    vector[2]);
	}
      else
	syslog (LOG_WARNING,
		"Couldn't unmount '%s'. (already readonly) (%m)", vector[2]);
    }
}

/* Add virtual memory */
void
act_swap (hchain top, cvec vector, cenv env)
{
  if (verify_parameters (vector, 1, 1))
    return;

  if (swapon (vector[1], 0) == -1)
    {
      syslog (LOG_WARNING,
	      "Could not mount swap partition %s (%m)", vector[1]);

      next_command (top);
    }
  else
    {
      next_command (top);
      swapoff (vector[1]);
    }
}
