/* 
 * Runset Init: Child process tracking
 *
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include "common.h"

#include <unistd.h>
#include <utmp.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/syslog.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>
#include <ctype.h>
#include <sys/ioctl.h>

/* Structure used to track child process
 * 
 * These are kept in a single-link-list via the pointer NEXT. PID is just
 * the process ID.
 *
 * COOKIE is the handler procedure of the command that started this
 * process. It is not called from this module at all - it is just used as 
 * a cookie so we can associate each process with an action.
 *
 * If this process is logged with utmp, UTID stores the utmp id of the
 * record so that can be removed on exit.
 */
struct child
{
  struct child *next;
  hchain cookie;
  pid_t pid;
  char utid[3];
};

/*
 * The root of the process list
 */
static struct child *children = 0;

static char *null_environment[1] = { 0 };


/* Transfer symbols from our environment chain ENV to the UNIX environment
 * A bit of a hack.
 */
static void
transfer_environment (cenv env)
{
  if (env)
    {
      transfer_environment (env->prev);

      if (isupper (env->data[0]))
	putenv ((char *) env->data);
    }
}

/* Set the persona
 * 
 *    PERSONA is a string containing either a username, or username and
 * groupname seperated by a dot.
 *
 *    If no group is present, the groups (including the extra groups) are
 * filled in from the passwd/group databases.
 */
static void
set_persona (const char *persona)
{
  char *username, *groupname;
  uid_t user;
  gid_t group;


  username = strdupa (persona);
  groupname = strchr (username, '.');
  if (groupname)
    *groupname++ = 0;

  /* This scope is *not* associated with the above `if' */
  {
    struct passwd *entry;
    entry = getpwnam (username);
    if (!entry)
      {
	syslog (LOG_WARNING, "Invalid user `%s'", username);
	_exit (128);
      }
    user = entry->pw_uid;
    group = entry->pw_gid;
  }

  if (groupname && *groupname)
    {
      struct group *group_entry;
      group_entry = getgrnam (groupname);
      if (!group_entry)
	{
	  syslog (LOG_WARNING, "Invalid group `%s'", username);
	  _exit (128);
	}
      group = group_entry->gr_gid;
      setgroups (1, &group);
    }
  else
    initgroups (username, group);

  setgid (group);
  setuid (user);
}

/* Launch a process
 * 
 * COOKIE is used as a cookie to associate the new process with
 * the calling action, and should be the handler passed to that command.
 * (*not* the handler that command created - the one before.)
 *
 * VECTOR is a vector to be passed to execv();
 *
 * ENV is an environment. The tags "utmp", "persona", and "ctty" are
 * checked.
 *  
 * FD is a file descriptor, used for the "service" command. It is 
 * dup2()'d to stdin and stdout. STDIN_FILENO may be specified to turn this
 * off.
 */
int
launch (hchain cookie, cvec vector, cenv env, int fd)
{
  struct child *new_child;
  const char *utid, *persona;
  int ctty;

  utid = get_env_string (env, "utmp", "");
  persona = get_env_string (env, "persona", 0);
  ctty = get_env_number (env, "ctty", 0);

  new_child = malloc (sizeof (struct child));
  if (!new_child)
    {
      syslog (LOG_WARNING,
	      "cannot alloc child structure for `%s' (%m)", vector[0]);
      return 1;
    }
  new_child->next = children;
  new_child->cookie = cookie;
  children = new_child;
  new_child->pid = fork ();

  if (new_child->pid == -1)
    {
      syslog (LOG_WARNING, "cannot launch child process for `%s' (%m)",
	      vector[0]);
      return 1;
    }

  if (!new_child->pid)
    {
      if (fd != STDIN_FILENO)
	{
	  dup2 (fd, STDIN_FILENO);
	  dup2 (fd, STDOUT_FILENO);
	  close (fd);
	}
      sigprocmask (SIG_SETMASK, &nonblocking, 0);
      if (*utid || ctty)
	setsid ();

      if (ctty)
	ioctl (STDIN_FILENO, TIOCSCTTY, 1);

      if (persona)
	set_persona (persona);

      environ = null_environment;
      transfer_environment (env);

      execvp (vector[0], (char **) vector);
      _exit (128);
    }

  strncpy (new_child->utid, utid, 2);

  if (new_child->utid[0])
    log_utmp (utid, new_child->pid, "", INIT_PROCESS);

  return 0;
}


/* 
 * Return a count of all active processes associated with COOKIE.
 */
int
count (hchain cookie)
{
  struct child *place;
  int countup = 0;

  place = children;

  while (place)
    {
      if (place->cookie == cookie)
	countup++;

      place = place->next;
    }

  return countup;
}

/* Send a signal SIGNUMBER to all processes associated with COOKIE */
void
killall (hchain cookie, int signumber)
{
  struct child *place;

  place = children;

  while (place)
    {
      if (place->cookie == cookie)
	kill (place->pid, signumber);
      place = place->next;
    }
}

/* Reap zombie processes, and remove their entries from the list */
void
reap (void)
{
  pid_t process;
  struct child *place, *temp = 0;

  process = waitpid (WAIT_ANY, 0, WNOHANG);

  if (process > 0)
    {
      place = children;
      if (place)
	{
	  if (process == place->pid)
	    {
	      children = place->next;
	      temp = place;
	    }
	  else
	    {
	      while (place->next && place->next->pid != process)
		place = place->next;

	      if (place->next)
		{
		  temp = place->next;
		  place->next = temp->next;
		}
	    }

	  if (temp)
	    {
	      if (temp->utid[0])
		log_utmp (temp->utid, process, "", DEAD_PROCESS);

	      free (temp);
	    }
	}
      reap ();
    }
}

/* Convienience function, which gives the shutdown semantics needed for
 * several commands. Specifically, the command sends SIGTERM to all 
 * processes owned by a command, and returns 0 when they are all dead. If 
 * one or more process has not exited by GRACE_PERIOD, they are SIGKILLed.  
 * 
 * A GRACE_PERIOD of zero causes an immediate SIGKILL.  A negative
 * GRACE_PERIOD means SIGKILL will never be used.
 *
 * COOKIE is the handler, still used as a cookie.
 *
 * GRACE_PERIOD is used as above. In an action, it should be taken from
 * get_env_number(env,"grace",10);
 *
 * GRACE is a pointer to a time_t object for the function's use. It should
 * be initialized to zero.
 */
int
graceful_shutdown (hchain cookie, int grace_period, time_t * grace)
{
  if (count (cookie))
    {
      if (!*grace)
	{
	  killall (cookie, grace_period ? SIGTERM : SIGKILL);

	  *grace = now + grace_period;

	  if (grace_period > 0)
	    setwake (*grace);
	}
      else if (grace_period <= 0)
	{
	  /* do nothing */
	}
      else if (now >= *grace)
	{
	  killall (cookie, SIGKILL);
	}
      else
	{
	  setwake (*grace);
	}
      return 1;
    }
  return 0;
}
