/* 
 * Runset Init: Commands - Socket service multiplexing
 * 
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include "common.h"
#include "dispatch.h"

#include <assert.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/syslog.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <netdb.h>

static const struct protocol *
find_protocol (const struct protocol *ptable, const char *name)
{
  if (ptable->name)
    {
      if (strcmp (ptable->name, name))
	return find_protocol (ptable + 1, name);
      else
	return ptable;
    }
  else
    return 0;
}

/* Make a socket. 
 *
 * SPECIFICATION is the description of the socket, which is a string of the 
 * form <protocol>:<address>, ie: "tcp:telnet".
 */
static int
make_socket (const char *specification, cenv env)
{
  char *address, *protocol_name;
  const struct protocol *protocol;

  protocol_name = strdupa (specification);
  address = strpbrk (protocol_name, ":");

  if (address)
    *address++ = 0;

  protocol = find_protocol (protocols_table, protocol_name);

  if (protocol)
    {
      int fd;

      if ((fd = socket (protocol->family, protocol->style, protocol->number))
	  == -1)
	{
	  syslog (LOG_ERR, "cannot make socket (%m)");
	  return -1;
	}

      if ((*protocol->bind) (fd, protocol->name, address, env) == -1)
	{
	  /* Kludge to deal with situation where the address to bind
	   * is given as a hostname, and gethostname() fails to obtain an
	   * IP address.
	   *
	   * That function passes its errors in h_errno, and there is no
	   * portable way to map those error codes to normal errno codes.
	   *
	   * So if errno is 0 when the function fails, we assume that the fail
	   * was due to gethostname(), and report the h_errno error.
	   *
	   * (This could cause confusion if a protocol-specific bind routine
	   * ever accidentally fails with an errno of 0, however.)
	   */
	  if (errno)
	    syslog (LOG_ERR, "cannot bind socket (%m)");
	  else
	    syslog (LOG_ERR, "cannot bind socket (%s)", hstrerror (h_errno));
	  close (fd);
	  return -1;
	}
      return fd;
    }
  else
    {
      syslog (LOG_ERR, "No such protocol `%s'", protocol_name);
      return -1;
    }
}

/* Run a circuit service */
void
act_service (hchain top, cvec vector, cenv env)
{
  int listen_socket, grace_period;
  time_t grace;

  int handler (int shutdown)
  {
    int accepting_socket;

    assert (!(*top) (0));

    if (shutdown)
      return graceful_shutdown (top, grace_period, &grace);

    if ((accepting_socket = accept (listen_socket, 0, 0)) != -1)
      {
	launch (top, vector + 2, env, accepting_socket);
	close (accepting_socket);
	handler (0);
      }
    return 0;
  }

  if (verify_parameters (vector, 2, 0))
    return;

  grace_period = get_env_number (env, "grace", 10);

  listen_socket = make_socket (vector[1], env);

  if (listen_socket == -1)
    return;

  fcntl (listen_socket, F_SETFD,
	 fcntl (listen_socket, F_GETFD, 0) | FD_CLOEXEC);
  fcntl (listen_socket, F_SETFL,
	 fcntl (listen_socket, F_GETFL, 0) | O_ASYNC | O_NONBLOCK);
  fcntl (listen_socket, F_SETOWN, getpid ());

  if (listen (listen_socket, get_env_number (env, "queue", 2)) == -1)
    {
      syslog (LOG_ERR, "can't listen to socket (%m)");
      return;
    }

  next_command (&handler);

  close (listen_socket);
}

/* Run a single-thread server 
 *
 * This supports two kinds of server:
 * 
 *   1) a datagram server which handles all pending packets and then 
 *   exits after a certain period of time.
 *
 *   2) a circuit or datagram server which expects it's socket to be given
 *   at startup, and then keeps running.
 *
 *   The server is launched once at startup, and then relaunched if a packet
 *   is ready and it is not still active.
 *
 */
void
act_pass_socket (hchain top, cvec vector, cenv env)
{
  int listen_socket, grace_period;
  time_t grace;

  int handler (int shutdown)
  {
    struct timeval zero = { 0, 0 };
    fd_set select_set;

    assert (!(*top) (0));

    if (shutdown)
      return graceful_shutdown (top, grace_period, &grace);

    if (count (top))
      return 0;			/* If server active, we do nothing */

    /* Make sure socket is in asynchrous mode.
     *
     * We have to do this each time, since we had to turn off asynchrous 
     * updates before hand the fd to a child. (Also, we don't care 
     * about socket activity while a server is running.)
     *
     * We also reassert that we own the fd. If the child process used
     * O_ASYNC itself, then it will have changed the owner.
     * 
     */
    fcntl (listen_socket,
	   F_SETFL, fcntl (listen_socket, F_GETFL, 0) | O_ASYNC);

    fcntl (listen_socket, F_SETOWN, getpid ());

    FD_SET (listen_socket, &select_set);

    if (select (0, &select_set, 0, 0, &zero) > 0)
      {
	/* A packet/connection is waiting */

	fcntl (listen_socket, F_SETFL,
	       fcntl (listen_socket, F_GETFL, 0) & ~O_ASYNC);

	launch (top, vector + 2, env, listen_socket);
	handler (0);
      }
    return 0;
  }

  if (verify_parameters (vector, 2, 0))
    return;

  grace_period = get_env_number (env, "grace", 10);

  listen_socket = make_socket (vector[1], env);

  if (listen_socket == -1)
    return;

  /* Set socket into listen-mode.
   *
   * This is inapplicable to datagram sockets, but in that case it fails 
   * harmlessly with EOPNOTSUPP.
   */
  if (listen (listen_socket, get_env_number (env, "queue", 2)) == -1
      && errno != EOPNOTSUPP)
    {
      syslog (LOG_ERR, "can't listen to socket (%m)");
      return;
    }

  /* Spawn one instance at startup */
  launch (top, vector + 2, env, listen_socket);

  next_command (&handler);

  close (listen_socket);
}
