/* 
 * Runset Init: Commands - simple process execution
 *
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include "common.h"
#include "dispatch.h"

#include <time.h>
#include <assert.h>
#include <unistd.h>
#include <signal.h>
#include <sys/syslog.h>

/* Launch a process, then go on. We don't care when/if it exits. */
void
act_once (hchain top, cvec vector, cenv env)
{
  time_t grace = 0;
  int grace_length;

  int handler (int shutdown)
  {
    assert (!(*top) (0));

    if (shutdown)
      return graceful_shutdown (top, grace_length, &grace);

    return 0;
  }

  if (verify_parameters (vector, 1, 0))
    return;

  grace_length = get_env_number (env, "grace", 10);

  if (!launch (top, vector + 1, env, STDIN_FILENO));
  next_command (&handler);
}

/* Launch a process. If it fails before we are told to shutdown, shut 
 * down the entire runset
 */
void
act_watch (hchain top, cvec vector, cenv env)
{
  time_t grace = 0;
  int grace_length;

  int handler (int shutdown)
  {
    assert (!(*top) (0));

    if (shutdown)
      return graceful_shutdown (top, grace_length, &grace);

    if (!count (top))		/* if process died */
      raise (SIGINT);		/* then kill ourself */

    return 0;
  }

  if (verify_parameters (vector, 1, 0))
    return;

  grace_length = get_env_number (env, "grace", 10);

  if (!launch (top, vector + 1, env, STDIN_FILENO));
  next_command (&handler);
}

/* Lauch a process, wait for it to finish, then go on */
void
act_wait (hchain top, cvec vector, cenv env)
{
  time_t grace = 0;
  int grace_length;

  int handler (int shutdown)
  {
    assert (!(*top) (0));

    if (shutdown)
      return graceful_shutdown (top, grace_length, &grace);

    return count (top);
  }

  if (verify_parameters (vector, 1, 0))
    return;

  grace_length = get_env_number (env, "grace", 10);

  if (!launch (top, vector + 1, env, STDIN_FILENO))
    next_command (&handler);
}

/* Lauch multiple processes asynchorously, but then wait for them all to 
 * finish.
 *
 * To use, all the processes *except the last* are given as `together' actions
 * the last must be a wait.  Consequences are undefined otherwise.
 */
void
act_together (hchain top, cvec vector, cenv env)
{
  if (verify_parameters (vector, 1, 0))
    return;

  /* This is sneaky.  Since we don't add any handler of our own, when we 
   * use the top one as a cookie for launch, we are using the same one 
   * the next wait or together will get.  Hence, the processes are 
   * treated as one pool.
   */

  if (!launch (top, vector + 1, env, STDIN_FILENO))
    next_command (top);
}

/* Don't do anything at startup, but launch process just before shutting
 * down */
void
act_leave (hchain top, cvec vector, cenv env)
{
  int launched = 0;
  int handler (int shutdown)
  {
    assert (!(*top) (0));

    if (shutdown)
      {
	if (launched)
	  return count (top);
	else
	  {
	    launch (top, vector + 1, env, STDIN_FILENO);
	    launched = 1;
	    return 1;
	  }
      }
    return 0;
  }

  if (verify_parameters (vector, 1, 0))
    return;

  next_command (&handler);
}


/* Like leave, but only activate if we were stopped by SIGTERM */
void
act_tidy (hchain top, cvec vector, cenv env)
{
  int launched = 0;
  int handler (int shutdown)
  {
    assert (!(*top) (0));

    if (shutdown && sigismember (&received_signals, SIGTERM))
      {
	if (launched)
	  return count (top);
	else
	  {
	    launch (top, vector + 1, env, STDIN_FILENO);
	    launched = 1;
	    return 1;
	  }
      }
    return 0;
  }

  if (verify_parameters (vector, 1, 0))
    return;

  next_command (&handler);
}

/* Launch a process again&again if necessary. */
void
act_respawn (hchain top, cvec vector, cenv env)
{
  time_t grace = 0, last_launch = 0;
  int grace_length, quiet_time, max_launches, launches = 0;

  int handler (int shutdown)
  {
    assert (!(*top) (0));

    if (shutdown)
      return graceful_shutdown (top, grace_length, &grace);

    if ((last_launch + quiet_time) <= now)
      launches = 0;

    if (launches >= max_launches)
      {
	setwake (last_launch + quiet_time);
	return 0;
      }

    if (!count (top))
      {
	last_launch = now;
	launches++;

	launch (top, vector + 1, env, STDIN_FILENO);
      }
    return 0;
  }

  if (verify_parameters (vector, 1, 0))
    return;

  grace_length = get_env_number (env, "grace", 10);
  quiet_time = get_env_number (env, "quiet", 60);
  max_launches = get_env_number (env, "max", 3);

  next_command (&handler);
}

/* Launch a process once, but if a SIGUSR1 is noted while it is dead,
 * it is respawned */
void
act_mrespawn (hchain top, cvec vector, cenv env)
{
  time_t grace = 0;
  int grace_length;
  int seen_usr1;

  int handler (int shutdown)
  {
    seen_usr1 = sigismember (&received_signals, SIGUSR1);

    assert (!(*top) (0));

    if (shutdown)
      return graceful_shutdown (top, grace_length, &grace);

    if (seen_usr1 && !count (top))
      launch (top, vector + 1, env, STDIN_FILENO);
    return 0;
  }

  if (verify_parameters (vector, 1, 0))
    return;

  grace_length = get_env_number (env, "grace", 10);

  if (!launch (top, vector + 1, env, STDIN_FILENO))
    next_command (&handler);
}

void
act_done (hchain top, cvec vector, cenv env)
{
  if (vector[1])
    {
      syslog (LOG_ERR, "Command `done' takes no parameters");
      return;
    }

  planned_shutdown = 1;
}
