/* 
 * Runset Init: Protocol-specific address-binding procedures
 *
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stddef.h>
#include <stdlib.h>
#include <errno.h>
#include <arpa/inet.h>

#include "common.h"

/* bind for internet
 * 
 * ADDRESS is interpreted as a name to look up in the services database
 */
static int
afinet_bind (int fd, const char *protocol, char *address, cenv env)
{
  struct sockaddr_in address_structure;
  char *host_address;

  address_structure.sin_family = AF_INET;

  host_address = strpbrk (address, "@");
  if (host_address)
    {
      struct hostent *host_entry;

      *host_address++ = 0;

      if (!inet_aton (host_address, &address_structure.sin_addr))
	{
	  host_entry = gethostbyname (host_address);
	  if (!host_entry)
	    {
	      /* Use a kludge inserted into services.c to handle 
	       * h_errno errors.
	       */
	      errno = 0;
	      return -1;
	    }
	  address_structure.sin_addr = *(struct in_addr *) host_entry->h_addr;
	}
    }
  else
    {
      address_structure.sin_addr.s_addr = INADDR_ANY;
    }

  {
    char *tail;

    errno = 0;
    address_structure.sin_port = htons (strtoul (address, &tail, 0));

    if (*tail || errno)
      {
	struct servent *service_entry;

	service_entry = getservbyname (address, protocol);
	if (!service_entry)
	  return -1;

	address_structure.sin_port = service_entry->s_port;
      }
  }

  return bind (fd, &address_structure, sizeof address_structure);
}

/* Bind for unix domain
 * 
 * ADDRESS is simply a path name.
 *
 */
static int
afunix_bind (int fd, const char *protocol, char *address, cenv env)
{
  struct sockaddr_un *address_structure;
  size_t address_size;

  address_size =
    strlen (address) + 1 + offsetof (struct sockaddr_un, sun_path);
  address_structure = alloca (address_size);
  address_structure->sun_family = AF_UNIX;
  strcpy (address_structure->sun_path, address);

  return bind (fd, address_structure, address_size);
}

const struct protocol protocols_table[] = {
  {"udp", PF_INET, SOCK_DGRAM, IPPROTO_UDP, &afinet_bind},
  {"tcp", PF_INET, SOCK_STREAM, IPPROTO_TCP, &afinet_bind},
  {"local", PF_UNIX, SOCK_STREAM, 0, &afunix_bind},
  {"localdg", PF_UNIX, SOCK_DGRAM, 0, &afunix_bind},
  {0, 0, 0, 0, 0}
};
