/* 
 * Runset Init: UTMP support (plus special command)
 *
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include "common.h"
#include "dispatch.h"

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/syslog.h>
#include <utmp.h>
#include <string.h>


/* A command that handles opening and closing the UTMP and WTMP logs
 * 
 * On startup it creates a fresh, blank UTMP file, and writes a startup
 * record to WTMP. On shutdown it writes the shutdown record and 
 * removes UTMP.
 */
void
act_openutmp (hchain top, cvec vector, cenv env)
{
  int file;

  if (vector[1])
    {
      syslog (LOG_ERR, "Command 'openutmp' takes no parameters");
      return;
    }

  /* Make sure UTMP exists, is empty, and has appropriate permissions
   *
   * (This is probably not compatible to glibc 2.1's utmpd.)
   */
  file = open (_PATH_UTMP, O_CREAT | O_TRUNC);
  fchmod (file, S_IRUSR | S_IRGRP | S_IROTH | S_IWUSR);
  fchown (file, 0, 0);
  close (file);
  log_utmp ("~~", 0, "reboot", BOOT_TIME);

  next_command (top);

  log_utmp ("~~", 0, "shutdown", RUN_LVL);
  unlink (_PATH_UTMP);
}

/* Log a record to UTMP and WTMP */
void
log_utmp (const char *id, pid_t pid, const char *user, int type)
{
  struct utmp record;

  /* Build record */

  memset (&record, 0, sizeof (struct utmp));
  strcpy (record.ut_user, user);
  strcpy (record.ut_id, id);
  strcpy (record.ut_line, "~");
  strcpy (record.ut_host, "");
  record.ut_pid = pid;
  record.ut_type = type;
  record.ut_time = now;

  /* Send it to status file */

  setutent ();
  pututline (&record);
  endutent ();

  /* And make a permanent record in WTMP */

  updwtmp (_PATH_WTMP, &record);

}
