/* 
 * Runset Init: Switch daemon control module
 *
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include "common.h"

#include <sys/socket.h>
#include <sys/un.h>
#include <sys/wait.h>
#include <sys/syslog.h>
#include <sys/stat.h>
#include <signal.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <getopt.h>

/* Original signal mask.  It is exposed because switcher.c needs to 
 * restore it before executing a child process.
 *
 * The blocking mask, for preventing switcher.c from being reentered,
 * is kept as sa.sa_mask in main().
 */
sigset_t nonblocking;

/* Socket we have open to listen for requests.  It is exposed so that the 
 * termination handler may close it on exit, and switcher.c may close it 
 * before executing a child process.
 */
int listening_socket;

/* SIGTERM/SIGINT handler 
 * 
 * This handler does not return.
 *
 */
static void
term_handler (int signum)
{
  /* Shut down incoming requests */
  close (listening_socket);
  unlink ("/dev/switch");

  /* Switch off all child processes */
  cancel_all ();

  /* Wait for child processes to shut down. 
   *
   * Since SIGCHLD is blocked (set up with our sa_mask), we may 
   * safely usurp zombie_handler's role. 
   */
  while (update ())
    reap (waitpid (WAIT_ANY, 0, 0));

  exit (EXIT_SUCCESS);
}

/* SIGCHLD handler
 * 
 * Pass along information about exited processes to the switcher.c module,
 * then do an update().
 *
 */
static void
zombie_handler (int signum)
{
  pid_t process;

  process = waitpid (WAIT_ANY, 0, WNOHANG);

  if (process > 0)
    {
      reap (process);
      zombie_handler (0);
    }
  else
    {
      update ();
    }
}

/* React to a command from `request'.
 * 
 * The protocol is
 *  '+RUNSET,UID\n' add a request for RUNSET by user UID
 *  '-RUNSET,UID\n' remove a request " " " " "
 *  '!RUNSET\n' cancel all requests for RUNSET - force it to shut down.
 *  '?RUNSET\n' list all users who have requests for RUNSET.
 *            (this is done by the all the above commands)
 *  '*\n' Dump a list of all active runsets.
 *
 * (No promise is made to keep the protocol constant between revisions)
 *
 * The reply is handled in switcher.c, and consists of
 *  "RUNSET: UID_1 UID_2 UID_3 ... UID_N\n"
 * 
 * Where RUNSET is the runset mentioned and UID_? is users requesting it.
 *
 */
static void
accept_command (int fd)
{
  FILE *stream;
  char *command;
  size_t cs;

  stream = fdopen (fd, "r+");

  if (!stream)
    {
      syslog (LOG_WARNING, "Couldn't open high-level stream (%m)");
      close (fd);
      return;
    }

  command = 0;
  cs = 0;

  if (getline (&command, &cs, stream))
    {
      char *runset;
      char *user;
      uid_t userid;

      runset = command + 1;
      user = strpbrk (command, ",");
      if (user)
	{
	  *user = 0;
	  user++;
	  userid = strtol (user, 0, 0);
	}
      else
	{
	  runset[strlen (runset) - 1] = 0;	/* Clip off newline */
	  userid = 0;
	}

      /* Protocol
       *  +<runset>,<user>   Request
       *  -<runset>,<user>   Remove request
       *  !<runset>          Force all requests dropped
       *  ?<runset>          Dump status of given runset.
       *  *                  Dump status of all runsets.
       */

      switch (command[0])
	{
	case '+':
	  syslog (LOG_INFO,
		  "Request for `%s' added for user %d", runset, userid);
	  add_request (runset, userid);
	  dump_requests (stream, runset);
	  break;
	case '-':
	  syslog (LOG_INFO,
		  "Request for `%s' removed for user %d", runset, userid);
	  remove_request (runset, userid);
	  dump_requests (stream, runset);
	  break;
	case '!':
	  syslog (LOG_INFO, "All requests for `%s' removed", runset, userid);
	  cancel_requests (runset);
	  dump_requests (stream, runset);
	  break;
	case '?':
	  dump_requests (stream, runset);
	  break;
	case '*':
	  dump_all (stream);
	  break;
	case 0:
	  syslog (LOG_WARNING, "Recieved empty command", command[0]);
	  break;
	default:
	  syslog (LOG_WARNING, "Recieved unknown command ('%c')", command[0]);
	}
      free (command);
    }
  else
    {
      if (feof (stream))
	syslog (LOG_WARNING, "EOF recieving command");
      else
	syslog (LOG_WARNING, "Error recieving command (%m)");
    }

  fclose (stream);
}

/* Entry point
 * 
 * Set up signal handlers, then listen on a control socket (/dev/switch) for
 * commands.
 */
int
main (int argc, char **argv)
{
  struct sigaction sa;
  struct sockaddr_un address = { AF_UNIX, "/dev/switch" };

  {
    static const struct option lopts[] = {
      {"version", no_argument, 0, 256},
      {"help", no_argument, 0, 257},
      {0, 0, 0, 0}
    };

    int x;

    while ((x = getopt_long (argc, argv, "", lopts, 0)) != -1)
      switch (x)
	{
	case 256:
	  printf ("switch (" PACKAGE ") " VERSION "\n");
	  return EXIT_FAILURE;
	case 257:
	  printf ("Usage: %s [OPTIONS] [SCRIPT-NAME]...\n\n"
		  "This is a server program that enables users"
		  " to start and stop specific\n"
		  "system runset scripts using the `request'"
		  " program.  The switch server\n"
		  "keeps track of all users who have"
		  " ``requested'' a script, and launches one\n"
		  "copy of it.  Once the last request"
		  " for a script has been retracted, it\n"
		  "then shuts down the script.\n\n"
		  "Scripts may be provided as arguments --"
		  " these will be automatically\n"
		  "started as if the superuser had" " requested them.\n\n"
		  "      --help                 "
		  "display this help and exit\n"
		  "      --version              "
		  "output version information and exit\n\n"
		  "Report bugs to <michael@talamasca.ocis.net>\n", argv[0]);
	  return EXIT_FAILURE;
	}
  }

  argc -= optind;
  argv += optind;

  sa.sa_flags = SA_RESTART;
  /* All signals are mutually exclusive.  
   *
   * sa.sa_mask is also our blocking mask to protect the switcher module 
   * from being reentered by a signal.  
   * 
   */
  sigemptyset (&sa.sa_mask);
  sigaddset (&sa.sa_mask, SIGCHLD);
  sigaddset (&sa.sa_mask, SIGTERM);
  sigaddset (&sa.sa_mask, SIGINT);

  sa.sa_handler = &zombie_handler;
  sigaction (SIGCHLD, &sa, 0);

  sa.sa_handler = &term_handler;
  sigaction (SIGTERM, &sa, 0);
  sigaction (SIGINT, &sa, 0);

  openlog (program_invocation_short_name, LOG_CONS, LOG_CRON);

  unlink (address.sun_path);

  listening_socket = socket (PF_UNIX, SOCK_STREAM, 0);
  if (listening_socket == -1)
    {
      syslog (LOG_ERR, "Can't open socket (%m)");
      exit (EXIT_FAILURE);
    }

  if (bind (listening_socket, &address, sizeof (address)))
    {
      syslog (LOG_ERR, "Can't bind socket to `/dev/switch' (%m)");
      exit (EXIT_FAILURE);
    }

  if (chmod (address.sun_path, S_IRUSR | S_IWUSR))
    {
      syslog (LOG_ERR, "Can't set permissions for `/dev/switch' (%m)");
      exit (EXIT_FAILURE);
    }

  if (listen (listening_socket, 0))
    {
      syslog (LOG_ERR, "Can't listen (%m)");
      exit (EXIT_FAILURE);
    }

  /* Preload active runsets... */
  if (argc > 0)
    {
      sigprocmask (SIG_BLOCK, &sa.sa_mask, &nonblocking);
      while (argc > 0)
	add_request (argv[--argc], 0);
      raise (SIGCHLD);
      sigprocmask (SIG_SETMASK, &nonblocking, 0);
    }

  while (1)
    {
      int asocket;

      asocket = accept (listening_socket, 0, 0);
      if (asocket > 0)
	{
	  sigprocmask (SIG_BLOCK, &sa.sa_mask, &nonblocking);
	  accept_command (asocket);
	  raise (SIGCHLD);
	  sigprocmask (SIG_SETMASK, &nonblocking, 0);
	}
      else
	syslog (LOG_WARNING, "Error on accepting connection (%m)");
    }

}
