/* 
 * Runset Init: Switch daemon (switching module)
 *
 * Copyright 1998, 1999, 2000, 2001, 2003, 2004, 2006,
 *           2007, 2009 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 3 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#define _GNU_SOURCE

#include "common.h"

#include <sys/types.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <unistd.h>
#include <sys/syslog.h>

struct request
{
  struct request *next;
  uid_t user;
};

struct runset
{
  struct runset *next;
  struct request *requests;
  const char *name;
  pid_t process;

  /* If pid_t nonzero (runset process is alive)
   *     flag == 1 means SIGINT has already been sent
   * If pid_t zero (runset process is dead, or not started)
   *     flag == 1 means runset failed, and we should cancel
   *        any requests for it, instead of restarting.
   */
  int flag;
};

static struct runset *runsets = 0;


/* Execute the runset program with the script in /etc/runsets/NAME.
 */
static pid_t
launch_runset (const char *name)
{
  pid_t childpid;
  char *path;

  syslog (LOG_INFO, "Starting runset `%s'", name);

  path = alloca (strlen (name) + 18);

  strcpy (path, "/etc/rs.d/switch/");
  strcat (path, name);

  childpid = fork ();
  if (childpid == -1)
    return 0;
  if (childpid != 0)
    return childpid;

  close (listening_socket);
  sigprocmask (SIG_SETMASK, &nonblocking, 0);
  execl ("/usr/sbin/runset", "runset", path, 0);
  execl ("/sbin/runset", "runset", path, 0);
  _exit (128);
}

/* Lookup a structure corresponding to runset NAME in our internal list.
 * 
 * 0 is returned if none is found.
 */
static struct runset *
find_runset (const char *name)
{
  struct runset *place;

  place = runsets;

  while (place && strcmp (place->name, name))
    place = place->next;

  return place;
}

/* Add a request for the runset NAME by USER 
 * 
 * This usually requires building new data structures, and may fail under
 * OOM conditions. 
 * 
 * Nothing happens if this user already requested the given runset
 */
void
add_request (const char *name, uid_t user)
{
  struct runset *a_runset;
  struct request *place;

  a_runset = find_runset (name);

  if (!a_runset)
    {
      a_runset = malloc (sizeof (struct runset));
      if (!a_runset)
	{
	  syslog (LOG_ERR, "Can't add request due to low memory (case 1)");
	  return;
	}

      a_runset->next = runsets;
      a_runset->name = strdup (name);
      a_runset->requests = 0;
      a_runset->process = 0;
      a_runset->flag = 0;
      runsets = a_runset;
    }

  place = a_runset->requests;

  while (place && place->user != user)
    place = place->next;

  if (!place)
    {
      place = malloc (sizeof (struct request));
      if (!place)
	{
	  syslog (LOG_ERR, "Can't add request due to low memory (case 2)");
	  return;
	}
      place->next = a_runset->requests;
      place->user = user;
      a_runset->requests = place;
    }
}

/* Remove a request for the runset NAME by USER 
 * 
 * Nothing happens if there was no request by this user.
 */
void
remove_request (const char *runset, uid_t user)
{
  struct runset *a_runset;
  struct request *place;

  a_runset = find_runset (runset);

  if (a_runset && a_runset->requests)
    {
      place = a_runset->requests;

      if (place->user == user)
	{
	  a_runset->requests = place->next;
	  free (place);
	}
      else
	{
	  while (place->next && place->next->user != user)
	    place = place->next;

	  if (place->next)
	    {
	      struct request *work;

	      work = place->next;
	      place->next = work->next;
	      free (work);
	    }
	}
    }
}

/* Recursive helper program to destroy a chain of requests REQS.
 * 
 * It is used by cancel_requests(), cancel_all(), and update().
 */
static void
help_remove_all_requests (struct request *reqs)
{
  if (reqs)
    {
      help_remove_all_requests (reqs->next);
      free (reqs);
    }
}

/* Cancel all requests for a given runset
 * 
 * This supports a superuser command to deactivate a runset willy-nilly, 
 * without having to worry about undoing each request. 
 */
void
cancel_requests (const char *name)
{
  struct runset *a_runset;

  a_runset = find_runset (name);

  if (a_runset)
    {
      help_remove_all_requests (a_runset->requests);
      a_runset->requests = 0;

      /* Use SIGINT, not SIGTERM.  This skips tidy actions */
      if (a_runset->process)
	{
	  kill (a_runset->process, SIGINT);
	  a_runset->flag = 1;
	}
    }
}

/* Recursive helper procedure for update */
static struct runset *
update_help (struct runset *place)
{
  if (place)
    {
      if (place->requests)
	{
	  if (!place->process)
	    {
	      if (place->flag
		  || !(place->process = launch_runset (place->name)))
		{
		  syslog (LOG_WARNING,
			  "Runset `%s' failed - cancelling requests",
			  place->name);
		  help_remove_all_requests (place->requests);
		  place->requests = 0;
		}
	    }
	}

      /* This is a seperate if, not an else, because events above can
       * remove all requests 
       */
      if (!place->requests)
	{
	  if (place->process)
	    {
	      if (!place->flag)
		{
		  place->flag = 1;
		  kill (place->process, SIGTERM);
		}
	    }
	  else
	    {
	      struct runset *temp;

	      temp = update_help (place->next);
	      free (place);
	      return temp;
	    }
	}
      place->next = update_help (place->next);
    }
  return place;
}

/* Update. This procedure walks our list of runsets, and 
 * 
 * A) Launches runsets that are requested by one or more users and haven't
 *    been launched before.
 *
 * B) SIGTERM runset processes that are no longer requested
 *
 * C) Cancel all requests for runsets that exited early.
 * 
 * D) Clean runsets with no process or request (including as a result of C)
 *    out of the list 
 *
 * The function returns nonzero if there are any runsets active. This 
 * indicates when it is not safe to exit.
 * 
 */
int
update (void)
{
  runsets = update_help (runsets);

  return !!runsets;
}

/* Cancel all requests for all runsets. This is done in response to 
 * termination signals.
 */
void
cancel_all (void)
{
  struct runset *place;

  place = runsets;

  while (place)
    {
      help_remove_all_requests (place->requests);
      place->requests = 0;
      place = place->next;
    }
}


/* Change list structure to reflect the exit of a runset process PROCESS
 */
void
reap (pid_t process)
{
  struct runset *place;

  place = runsets;

  while (place)
    {
      if (place->process == process)
	{
	  place->process = 0;
	  place->flag = 1;
	  return;
	}
      place = place->next;
    }
}

/* Helper procedure for dumping all requests for a runset.
 */
static void
dump_requests_internal (FILE * stream, struct runset *a_runset)
{
  struct request *place;

  fprintf (stream, "%s:", a_runset->name);

  place = a_runset->requests;

  while (place)
    {
      fprintf (stream, " %d", (int) place->user);
      place = place->next;
    }

  fprintf (stream, "\n");
}

/* Output a list of all requests standing for runset NAME, to STREAM.
 * 
 * If we have no structure representing that runset (because nobody 
 * requested it since the last update()), output a line with no requests.
 */
void
dump_requests (FILE * stream, const char *name)
{
  struct runset *a_runset;

  a_runset = find_runset (name);
  if (a_runset)
    dump_requests_internal (stream, a_runset);
  else
    fprintf (stream, "%s:\n", name);
}

/* Output a list of all requests for every runset on our list.
 * 
 * (this does not list all the ones possible, since we soon forget any runset
 * that is not active.)
 */
void
dump_all (FILE * stream)
{
  struct runset *place;

  place = runsets;

  while (place)
    {
      dump_requests_internal (stream, place);
      place = place->next;
    }
}
