/*  Gringotts - a small utility to safe-keep sensitive data
 *  (c) 2002, Germano Rizzo <mano@pluto.linux.it>
 *
 *  grg_list.c - widget used to display the list of entries
 *  Author: Germano Rizzo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include "gringotts.h"
#include "grg_entries.h"
#include "grg_defs.h"

enum {
    COL_INDEX,
    COL_ID,
    NUM_COL
};

GtkWidget *treeview;
GtkTreeModel *model;

/**
 * create_model:
 *
 * Creates the model for the GTK list widget
 *
 * Returns: a GtkTreeModel with the desired model
 */
GtkTreeModel *create_model()
{
    gint i = 0, pos = grg_entries_position(), max = grg_entries_n_el();
    GtkListStore *store;
    GtkTreeIter iter;

    store = gtk_list_store_new(NUM_COL, G_TYPE_UINT, G_TYPE_STRING);

    grg_entries_first();
    for (i = 0; i < max; i++, grg_entries_next()) {
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter,
			   COL_INDEX, i + 1,
			   COL_ID, grg_entries_get_ID(), -1);
    }

    grg_entries_nth(pos);

    return GTK_TREE_MODEL(store);
}

/**
 * add_columns:
 * @treeview: the GtkTreeView to add the column to
 *
 * Adds a column to a GtkTreeView.
 */
void add_columns(GtkTreeView * treeview)
{
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;

    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes(_(" Index "),
						      renderer,
						      "text",
						      COL_INDEX, NULL);
    gtk_tree_view_append_column(treeview, column);

    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes(_(" Title "),
						      renderer,
						      "text",
						      COL_ID, NULL);
    gtk_tree_view_append_column(treeview, column);
}

/**
 * get_sel_row_num:
 *
 * Gets the number of the selected row
 *
 * returns: a gint with the number
 */
gint get_sel_row_num()
{
    gint ret;

    GtkTreeIter i;
    gtk_tree_selection_get_selected(gtk_tree_view_get_selection
				    (GTK_TREE_VIEW(treeview)), NULL, &i);

    gtk_tree_model_get(model, &i, 0, &ret, -1);

    return ret - 1;
}

/**
 * move_u:
 *
 * Shifts a row upwards
 */
void move_u()
{
    gint pos = get_sel_row_num(), old = grg_entries_position();
    GtkTreePath *path = gtk_tree_path_new();

    if (pos == 0)
	return;
    grg_entries_nth(get_sel_row_num());
    grg_entries_raise();
    model = create_model();
    gtk_tree_view_set_model(GTK_TREE_VIEW(treeview), model);
    gtk_tree_path_append_index(path, pos - 1);
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
    gtk_tree_path_free(path);
    grg_entries_nth(old);
	update_saveable(GRG_SAVE_ACTIVE);
}

/**
 * move_d:
 *
 * Shifts a row downwards
 */
void move_d()
{
    gint pos = get_sel_row_num(), old = grg_entries_position();
    GtkTreePath *path = gtk_tree_path_new();

    if (pos == grg_entries_n_el() - 1)
	return;
    grg_entries_nth(get_sel_row_num());
    grg_entries_sink();
    model = create_model();
    gtk_tree_view_set_model(GTK_TREE_VIEW(treeview), model);
    gtk_tree_path_append_index(path, pos + 1);
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);
    gtk_tree_path_free(path);
    grg_entries_nth(old);
	update_saveable(GRG_SAVE_ACTIVE);
}

/**
 * grg_list_run:
 *
 * Creates, manages and displays a dialog with the entry list
 *
 * Returns: a gboolean with GRG_TRUE if the user pressed the OK button
 */
gboolean grg_list_run()
{
    GtkWidget *dialog, *sw, *tbar, *hbox, *bup, *bdown;

    guint res;
    gboolean ret = GRG_FALSE;

    dialog =
	gtk_dialog_new_with_buttons(_("Index of entries"), NULL,
				    GTK_DIALOG_MODAL, GTK_STOCK_OK,
				    GTK_RESPONSE_ACCEPT, GTK_STOCK_CANCEL,
				    GTK_RESPONSE_REJECT, NULL);
    gtk_container_set_border_width(GTK_CONTAINER(dialog), GRG_PAD);
    gtk_box_set_spacing(GTK_BOX(GTK_DIALOG(dialog)->vbox), GRG_PAD);

    sw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
					GTK_SHADOW_ETCHED_IN);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
				   GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

    model = create_model();

    treeview = gtk_tree_view_new_with_model(model);
    gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), TRUE);
    gtk_tree_view_set_search_column(GTK_TREE_VIEW(treeview), COL_ID);
    g_object_unref(G_OBJECT(model));

    gtk_container_add(GTK_CONTAINER(sw), treeview);

    add_columns(GTK_TREE_VIEW(treeview));

    tbar = gtk_toolbar_new();
    gtk_toolbar_set_orientation(GTK_TOOLBAR(tbar),
				GTK_ORIENTATION_VERTICAL);
    gtk_toolbar_set_icon_size(GTK_TOOLBAR(tbar),
			      GTK_ICON_SIZE_LARGE_TOOLBAR);
    bup = gtk_toolbar_insert_stock(GTK_TOOLBAR(tbar),
				   GTK_STOCK_GO_UP,
				   _("Move up"), "", move_u, NULL, -1);
    bdown = gtk_toolbar_insert_stock(GTK_TOOLBAR(tbar),
				     GTK_STOCK_GO_DOWN,
				     _("Move down"), "", move_d, NULL, -1);

    hbox = gtk_hbox_new(FALSE, GRG_PAD);
    gtk_box_set_homogeneous(GTK_BOX(hbox), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), tbar, FALSE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), sw, TRUE, TRUE, 0);

    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, TRUE, TRUE,
		       0);
    gtk_window_set_default_size(GTK_WINDOW(dialog), 180, 250);
    gtk_widget_show_all(dialog);
    res = gtk_dialog_run(GTK_DIALOG(dialog));

    if (res == GTK_RESPONSE_ACCEPT) {
	grg_entries_nth(get_sel_row_num());
	ret = GRG_TRUE;
    }

    gtk_widget_destroy(dialog);

    return ret;
}
