/*  Gringotts - a small utility to safe-keep sensitive data
 *  (c) 2002, Germano Rizzo <mano@pluto.linux.it>
 *
 *  grg_utils.c - utility functions for gringotts
 *  Author: Germano Rizzo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "grg_defs.h"

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <time.h>

// Basically, the following 2 functions are the adaptation of string.h
// functions for byte sequences. Strings are byte sequence that are
// \0-terminated; in this case, this is dangerous, because a \0 can
// occour _inside_ the very sequence. Of course, there's no way to tell
// how long the sequence is, other than "remembering" it, and passing it
// to the functions.

FILE *fp = NULL;

/**
 * grg_strdup
 * @src: the source
 * @len: its length
 *
 * Duplicates a byte sequence into a new one
 *
 * Returns: a newly allocated (to free afterward) byte sequence
 */
inline unsigned char *
grg_strdup (unsigned char *src, unsigned long len)
{
	unsigned char *ret;

	ret = (unsigned char *) malloc (len * sizeof (unsigned char));
	memcpy (ret, src, len);

	return ret;
}

/**
 * grg_strconcat:
 * @src1: the first byte sequence to concatenate
 * @src2: the second byte sequence to concatenate
 * @src3: the third byte sequence to concatenate
 * @len1: the length of the first sequence
 * @len2: the second of the first sequence
 * @len3: the third of the first sequence
 *
 * Concatenates three byte sequences
 *
 * Returns: a newly allocated (to free afterward) byte sequence
 */
inline unsigned char *
grg_strconcat (unsigned char *src1,
	       unsigned char *src2,
	       unsigned char *src3,
	       unsigned long len1, unsigned long len2, unsigned long len3)
{
	unsigned char *ret, *tmp;

	ret = (unsigned char *) malloc ((len1 + len2 + len3) *
					sizeof (unsigned char));
	tmp = ret;
	memcpy (tmp, src1, len1);
	tmp += len1 * sizeof (unsigned char);
	memcpy (tmp, src2, len2);
	tmp += len2 * sizeof (unsigned char);
	memcpy (tmp, src3, len3);

	return ret;
}

/**
 * grg_long2char:
 * @seed: the long to convert
 *
 * Converts a long into four bytes
 *
 * Returns: a newly allocated 4-bytes sequence, to free after use
 */
unsigned char *
grg_long2char (unsigned long seed)
{
	unsigned char *ret;
	int i;

	ret = (unsigned char *) malloc (4 * sizeof (unsigned char));

	for (i = 3; i >= 0; i--, seed >>= 8)
		ret[i] = seed & 0x0ff;

	return ret;
}

/**
 * grg_char2long:
 * @seed: the 4-char sequence to convert
 *
 * Reverts grg_long2char(), converting back into a long
 *
 * Returns: a long
 */
unsigned long
grg_char2long (unsigned char *seed)
{
	unsigned long ret = 0;
	int i;

	for (i = 3; i >= 0; i--)
		ret |= seed[i] << (3 - i) * 8;

	return ret;
}

/**
 * grg_randomize:
 *
 * Initialize the random subsystem
 */
void
grg_randomize ()
{
	int rnd;

	fp = fopen ("/dev/random", "r");

	if (fp == NULL)
	{
		unsigned int i;

		srand (time (NULL));
		for (i = 0; i < getpid (); i++)
			rnd = rand ();
	}
}

/**
 * grg_derandomize:
 *
 * Finalize the random subsystem
 */
void
grg_derandomize ()
{
	if (fp != NULL)
		fclose (fp);
}

/**
 * grg_rnd_seq:
 * @size: the size of the sequence to generate
 *
 * Generates a random sequence of bytes.
 *
 * Returns: a newly-allocated byte sequence
 */
unsigned char *
grg_rnd_seq (unsigned int size)
{
	unsigned char ret[size];

	if (fp != NULL)
	{
		fread (&ret, 1, size, fp);
		return grg_strdup (ret, size);
	}
	else
	{
		unsigned int i;

		for (i = 0; i < size; i++)
			ret[i] = 1 +
				(unsigned char) (255.0 * rand () /
						 (RAND_MAX + 1.0));
		return grg_strdup (ret, size);
	}
}

/**
 * grg_rnd_chr:
 *
 * Returns a random byte.
 *
 * Returns: a random byte.
 */
unsigned char
grg_rnd_chr ()
{
	if (fp != NULL)
		return fgetc(fp);
	else
		return (unsigned int) (255.0 * rand () / (RAND_MAX + 1.0));
}

/**
 * grg_free:
 * @pntr: pointer to the memory to free
 * @dim: length of the sequence; if -1 it must be NULL-terminated
 *
 * Frees a sequence of bytes, overwriting it with random data
 */
void
grg_free (unsigned char *pntr, long dim)
{
	guint i = 0;

	if (pntr == NULL)
		return;

#ifdef ENABLE_SAFETY_SLOWDOWNS
	if (dim < 0)
		while (pntr[i] != 0)
			pntr[i++] = grg_rnd_chr ();
	else
		for (i = 0; i < dim; i++)
			pntr[i] = grg_rnd_chr ();
#endif

	free (pntr);
}

/**
 * grg_string_endswith:
 * @haystack: the string to search into
 * @needle: the string to search
 *
 * Checks if a NULL-terminated string ands with another one
 *
 * Returns: 0 if false, 1 if true
 */
unsigned int
grg_string_endswith (unsigned char *haystack, unsigned char *needle)
{
	unsigned int len1 = strlen (haystack), len2 = strlen (needle), i;

	if (len2 > len1)
		return 0;

	for (i = 1; i <= len2; i++)
		if (haystack[len1 - i] != needle[len2 - i])
			return 0;

	return 1;
}
