/*  Gringotts - a small utility to safe-keep sensitive data
 *  (c) 2002, Germano Rizzo <mano@pluto.linux.it>
 *
 *  gringotts.c - main program and interface
 *  Authors: Germano Rizzo, Nicholas Pouillon
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#define __USE_GNU		//to use the global variable environ in stdlib.h
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <regex.h>
#include <stdio.h>

#include <gtk/gtk.h>

#include "grg_entries.h"
#include "grg_widgets.h"
#include "grg_list.h"
#include "grg_crypt.h"
#include "grg_defs.h"
#include "grg_menus.h"
#include "grg_print.h"
#include "grg_utils.h"

/* - some menu buttons are never deactivated (i.e. Quit), so their widgets aren't really needed
 * - remember to remove the "not yet implemented" method (nyi()) when not necessary anymore
 */

// general
GtkWidget *title, *win1, *edit, *lbl;

// toolbar Navigation
GtkWidget *bfirst, *bback, *bfor, *blast, *bind;

// others
GtkWidget *btitle;

// main toolbar 
GtkWidget *tnew, *topen, *tsave, *tclose, *tquit;
GtkWidget *tadd, *trem, *tcut, *tcopy, *tpast, *tpref;

gchar *pwd = NULL, *grgfile = NULL, *caption = NULL;
gboolean started = FALSE, created = FALSE;
GtkClipboard *clip = NULL;

/**
 * nyi:
 *
 * Displays a "not yet implemented" message box
 */
void
nyi ()
{
	grg_msg (_("Sorry, this function hasn't been implemented yet"),
		 GTK_MESSAGE_ERROR, win1);
}


/**
 * about:
 *
 * Displays the `About' box.
 */
void
about ()
{
	gchar *info =
		g_strconcat (GRG_CAP_NAME, _(" version "), GRG_VERSION, "\n",
			     _
			     ("(c) 2002, Germano Rizzo <mano@pluto.linux.it> \n"
			      "Authors:\n"
			      "    Germano Rizzo <mano@pluto.linux.it>\n"
			      "    Nicolas Pouillon <nipo@ssji.net>\n"
			      "\n"
			      "Gringotts is a small but (hopely ;) useful utility that stores "
			      "sensitive data (passwords, credit card numbers, girlfriends' "
			      "addresses) in an organized and most of all very secure form.\n"
			      "It uses libmcrypt and libmhash to provide a strong level of "
			      "encryption, just trying to be as trustworthy as possible.\n\n"
			      "This program is released under the GNU GPL, v.2 or later\n"
			      "See COPYING or go to http://www.gnu.org/copyleft/gpl.html\n\n"
			      "libmcrypt is (c) 1998,1999 Nikos Mavroyanopoulos\n"
			      "libmhash is (c) 2001 Nikos Mavroyanopoulos"),
			     NULL);

	grg_msg (info, GTK_MESSAGE_INFO, win1);
	g_free (info);
}

/**
 * quit:
 *
 * Exits the main application cycle, releasing the resources.
 */
void
quit ()
{
	if (pwd != NULL)
		grg_free (pwd, -1);
	if (grgfile != NULL)
		g_free (grgfile);
	if (caption != NULL)
		g_free (caption);
	gtk_main_quit ();
	grg_derandomize ();
	exit (0);
}

/**
 * update_saveable:
 * @mode: the operation to perform:
 *        GRG_SAVE_ACTIVE -> activates all the save widgets.
 *        GRG_SAVE_INACTIVE -> deactivates all the save widgets.
 *        GRG_SAVE_QUERY -> query for saveability state.
 *
 * Performs an action about saveability.
 *
 * Returns: if GRG_SAVE_QUERY, the saveability state.
 */
gboolean
update_saveable (gint mode)
{
	static gboolean saveable = FALSE;

	if (mode == GRG_SAVE_ACTIVE)
	{
		if (started && grgfile)
		{
			caption = g_strconcat (grgfile, " *", NULL);
			gtk_window_set_title (GTK_WINDOW (win1), caption);
		}
		if (saveable == FALSE)
		{
			saveable = TRUE;
			gtk_widget_set_sensitive (bsave, TRUE);
			gtk_widget_set_sensitive (tsave, TRUE);
			if ((grgfile != NULL)
			    && (strcmp (grgfile, _("New file")) != 0))
				gtk_widget_set_sensitive (brev, TRUE);
		}
		else
			return saveable;	//don't care what it returns, actually
	}

	if (mode == GRG_SAVE_INACTIVE)
	{
		if (started && grgfile)
		{
			caption = g_strdup (grgfile);
			gtk_window_set_title (GTK_WINDOW (win1), caption);
		}
		if (saveable == TRUE)
		{
			saveable = FALSE;
			gtk_widget_set_sensitive (bsave, FALSE);
			gtk_widget_set_sensitive (tsave, FALSE);
			gtk_widget_set_sensitive (brev, FALSE);
		}
		else
			return saveable;	//don't care what it returns, actually
	}
	//let's assume mode is GRG_SAVE_QUERY
	return saveable;
}

/**
 * meta_saveable:
 * @data: unused
 * @user_data: a gpointer to the mode to pass to update_saveable().
 *
 * Controls if update_saveable() can be called, and if yes call it.
 */
void
meta_saveable (gpointer data, gpointer user_data)
{
	if (started)
		update_saveable (GPOINTER_TO_INT (user_data));
}

/**
 * grg_idle_text:
 *
 * Displays a text when no file is opened.
 */
void
grg_idle_text ()
{
	GtkTextBuffer *gtb = gtk_text_view_get_buffer (GTK_TEXT_VIEW (edit));
	gtk_text_buffer_set_text (gtb,
				  " " /*put the splash text here */ , -1);
}

/**
 * update:
 *
 * Updates various widgets (title of the window, button sensitivity...)
 */
void
update ()
{
	static gchar *tlbl;

	if (grg_entries_is_empty ())
	{
		gtk_widget_set_sensitive (btitle, FALSE);
		gtk_widget_set_sensitive (bind, FALSE);
		gtk_widget_set_sensitive (bsas, FALSE);
		gtk_widget_set_sensitive (badd, FALSE);
		gtk_widget_set_sensitive (tadd, FALSE);
		gtk_widget_set_sensitive (bmind, FALSE);
		gtk_widget_set_sensitive (bpwd, FALSE);
		gtk_widget_set_sensitive (tclose, FALSE);
		gtk_widget_set_sensitive (bclose, FALSE);
		gtk_widget_set_sensitive (edit, FALSE);
		gtk_widget_set_sensitive (bcut, FALSE);
		gtk_widget_set_sensitive (tcut, FALSE);
		gtk_widget_set_sensitive (bcop, FALSE);
		gtk_widget_set_sensitive (tcopy, FALSE);
		gtk_widget_set_sensitive (bpaste, FALSE);
		gtk_widget_set_sensitive (tpast, FALSE);
		gtk_widget_set_sensitive (bpwd, FALSE);
		gtk_label_set_text (GTK_LABEL (title),
				    GRG_CAP_NAME " " GRG_VERSION);
		gtk_text_view_set_buffer (GTK_TEXT_VIEW (edit),
					  gtk_text_buffer_new (NULL));
		grg_idle_text ();
	}
	else
	{
		gtk_widget_set_sensitive (btitle, TRUE);
		gtk_widget_set_sensitive (bind, TRUE);
		gtk_widget_set_sensitive (bmind, TRUE);
		gtk_widget_set_sensitive (badd, TRUE);
		gtk_widget_set_sensitive (tadd, TRUE);
		gtk_widget_set_sensitive (bsas, TRUE);
		gtk_widget_set_sensitive (bpwd, TRUE);
		gtk_widget_set_sensitive (bclose, TRUE);
		gtk_widget_set_sensitive (tclose, TRUE);
		gtk_widget_set_sensitive (edit, TRUE);
		gtk_widget_set_sensitive (bcut, TRUE);
		gtk_widget_set_sensitive (tcut, TRUE);
		gtk_widget_set_sensitive (bcop, TRUE);
		gtk_widget_set_sensitive (tcopy, TRUE);
		gtk_widget_set_sensitive (bpaste, TRUE);
		gtk_widget_set_sensitive (tpast, TRUE);
		gtk_widget_set_sensitive (bpwd, TRUE);
		gtk_label_set_text (GTK_LABEL (title), grg_entries_get_ID ());
		gtk_text_view_set_buffer (GTK_TEXT_VIEW (edit),
					  grg_entries_get_Body ());
	}

	if (grg_entries_is_last ())
	{
		gtk_widget_set_sensitive (bfor, FALSE);
		gtk_widget_set_sensitive (blast, FALSE);
		gtk_widget_set_sensitive (bmfor, FALSE);
		gtk_widget_set_sensitive (bmlast, FALSE);
	}
	else
	{
		gtk_widget_set_sensitive (bfor, TRUE);
		gtk_widget_set_sensitive (blast, TRUE);
		gtk_widget_set_sensitive (bmfor, TRUE);
		gtk_widget_set_sensitive (bmlast, TRUE);
	}

	if (grg_entries_is_first ())
	{
		gtk_widget_set_sensitive (bback, FALSE);
		gtk_widget_set_sensitive (bfirst, FALSE);
		gtk_widget_set_sensitive (bmback, FALSE);
		gtk_widget_set_sensitive (bmfirst, FALSE);
	}
	else
	{
		gtk_widget_set_sensitive (bback, TRUE);
		gtk_widget_set_sensitive (bfirst, TRUE);
		gtk_widget_set_sensitive (bmback, TRUE);
		gtk_widget_set_sensitive (bmfirst, TRUE);
	}

	if (grg_entries_n_el () <= 1)
	{
		gtk_widget_set_sensitive (brem, FALSE);
		gtk_widget_set_sensitive (trem, FALSE);
	}
	else
	{
		gtk_widget_set_sensitive (brem, TRUE);
		gtk_widget_set_sensitive (trem, TRUE);
	}
	tlbl = g_strdup_printf ("%d/%d", grg_entries_position () + 1,
				grg_entries_n_el ());
	gtk_label_set_text (GTK_LABEL (lbl), tlbl);
	gtk_window_set_title (GTK_WINDOW (win1), caption);
	g_free (tlbl);
}

void save();

/**
 * file_close:
 *
 * Closes the opened document.
 */
void
file_close ()
{
	if (grg_entries_is_empty ())
		return;

	if (update_saveable (GRG_SAVE_QUERY))
		if (grg_ask_dialog
		    (_("Save?"),
		     _
		     ("Some changes have not been saved.\nDo you wish to save them now?"),
		     win1))
			save();
	grg_entries_free ();
	grg_free (pwd, -1);
	pwd = NULL;
	g_free (grgfile);
	grgfile = NULL;
	g_free (caption);
	caption = g_strconcat (GRG_CAP_NAME, " ", GRG_VERSION, NULL);
	update_saveable (GRG_SAVE_INACTIVE);
	update ();
}

/**
 * meta_quit:
 *
 * Asks for save if not saved, and then exits.
 */
void
meta_quit ()
{
	file_close ();
	quit ();
}

/**
 * revert:
 *
 * Reverts to last saved version.
 */
void
revert ()
{
	gint err;

	if (update_saveable (GRG_SAVE_QUERY) &&
	    grg_ask_dialog
	    (_("Confirm"),
	     _("You'll lose all the changes from\nlast save! Are you sure?"),
	     win1))
	{
		err = grg_entries_load (grgfile, pwd);

		switch (err)
		{
		case GRG_OK:
		{
			update_saveable (GRG_SAVE_INACTIVE);
			update ();
			break;
		}

		case GRG_READ_MAGIC_ERR:
		case GRG_READ_CRC_ERR:
		case GRG_READ_PWD_ERR:
		case GRG_READ_INVALID_CHARSET_ERR:
		{
			grg_msg (_
				 ("The file appears to be corrupted! Oh, God!"),
				 GTK_MESSAGE_ERROR, win1);
			break;
		}

		case GRG_READ_TOO_BIG_ERR:
		{
			grg_msg (_
				 ("File is too big. Try to increase max-comp-size in ./configure"),
				 GTK_MESSAGE_ERROR, win1);
			break;
		}

		case GRG_READ_FILE_ERR:
		{
			grg_msg (_("Uh-oh! I can't read from the file!"),
				 GTK_MESSAGE_ERROR, win1);
			break;
		}

		case GRG_READ_ENC_INIT_ERR:
		{
			grg_msg (_
				 ("Problem with libmcrypt, probably a faulty installation"),
				 GTK_MESSAGE_ERROR, win1);
			break;
		}
		
		default:
			g_assert_not_reached ();
			break;
		}
	}
}

/**
 * load_file:
 * @filename: a filename.
 *
 * Loads a gringotts file.
 */
void
load_file (gchar * filename)
{
	gchar *tmppwd;
	gint err;
	guchar *res;
	gboolean exit;
	struct stat buf1, buf2;

	if (!filename || !*filename)
		return;

	//if this and the opened one are the very same file, fall back on revert()
	if (grgfile != NULL &&
	    strcmp (grgfile, _("New file")) != 0 &&
	    stat (grgfile, &buf1) == 0 &&
	    stat (filename, &buf2) == 0 &&
	    buf1.st_dev == buf2.st_dev && buf1.st_ino == buf2.st_ino)
	{
		revert ();
		return;
	}

	if (!g_file_test (filename, G_FILE_TEST_EXISTS))
	{
		grg_msg (_("The selected file doesn't exists"),
			 GTK_MESSAGE_ERROR, win1);
		return;
	}

	if (g_file_test (filename, G_FILE_TEST_IS_DIR))
	{
		grg_msg (_("You've selected a directory"), GTK_MESSAGE_ERROR,
			 win1);
		return;
	}

	file_close ();

	err = grg_load_wrapper (NULL, NULL, filename, 1);

	switch (err)
	{
	case GRG_OK:
		break;

	case GRG_READ_MAGIC_ERR:
	{
		grg_msg (_
			 ("This file doesn't seem to be a valid Gringotts one!"),
			 GTK_MESSAGE_ERROR, win1);
		return;
	}

	case GRG_READ_FILE_ERR:
	{
		grg_msg (_("Uh-oh! I can't read from the file!"),
			 GTK_MESSAGE_ERROR, win1);
		return;
	}

	case GRG_READ_CRC_ERR:
	{
		grg_msg (_("The file appears to be corrupted! Oh, God!"),
			 GTK_MESSAGE_ERROR, win1);
		return;
	}

	case GRG_READ_TOO_BIG_ERR:
	{
		grg_msg (_
			 ("File is too big. Try to increase max-comp-size in ./configure"),
			 GTK_MESSAGE_ERROR, win1);
		return;
	}
	
	default:
	{
		if (err < 0)
		{
			grg_msg (_
				 ("The file appears to be corrupted! Oh, God!"),
				 GTK_MESSAGE_ERROR, win1);
			return;
		}
	}
	}

	while (TRUE)
	{
		exit = FALSE;

		tmppwd = grg_input_dialog (_("Enter password..."),
					   _("Password for your data:"), NULL,
					   TRUE, win1);
		if (tmppwd == NULL)
			break;

		err = grg_load_wrapper (&res, tmppwd, filename, 0);

		switch (err)
		{
		case GRG_OK:
		{
			grg_entries_load_from_string (res);
			grg_free (res, -1);
			exit = TRUE;
		}
			break;

		case GRG_READ_PWD_ERR:
		{
			grg_msg (_("Wrong password! Re-enter it"),
				 GTK_MESSAGE_ERROR, win1);
		}
			break;

		case GRG_READ_ENC_INIT_ERR:
		{
			grg_msg (_
				 ("Problem with libmcrypt, probably a faulty installation"),
				 GTK_MESSAGE_ERROR, win1);
		}
			break;

		case GRG_READ_INVALID_CHARSET_ERR:
		{
			grg_msg (_
				 ("Saved data contain invalid UTF-8 chars"),
				 GTK_MESSAGE_ERROR, win1);
		}
			break;

			//just to be sure...
		default:
		{
			if (err < 0)
			{
				grg_msg (_
					 ("The file appears to be corrupted! Oh, God!"),
					 GTK_MESSAGE_ERROR, win1);
				grg_free (tmppwd, -1);
				return;
			}
		}
		}
		if (exit)
			break;
	}

	if (grgfile != NULL)
		g_free (grgfile);
	grgfile = g_strdup (filename);
	if (pwd != NULL)
		grg_free (pwd, -1);
	pwd = tmppwd;
	if (caption != NULL)
		g_free (caption);
	caption = g_strdup (grgfile);
	update ();
	update_saveable (GRG_SAVE_INACTIVE);
}

/**
 * load:
 * @data: unused
 * @fsel: a gpointer to the file selector.
 *
 * Loads a gringotts file.
 */
void
load (gpointer data, gpointer fsel)
{
	gchar *tmpfile;

	tmpfile =
		g_strdup (gtk_file_selection_get_filename
			  (GTK_FILE_SELECTION (fsel)));
	load_file (tmpfile);
	g_free (tmpfile);
}

/**
 * meta_load:
 *
 * Displays a file loading dialog, then calls load().
 */
void
meta_load ()
{
	GtkWidget *file_selector;
	file_selector = gtk_file_selection_new (_("Open..."));
	gtk_window_set_transient_for (GTK_WINDOW (file_selector),
				      GTK_WINDOW (win1));
	g_signal_connect (GTK_OBJECT
			  (GTK_FILE_SELECTION (file_selector)->ok_button),
			  "clicked", G_CALLBACK (load),
			  (gpointer) file_selector);
	g_signal_connect_swapped (GTK_OBJECT
				  (GTK_FILE_SELECTION (file_selector)->
				   ok_button), "clicked",
				  G_CALLBACK (gtk_widget_destroy),
				  (gpointer) file_selector);
	g_signal_connect_swapped (GTK_OBJECT
				  (GTK_FILE_SELECTION (file_selector)->
				   cancel_button), "clicked",
				  G_CALLBACK (gtk_widget_destroy),
				  (gpointer) file_selector);
	gtk_widget_show (file_selector);
}

/**
 * move_f:
 *
 * Displays the next entry.
 */
void
move_f ()
{
	grg_entries_next ();
	update ();
}

/**
 * move_b:
 *
 * Displays the previous entry.
 */
void
move_b ()
{
	grg_entries_prev ();
	update ();
}

/**
 * move_l:
 *
 * Displays the last entry.
 */
void
move_l ()
{
	grg_entries_last ();
	update ();
}

/**
 * move_1:
 *
 * Displays the first entry.
 */
void
move_1 ()
{
	grg_entries_first ();
	update ();
}

/**
 * do_new:
 *
 * Creates a new document.
 */
void
do_new ()
{
	gchar *tmppwd;

	file_close ();

	tmppwd = grg_new_pwd_dialog (win1);
	if (tmppwd == NULL)
	{
		grg_msg (_("You must enter a valid password!"),
			 GTK_MESSAGE_ERROR, win1);
		return;
	}

	if (pwd != NULL)
		grg_free (pwd, -1);

	if (grgfile != NULL)
	{
		g_free (grgfile);
	}
	grgfile = g_strdup (_("New file"));

	if (grg_entries_is_empty ())
		grg_entries_free ();

	pwd = g_strdup (tmppwd);
	grg_free (tmppwd, -1);

	grg_entries_append (_("My first page"));
	grg_entries_first ();
	update_saveable (GRG_SAVE_ACTIVE);
	update ();
}

/**
 * retitle:
 *
 * Gives the entry another title.
 */
void
retitle ()
{
	gchar *new = NULL;
	new = grg_input_dialog (_("New title..."),
				_("New title for this entry:"),
				grg_entries_get_ID (), FALSE, win1);
	if (new != NULL)
	{
		grg_entries_set_ID (new);
		grg_free (new, -1);
		update_saveable (GRG_SAVE_ACTIVE);
		update ();
	}
}

/**
 * save_as:
 * @data: unused
 * @fsel: a gpointer to the file selector.
 *
 * Saves a file with another name.
 */
void
save_as (gpointer data, gpointer fsel)
{
	gchar *tmpfile;
	gint err;

	tmpfile =
		g_strdup (gtk_file_selection_get_filename
			  (GTK_FILE_SELECTION (fsel)));

	if (!grg_string_endswith (tmpfile, ".grg"))
	{
		guchar *tmp;

		tmp = g_strconcat (tmpfile, ".grg");
		g_free (tmpfile);
		tmpfile = g_strdup (tmp);
		g_free (tmp);
	}

	if (g_file_test (tmpfile, G_FILE_TEST_EXISTS)
	    && !grg_ask_dialog (_("Overwrite?"),
				_
				("Do you want to overwrite the existing file?"),
				win1))
	{
		g_free (tmpfile);
		return;
	}

	err = grg_entries_save (tmpfile, pwd);

	switch (err)
	{
	case GRG_OK:
	{
		if (grgfile != NULL)
			g_free (grgfile);
		grgfile = g_strdup (tmpfile);
		g_free (tmpfile);
		update_saveable (GRG_SAVE_INACTIVE);
		break;
	}

	case GRG_WRITE_COMP_ERR:
	{
		grg_msg (_("Error in compression! Probably a zlib problem"),
			 GTK_MESSAGE_ERROR, win1);
		g_free (tmpfile);
		return;
	}

	case GRG_WRITE_FILE_ERR:
	{
		grg_msg (_("Uh-oh! I can't write to the file!"),
			 GTK_MESSAGE_ERROR, win1);
		g_free (tmpfile);
		return;
	}

	case GRG_WRITE_ENC_INIT_ERR:
	{
		grg_msg (_
			 ("Problem with libmcrypt, probably a faulty installation"),
			 GTK_MESSAGE_ERROR, win1);
		g_free (tmpfile);
		return;
	}

	case GRG_WRITE_TOO_BIG_ERR:
	{
		grg_msg (_
			 ("Too many data to write"), GTK_MESSAGE_ERROR, win1);
		g_free (tmpfile);
		return;
	}
	
	default:
		g_assert_not_reached ();
		break;
	}
}

/**
 * meta_save_as:
 *
 * Displays the Save As dialog, then calls save_as().
 */
void
meta_save_as ()
{
	GtkWidget *file_selector;
	file_selector = gtk_file_selection_new (_("Save as...."));
	gtk_window_set_transient_for (GTK_WINDOW (file_selector),
				      GTK_WINDOW (win1));
	g_signal_connect (GTK_OBJECT
			  (GTK_FILE_SELECTION (file_selector)->ok_button),
			  "clicked", G_CALLBACK (save_as),
			  (gpointer) file_selector);
	g_signal_connect_swapped (GTK_OBJECT
				  (GTK_FILE_SELECTION (file_selector)->
				   ok_button), "clicked",
				  G_CALLBACK (gtk_widget_destroy),
				  (gpointer) file_selector);
	g_signal_connect_swapped (GTK_OBJECT
				  (GTK_FILE_SELECTION (file_selector)->
				   cancel_button), "clicked",
				  G_CALLBACK (gtk_widget_destroy),
				  (gpointer) file_selector);
	gtk_widget_show (file_selector);
}

/**
 * save:
 *
 * Saves the current version of the opened document.
 */
void
save ()
{
	gint err;

	if ((grgfile == NULL) || (strcmp (grgfile, _("New file")) == 0))
	{
		meta_save_as ();
		return;
	}

	err = grg_entries_save (grgfile, pwd);

	switch (err)
	{
	case GRG_OK:
		break;

	case GRG_WRITE_COMP_ERR:
	{
		grg_msg (_("Error in compression! Probably a zlib problem"),
			 GTK_MESSAGE_ERROR, win1);
		return;
	}

	case GRG_WRITE_FILE_ERR:
	{
		grg_msg (_("Uh-oh! I can't write to the file!"),
			 GTK_MESSAGE_ERROR, win1);
		return;
	}

	case GRG_WRITE_ENC_INIT_ERR:
	{
		grg_msg (_
			 ("Problem with libmcrypt, probably a faulty installation"),
			 GTK_MESSAGE_ERROR, win1);
		return;
	}

	case GRG_WRITE_TOO_BIG_ERR:
	{
		grg_msg (_
			 ("Too many data to write"), GTK_MESSAGE_ERROR, win1);
		return;
	}
	
	default:
		g_assert_not_reached ();
		break;
	}

	update_saveable (GRG_SAVE_INACTIVE);
}

/**
 * insert:
 *
 * Appends a new entry.
 */
void
insert ()
{
	gchar *new = NULL;
	new = grg_input_dialog (_("Title..."), _("Title for the new entry:"),
				_("New"), FALSE, win1);
	if (new != NULL)
	{
		grg_entries_append (new);
		grg_free (new, -1);
		update_saveable (GRG_SAVE_ACTIVE);
		update ();
	}
}

/**
 * del:
 *
 * Deletes the current entry.
 */
void
del ()
{
	if (grg_ask_dialog
	    (_("Confirm..."),
	     _("Are you sure you want to remove this entry?"), win1))
	{
		grg_entries_remove ();
		update_saveable (GRG_SAVE_ACTIVE);
		update ();
	}
}

/**
 * chpwd:
 *
 * Calls the change password dialog.
 */
void
chpwd ()
{
	gchar *tmppwd;

	if (strcmp
	    (pwd,
	     grg_input_dialog (_("Enter password"), _("Enter old password:"),
			       NULL, TRUE, win1)) != 0)
	{
		grg_msg (_("Wrong password"), GTK_MESSAGE_ERROR, win1);
		return;
	}

	tmppwd = grg_new_pwd_dialog (win1);

	if (tmppwd != NULL)
	{
		grg_free (pwd, -1);
		pwd = g_strdup (tmppwd);
		grg_free (tmppwd, -1);
		grg_msg (_("Password successfully changed"), GTK_MESSAGE_INFO,
			 win1);
		update_saveable (GRG_SAVE_ACTIVE);
	}
}

/**
 * meta_list:
 *
 * Calls the list window.
 */
void
meta_list ()
{
	if (grg_list_run ())
		update ();
}

/**
 * cucopa:
 * @callback_data: unused callback param
 * @callback_action: action to perform (GRG_CUT, GRG_COPY, GRG_PASTE)
 *
 * basic CUT/COPY/PASTE clipboard operation.
 */
void
cucopa (gpointer callback_data, guint callback_action)
{
	GtkTextBuffer *gtb = grg_entries_get_Body ();

	if (gtb == NULL)
		return;

	switch (callback_action)
	{
	case GRG_CUT:
		gtk_text_buffer_cut_clipboard (gtb, clip, TRUE);
		return;
	
	case GRG_COPY:
		gtk_text_buffer_copy_clipboard (gtb, clip);
		return;
	
	case GRG_PASTE:
		gtk_text_buffer_paste_clipboard (gtb, clip, NULL, TRUE);
		return;
	
	default:
		g_assert_not_reached ();
		break;
	}
}

/**
 * grg_interface:
 *
 * Builds and "activates" the Gringotts main interface.
 */
void
grg_interface ()
{
	GtkWidget *vbox, *hbox, *scroll, *menu;
	GtkWidget *tbar_nav, *handle_nav, *tbar_main, *handle_main;
	PangoFontDescription *pfd;

	// window
	win1 = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	grg_window_set_icon (GTK_WINDOW (win1));

	caption = g_strconcat (GRG_CAP_NAME, " ", GRG_VERSION, NULL);

	gtk_container_set_border_width (GTK_CONTAINER (win1), GRG_PAD);
	g_signal_connect (G_OBJECT (win1), "destroy-event",
			  G_CALLBACK (meta_quit), NULL);
	g_signal_connect (G_OBJECT (win1), "delete-event",
			  G_CALLBACK (meta_quit), NULL);
	g_signal_connect (G_OBJECT (win1), "destroy", G_CALLBACK (meta_quit),
			  NULL);
	// the "notepad" widget
	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scroll),
					     GTK_SHADOW_ETCHED_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	edit = gtk_text_view_new ();
	gtk_widget_set_usize (scroll, 400, 300);
	gtk_container_add (GTK_CONTAINER (scroll), edit);
	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (edit), GTK_WRAP_WORD);

	// the title widget
	title = gtk_label_new ("");
	gtk_misc_set_alignment (GTK_MISC (title), 0.1, 0.5);
	btitle = gtk_button_new_with_mnemonic (_("_Edit..."));
	g_signal_connect (G_OBJECT (btitle), "clicked", G_CALLBACK (retitle),
			  NULL);
	pfd = pango_font_description_new ();
	pango_font_description_set_weight (pfd, PANGO_WEIGHT_BOLD);
	gtk_widget_modify_font (title, pfd);
	pango_font_description_free (pfd);
	lbl = gtk_label_new ("");

	// the "navigation" lateral toolbar
	tbar_nav = gtk_toolbar_new ();
	gtk_toolbar_set_style (GTK_TOOLBAR (tbar_nav), GTK_TOOLBAR_ICONS);
	gtk_toolbar_set_orientation (GTK_TOOLBAR (tbar_nav),
				     GTK_ORIENTATION_VERTICAL);
	gtk_toolbar_set_icon_size (GTK_TOOLBAR (tbar_nav),
				   GTK_ICON_SIZE_LARGE_TOOLBAR);
	handle_nav = gtk_handle_box_new ();
	gtk_handle_box_set_handle_position (GTK_HANDLE_BOX (handle_nav),
					    GTK_POS_TOP);
	gtk_container_add (GTK_CONTAINER (handle_nav), tbar_nav);
	blast = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_nav),
					  GTK_STOCK_GOTO_LAST,
					  _("Go to last entry"), "", move_l,
					  NULL, -1);
	bfor = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_nav),
					 GTK_STOCK_GO_FORWARD,
					 _("Go to next entry"), "", move_f,
					 NULL, -1);
	gtk_toolbar_append_space (GTK_TOOLBAR (tbar_nav));
	bind = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_nav),
					 GTK_STOCK_INDEX, _("View index"), "",
					 meta_list, NULL, -1);
	gtk_toolbar_append_space (GTK_TOOLBAR (tbar_nav));
	bback = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_nav),
					  GTK_STOCK_GO_BACK,
					  _("Go to previous entry"), "",
					  move_b, NULL, -1);
	bfirst = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_nav),
					   GTK_STOCK_GOTO_FIRST,
					   _("Go to first entry"), "", move_1,
					   NULL, -1);

	//the "main" toolbar
	tbar_main = gtk_toolbar_new ();
	gtk_toolbar_set_icon_size (GTK_TOOLBAR (tbar_main),
				   GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_toolbar_set_style (GTK_TOOLBAR (tbar_main), GTK_TOOLBAR_ICONS);
	handle_main = gtk_handle_box_new ();
	gtk_container_add (GTK_CONTAINER (handle_main), tbar_main);

	tnew = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_main),
					 GTK_STOCK_NEW,
					 _("New document"), "", do_new,
					 NULL, -1);

	gtk_toolbar_append_space (GTK_TOOLBAR (tbar_main));

	topen = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_main),
					  GTK_STOCK_OPEN,
					  _("Open document"), "", meta_load,
					  NULL, -1);

	tsave = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_main),
					  GTK_STOCK_SAVE,
					  _("Save document"), "", save,
					  NULL, -1);

	tclose = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_main),
					   GTK_STOCK_CLOSE,
					   _("Close document"), "",
					   file_close, NULL, -1);

	gtk_toolbar_append_space (GTK_TOOLBAR (tbar_main));

	tcut = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_main),
					 GTK_STOCK_CUT,
					 _("Cut selection"), "",
					 (GtkSignalFunc) cucopa,
					 GINT_TO_POINTER (GRG_CUT), -1);

	tcopy = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_main),
					  GTK_STOCK_COPY,
					  _("Copy selection"), "",
					  (GtkSignalFunc) cucopa,
					  GINT_TO_POINTER (GRG_COPY), -1);

	tpast = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_main),
					  GTK_STOCK_PASTE,
					  _("Paste selection"), "",
					  (GtkSignalFunc) cucopa,
					  GINT_TO_POINTER (GRG_PASTE), -1);

	gtk_toolbar_append_space (GTK_TOOLBAR (tbar_main));

	tadd = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_main),
					 GTK_STOCK_ADD,
					 _("Add an entry"), "", insert,
					 NULL, -1);

	trem = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_main),
					 GTK_STOCK_REMOVE,
					 _("Remove this entry"), "", del,
					 NULL, -1);

	gtk_toolbar_append_space (GTK_TOOLBAR (tbar_main));

	tpref = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_main),
					  GTK_STOCK_PREFERENCES,
					  _("Preferences"),
					  "", nyi, NULL, -1);

	gtk_toolbar_append_space (GTK_TOOLBAR (tbar_main));

	tquit = gtk_toolbar_insert_stock (GTK_TOOLBAR (tbar_main),
					  GTK_STOCK_QUIT,
					  _("Quit Gringotts"), "",
					  meta_quit, NULL, -1);

	//by Nipo: using a GTKtable would have grown/shrunk 
	//the toolbars along with the window
	vbox = gtk_vbox_new (FALSE, 1);
	gtk_widget_show (vbox);

	menu = grg_menu_create (win1);
	gtk_box_pack_start (GTK_BOX (vbox), menu, FALSE, TRUE, 1);

	gtk_box_pack_start (GTK_BOX (vbox), handle_main, FALSE, TRUE, 1);

	hbox = gtk_hbox_new (FALSE, 1);
	gtk_box_pack_start (GTK_BOX (hbox), btitle, FALSE, TRUE, 1);
	gtk_box_pack_start (GTK_BOX (hbox), title, TRUE, TRUE, 1);
	gtk_box_pack_start (GTK_BOX (hbox), lbl, FALSE, TRUE, 1);

	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 1);

	hbox = gtk_hbox_new (FALSE, 1);
	gtk_box_pack_start (GTK_BOX (hbox), handle_nav, FALSE, TRUE, 1);
	gtk_box_pack_start (GTK_BOX (hbox), scroll, TRUE, TRUE, 1);

	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 1);

	gtk_container_add (GTK_CONTAINER (win1), vbox);
	update_saveable (GRG_SAVE_ACTIVE);
	update_saveable (GRG_SAVE_INACTIVE);
	created = TRUE;
	update ();

	started = TRUE;
	gtk_widget_show_all (win1);
}

/**
 * main:
 * @argc: No need to tell... ;)
 * @argv: No need to tell... ;)
 *
 * No need to tell... ;)
 *
 * Returns: No need to tell... ;)
 */
gint
main (gint argc, gchar ** argv)
{
	gchar *out;
#ifdef ENABLE_SAFETY_SLOWDOWNS
	regex_t regexp1, regexp2;
	extern char **environ;
	gchar *lang, *display, *xauth;
	gchar ulimit;
	FILE *pipe;
#endif

	grg_randomize ();

#ifndef ALLOW_ROOT_USE
	//forbid root usage
	if (strcmp (g_get_user_name (), "root") == 0)
	{
		grg_print_err (_
			       ("For security reasons, you cannot run Gringotts as root. Try using --enable-root-use in ./configure"));
		return (1);
	}
#endif
	
	//check core dump generation
	pipe = popen("ulimit -Sc","r");
	ulimit = fgetc(pipe);
	pclose(pipe);
	
	if(ulimit != '0')
	{
		grg_print_err (_
			       ("Your shell is configured to produce core dumps. I cannot go on. Please run gringotts via its shell script."));
		return (1);
	}

	//extract needed environmental vars, validate, erase environment,
	//and re-set them (see Secure Programming HowTo, sect.4.2)

#ifdef ENABLE_SAFETY_SLOWDOWNS
	//extract
	lang = getenv ("LANG");
	display = getenv ("DISPLAY");
	xauth = getenv ("XAUTHORITY");
	//validate
	regcomp (&regexp1, "[[:alpha:]][[:alnum:]_,+@\\-\\.=]*", REG_EXTENDED);
	regcomp (&regexp2, "[[:digit:]{1,3}\\.[:digit:]{1,3}\\.[:digit:]{1,3}\\.[:digit:]{1,3}]?:[[:digit:]\\.]+", REG_EXTENDED);
	if (((lang != NULL) &&
	     (regexec (&regexp1, lang, 0, NULL, 0))) ||
	    ((xauth != NULL) &&
	     (!g_file_test (xauth, G_FILE_TEST_EXISTS))) ||
	    (regexec (&regexp2, display, 0, NULL, 0)))
	{
		grg_print_err (_
			       ("Invalid environment variables. They could have been manipulated."));
		return (1);
	}
	//erase
	environ = NULL;
	//re-set (warning: don't free() the g_strconcat'ed strings)
	if (lang != NULL)
		putenv (g_strconcat ("LANG=", lang, NULL));
	putenv (g_strconcat ("DISPLAY=", display, NULL));
	if (xauth != NULL)
		putenv (g_strconcat ("XAUTHORITY=", xauth, NULL));
#endif

	putenv ("MALLOC_CHECK_=2");
	setlocale (LC_ALL, "");
	bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	out = g_strdup_printf ("\n\
" GRG_CAP_NAME " %s " GRG_VERSION "\n\
%s\n\
\n\
%s\n\
%s\
\n\
\n", _("version"), _("(c) 2002, Germano Rizzo <mano@pluto.linux.it>"), _("released under GNU General Public License (GPL) v.2 or later"), _("See COPYING or go to http://www.gnu.org/copyleft/gpl.html"));
	grg_print (out);
	g_free (out);

	gtk_init (&argc, &argv);

	clip = gtk_clipboard_get (GDK_NONE);

	grg_interface ();

	//by Nipo
	if ((argc > 1) && (argv[1][0] != '-'))
	{
		if (g_file_test (argv[1], G_FILE_TEST_EXISTS))
			load_file (argv[1]);
		else
			grg_print_warn (_("Specified file does not exists"));
	}

	gtk_main ();

	return (0);
}
