/*!\file netinet/tcp_var.h
 * Internal TCP structures and statistics.
 */

/*
 * Copyright (c) 1982, 1986, 1993, 1994, 1995
 *      The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef __NETINET_TCP_VAR_H
#define __NETINET_TCP_VAR_H

#ifndef __SYS_MBUF_H
#include <sys/mbuf.h>
#endif

#ifndef __NETINET_TCP_TIMER_H
#include <netinet/tcp_time.h>
#endif

#include <netinet/tcpip.h>

/*
 * Kernel variables for tcp.
 */

#ifdef TCP_AUTO
  /* AUTO_RCV_HITHRESH flush reassembly queue, drop incoming packets
   */
  #define AUTO_RCV_HITHRESH   (u_long)(0.95 * NMBCLUSTERS)

  /* AUTO_RCV_LOWTHRESH drop incoming packets
   */
  #define AUTO_RCV_LOWTHRESH  (u_long)(0.9 * NMBCLUSTERS)

  /* AUTO_SND_THRESH reduce so_snd.sb_hiwat by acked
   */
  #define AUTO_SND_THRESH     (u_long)(0.50 * NMBCLUSTERS)
#endif

#if defined(TCP_SACK) || defined(TCP_FACK)
#include <netinet/tcp_scor.h>

struct sackblock {
       tcp_seq start;
       tcp_seq end;
     };
#endif

/*
 * Tcp control block, one per tcp; fields:
 */
struct tcpcb {
        struct  tcpiphdr *seg_next;     /* sequencing queue */
        struct  tcpiphdr *seg_prev;
  /* !! struct  ipqehead segq; */       /* sequencing queue */
        int     t_state;                /* state of this connection */
        int     t_timer[TCPT_NTIMERS];  /* tcp timers */
        int     t_rxtshift;             /* log(2) of rexmt exp. backoff */
        int     t_rxtcur;               /* current retransmit value */
#ifdef TCP_FACK
        short   t_padd;                 /* filler to preserve alignment */
#else
        short   t_dupacks;              /* consecutive dup acks recd */
#endif
        u_int   t_maxseg;               /* maximum segment size */
        u_int   t_maxopd;               /* mss plus options */
        int     t_force;                /* 1 if forcing out a byte */
        u_int   t_flags;
#define TF_ACKNOW       0x0001          /* ack peer immediately */
#define TF_DELACK       0x0002          /* ack, but try to delay it */
#define TF_NODELAY      0x0004          /* don't delay packets to coalesce */
#define TF_NOOPT        0x0008          /* don't use tcp options */
#define TF_SENTFIN      0x0010          /* have sent FIN */
#define TF_REQ_SCALE    0x0020          /* have/will request window scaling */
#define TF_RCVD_SCALE   0x0040          /* other side has requested scaling */
#define TF_REQ_TSTMP    0x0080          /* have/will request timestamps */
#define TF_RCVD_TSTMP   0x0100          /* a timestamp was received in SYN */
#define TF_SACK_PERMIT  0x0200          /* other side said I could SACK */
#define TF_NEEDSYN      0x0400          /* send SYN (implicit state) */
#define TF_NEEDFIN      0x0800          /* send FIN (implicit state) */
#define TF_NOPUSH       0x1000          /* don't push */
#define TF_REQ_CC       0x2000          /* have/will request CC */
#define TF_RCVD_CC      0x4000          /* a CC was received in SYN */
#define TF_SENDCCNEW    0x8000          /* send CCnew instead of CC in SYN */

        struct  tcpiphdr *t_template;   /* skeletal packet for transmit */
        struct  inpcb *t_inpcb;         /* back pointer to internet pcb */
/*
 * The following fields are used as in the protocol specification.
 * See RFC783, Dec. 1981, page 21.
 */
/* send sequence variables */
        tcp_seq snd_una;                /* send unacknowledged */
        tcp_seq snd_nxt;                /* send next */
        tcp_seq snd_up;                 /* send urgent pointer */
        tcp_seq snd_wl1;                /* window update seg seq number */
        tcp_seq snd_wl2;                /* window update seg ack number */
        tcp_seq iss;                    /* initial send sequence number */
        u_long  snd_wnd;                /* send window */
/* receive sequence variables */
        u_long  rcv_wnd;                /* receive window */
        tcp_seq rcv_nxt;                /* receive next */
        tcp_seq rcv_up;                 /* receive urgent pointer */
        tcp_seq irs;                    /* initial receive sequence number */
/*
 * Additional variables for this implementation.
 */
/* receive variables */
        tcp_seq rcv_adv;                /* advertised window */
/* retransmit variables */
        tcp_seq snd_max;                /* highest sequence number sent;
                                         * used to recognize retransmits
                                         */
/* congestion control (for slow start, source quench, retransmit after loss) */
        u_long  snd_cwnd;               /* congestion-controlled window */
        u_long  snd_ssthresh;           /* snd_cwnd size threshold for
                                         * for slow start exponential to
                                         * linear switch
                                         */
/*
 * transmit timing stuff.  See below for scale of srtt and rttvar.
 * "Variance" is actually smoothed difference.
 */
        u_int   t_idle;                 /* inactivity time */
        int     t_rtt;                  /* round trip time */
        tcp_seq t_rtseq;                /* sequence number being timed */
        int     t_srtt;                 /* smoothed round-trip time */
        int     t_rttvar;               /* variance in round-trip time */
        u_int   t_rttmin;               /* minimum rtt allowed */
        u_long  max_sndwnd;             /* largest window peer has offered */

/* out-of-band data */
        char    t_oobflags;             /* have some */
        char    t_iobc;                 /* input character */
#define TCPOOB_HAVEDATA 0x01
#define TCPOOB_HADDATA  0x02
        int     t_softerror;            /* possible error not yet reported */

/* RFC 1323 variables */
        u_char  snd_scale;              /* window scaling for send window */
        u_char  rcv_scale;              /* window scaling for recv window */
        u_char  request_r_scale;        /* pending window scaling */
        u_char  requested_s_scale;
        u_long  ts_recent;              /* timestamp echo data */
        u_long  ts_recent_age;          /* when last updated */
        tcp_seq last_ack_sent;
/* RFC 1644 variables */
        tcp_cc  cc_send;                /* send connection count */
        tcp_cc  cc_recv;                /* receive connection count */
        u_long  t_duration;             /* connection duration */

/* TUBA stuff */
        caddr_t t_tuba_pcb;             /* next level down pcb for TCP over z */
/* More RTT stuff */
        u_long  t_rttupdated;           /* number of times rtt sampled */

#if defined(TCP_FACK) || defined(TCP_AUTO)
        u_short t_alt_flags;   /* experimental flags */
#define TAF_SACK_SEEN   0x0001 /* other side can send SACKs */
#define TAF_RECOVERY    0x0002 /* We are recovering from a lost segment */
#define TAF_RATEHALF    0x0004 /* We are reducing the window during recovery */
#define TAF_REPAIRED    0x0008 /* We have retransmitted something */
#define TAF_WHOLD       0x0010 /* we are in the window hold state */
#define TAF_TOGGLE      0x0020 /* divide by 2 toggle */
#define TAF_AUTO_OFF    0x0040 /* autotuning is forced off */
#endif

#if defined(TCP_SACK) || defined(TCP_FACK)
#define SACK_LIST_LEN 10

/* Needed for SACK and FACK:  */
        struct sackblock sack_list[SACK_LIST_LEN];  /*  Hack, keep 5 most recent SACKs  */
        struct scoreboard scrb;
        int snd_retran_data;
        tcp_seq recover;
#endif
#ifdef TCP_SACK
        int pipe;
#endif
#ifdef TCP_FACK
        tcp_seq snd_fack;
#define TCP_FACK_REXMTTHRESH 3    /* number of SACKs before retransmitting a block */
        u_long lothresh;                /* recovery window floor */
        u_long hithresh;                /* maximum window following recovery */
#endif
};

/*
 * Structure to hold TCP options that are only used during segment
 * processing (in tcp_input), but not held in the tcpcb.
 * It's basically used to reduce the number of parameters
 * to tcp_dooptions.
 */
struct tcpopt {
        u_long  to_flag;                /* which options are present */
#define TOF_TS          0x0001          /* timestamp */
#define TOF_CC          0x0002          /* CC and CCnew are exclusive */
#define TOF_CCNEW       0x0004
#define TOF_CCECHO      0x0008
        u_long  to_tsval;
        u_long  to_tsecr;
        tcp_cc  to_cc;          /* holds CC or CCnew */
        tcp_cc  to_ccecho;
};

/*
 * The TAO cache entry which is stored in the protocol family specific
 * portion of the route metrics.
 */
struct rmxp_tao {
        tcp_cc  tao_cc;                 /* latest CC in valid SYN */
        tcp_cc  tao_ccsent;             /* latest CC sent to peer */
        u_short tao_mssopt;             /* peer's cached MSS */
#ifdef notyet
        u_short tao_flags;              /* cache status flags */
#define TAOF_DONT       0x0001          /* peer doesn't understand rfc1644 */
#define TAOF_OK         0x0002          /* peer does understand rfc1644 */
#define TAOF_UNDEF      0               /* we don't know yet */
#endif /* notyet */
};
#define rmx_taop(r)     ((struct rmxp_tao *)(r).rmx_filler)

#define intotcpcb(ip)   ((struct tcpcb *)(ip)->inp_ppcb)
#define sototcpcb(so)   (intotcpcb(sotoinpcb(so)))

/*
 * The smoothed round-trip time and estimated variance
 * are stored as fixed point numbers scaled by the values below.
 * For convenience, these scales are also used in smoothing the average
 * (smoothed = (1/scale)sample + ((scale-1)/scale)smoothed).
 * With these scales, srtt has 3 bits to the right of the binary point,
 * and thus an "ALPHA" of 0.875.  rttvar has 2 bits to the right of the
 * binary point, and is smoothed with an ALPHA of 0.75.
 */
#define TCP_RTT_SCALE           32      /* multiplier for srtt; 3 bits frac. */
#define TCP_RTT_SHIFT           5       /* shift for srtt; 3 bits frac. */
#define TCP_RTTVAR_SCALE        16      /* multiplier for rttvar; 2 bits */
#define TCP_RTTVAR_SHIFT        4       /* shift for rttvar; 2 bits */
#define TCP_DELTA_SHIFT         2       /* see tcp_input.c */

/*
 * The initial retransmission should happen at rtt + 4 * rttvar.
 * Because of the way we do the smoothing, srtt and rttvar
 * will each average +1/2 tick of bias.  When we compute
 * the retransmit timer, we want 1/2 tick of rounding and
 * 1 extra tick because of +-1/2 tick uncertainty in the
 * firing of the timer.  The bias will give us exactly the
 * 1.5 tick we need.  But, because the bias is
 * statistical, we have to test that we don't drop below
 * the minimum feasible timer (which is 2 ticks).
 * This version of the macro adapted from a paper by Lawrence
 * Brakmo and Larry Peterson which outlines a problem caused
 * by insufficient precision in the original implementation,
 * which results in inappropriately large RTO values for very
 * fast networks.
 */
#define TCP_REXMTVAL(tp) \
        ((((tp)->t_srtt >> (TCP_RTT_SHIFT - TCP_DELTA_SHIFT))  \
          + (tp)->t_rttvar) >> TCP_DELTA_SHIFT)

/* XXX
 * We want to avoid doing m_pullup on incoming packets but that
 * means avoiding dtom on the tcp reassembly code.  That in turn means
 * keeping an mbuf pointer in the reassembly queue (since we might
 * have a cluster).  As a quick hack, the source & destination
 * port numbers (which are no longer needed once we've located the
 * tcpcb) are overlayed with an mbuf pointer.
 */
#define REASS_MBUF(ti) (*(struct mbuf **)&((ti)->ti_t))

/*
 * TCP statistics.
 * Many of these should be kept per connection,
 * but that's inconvenient at the moment.
 */
struct  tcpstat {
        u_long  tcps_connattempt;       /* connections initiated */
        u_long  tcps_accepts;           /* connections accepted */
        u_long  tcps_connects;          /* connections established */
        u_long  tcps_drops;             /* connections dropped */
        u_long  tcps_conndrops;         /* embryonic connections dropped */
        u_long  tcps_closed;            /* conn. closed (includes drops) */
        u_long  tcps_segstimed;         /* segs where we tried to get rtt */
        u_long  tcps_rttupdated;        /* times we succeeded */
        u_long  tcps_delack;            /* delayed acks sent */
        u_long  tcps_timeoutdrop;       /* conn. dropped in rxmt timeout */
        u_long  tcps_rexmttimeo;        /* retransmit timeouts */
        u_long  tcps_persisttimeo;      /* persist timeouts */
        u_long  tcps_keeptimeo;         /* keepalive timeouts */
        u_long  tcps_keepprobe;         /* keepalive probes sent */
        u_long  tcps_keepdrops;         /* connections dropped in keepalive */

        u_long  tcps_sndtotal;          /* total packets sent */
        u_long  tcps_sndpack;           /* data packets sent */
        u_long  tcps_sndbyte;           /* data bytes sent */
        u_long  tcps_sndrexmitpack;     /* data packets retransmitted */
        u_long  tcps_sndrexmitbyte;     /* data bytes retransmitted */
        u_long  tcps_sndacks;           /* ack-only packets sent */
        u_long  tcps_sndprobe;          /* window probes sent */
        u_long  tcps_sndurg;            /* packets sent with URG only */
        u_long  tcps_sndwinup;          /* window update-only packets sent */
        u_long  tcps_sndctrl;           /* control (SYN|FIN|RST) packets sent */

        u_long  tcps_rcvtotal;          /* total packets received */
        u_long  tcps_rcvpack;           /* packets received in sequence */
        u_long  tcps_rcvbyte;           /* bytes received in sequence */
        u_long  tcps_rcvbadsum;         /* packets received with ccksum errs */
        u_long  tcps_rcvbadoff;         /* packets received with bad offset */
        u_long  tcps_rcvshort;          /* packets received too short */
        u_long  tcps_rcvduppack;        /* duplicate-only packets received */
        u_long  tcps_rcvdupbyte;        /* duplicate-only bytes received */
        u_long  tcps_rcvpartduppack;    /* packets with some duplicate data */
        u_long  tcps_rcvpartdupbyte;    /* dup. bytes in part-dup. packets */
        u_long  tcps_rcvoopack;         /* out-of-order packets received */
        u_long  tcps_rcvoobyte;         /* out-of-order bytes received */
        u_long  tcps_rcvpackafterwin;   /* packets with data after window */
        u_long  tcps_rcvbyteafterwin;   /* bytes rcvd after window */
        u_long  tcps_rcvafterclose;     /* packets rcvd after "close" */
        u_long  tcps_rcvwinprobe;       /* rcvd window probe packets */
        u_long  tcps_rcvdupack;         /* rcvd duplicate acks */
        u_long  tcps_rcvacktoomuch;     /* rcvd acks for unsent data */
        u_long  tcps_rcvackpack;        /* rcvd ack packets */
        u_long  tcps_rcvackbyte;        /* bytes acked by rcvd acks */
        u_long  tcps_rcvwinupd;         /* rcvd window update packets */
        u_long  tcps_pawsdrop;          /* segments dropped due to PAWS */
        u_long  tcps_predack;           /* times hdr predict ok for acks */
        u_long  tcps_preddat;           /* times hdr predict ok for data pkts */
        u_long  tcps_pcbcachemiss;
        u_long  tcps_cachedrtt;         /* times cached RTT in route updated */
        u_long  tcps_cachedrttvar;      /* times cached rttvar updated */
        u_long  tcps_cachedssthresh;    /* times cached ssthresh updated */
        u_long  tcps_usedrtt;           /* times RTT initialized from route */
        u_long  tcps_usedrttvar;        /* times RTTVAR initialized from rt */
        u_long  tcps_usedssthresh;      /* times ssthresh initialized from rt*/
        u_long  tcps_persistdrop;       /* timeout in persist state */
        u_long  tcps_badsyn;            /* bogus SYN, e.g. premature ACK */
        u_long  tcps_mturesent;         /* resends due to MTU discovery */
        u_long  tcps_listendrop;        /* listen queue overflows */

#ifdef TCP_FACK
       /* NOTE: This may break some programs that rely on this structure
        * being fixed length.
        */
        u_long  tcps_fack_recovery;     /* fack: recovery episodes */
        u_long  tcps_fack_sndpack;      /* fack: data packets sent */
        u_long  tcps_fack_sndbyte;      /* fack: data bytes sent */
        u_long  tcps_fack_sndrexmitpack;/* fack: data packets retransmitted */
        u_long  tcps_fack_sndrexmitbyte;/* fack: data bytes retransmitted */
#endif
};

/*
 * Names for TCP sysctl objects
 */
#define TCPCTL_DO_RFC1323       1       /* use RFC-1323 extensions */
#define TCPCTL_DO_RFC1644       2       /* use RFC-1644 extensions */
#define TCPCTL_MSSDFLT          3       /* MSS default */
#define TCPCTL_STATS            4       /* statistics (read-only) */
#define TCPCTL_RTTDFLT          5       /* default RTT estimate */
#define TCPCTL_KEEPIDLE         6       /* keepalive idle timer */
#define TCPCTL_KEEPINTVL        7       /* interval to send keepalives */
#define TCPCTL_SENDSPACE        8       /* send buffer space */
#define TCPCTL_RECVSPACE        9       /* receive buffer space */
#define TCPCTL_KEEPINIT         10      /* receive buffer space */
#define TCPCTL_MAXID            11

#define TCPCTL_NAMES { \
        { 0, 0 }, \
        { "rfc1323", CTLTYPE_INT }, \
        { "rfc1644", CTLTYPE_INT }, \
        { "mssdflt", CTLTYPE_INT }, \
        { "stats", CTLTYPE_STRUCT }, \
        { "rttdflt", CTLTYPE_INT }, \
        { "keepidle", CTLTYPE_INT }, \
        { "keepintvl", CTLTYPE_INT }, \
        { "sendspace", CTLTYPE_INT }, \
        { "recvspace", CTLTYPE_INT }, \
        { "keepinit", CTLTYPE_INT }, \
}

#endif
