/* This file has been put into the public domain by its author.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
   OTHER DEALINGS IN THE SOFTWARE.
*/

/* casinh, casinhf, casinhl - complex arc hyperbolic sine functions

   AUTHOR: Gregory Pietsch

   DESCRIPTION:

   The functionality described on this reference page is aligned with the 
   ISO C standard. Any conflict between the requirements described here and 
   the ISO C standard is unintentional. This volume of POSIX.1-2008 defers 
   to the ISO C standard.

   These functions shall compute the complex arc hyperbolic sine of z, with 
   branch cuts outside the interval [-i, +i] along the imaginary axis.

   RETURN VALUE

   These functions shall return the complex arc hyperbolic sine value, in the 
   range of a strip mathematically unbounded along the real axis and in the 
   interval [-(i * pi/2), +(i * pi/2)] along the imaginary axis.

   ERRORS:

   No errors are defined.
*/

#include <complex.h>
#include "xmath.h"

/* asinh(z) = -i asin(i z) */

double _Complex (casinh) (double _Complex z)
{
  z = casin (CMPLX (-cimag (z), creal (z)));
  return CMPLX (cimag (z), -creal (z));
}

float _Complex (casinhf) (float _Complex z)
{
  z = casinf (CMPLXF (-cimagf (z), crealf (z)));
  return CMPLXF (cimagf (z), -crealf (z));
}

long double _Complex (casinhl) (long double _Complex z)
{
  z = casinl (CMPLXL (-cimagl (z), creall (z)));
  return CMPLXL (cimagl (z), -creall (z));
}

/* END OF FILE */
