/* This file has been put into the public domain by its author.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
   OTHER DEALINGS IN THE SOFTWARE.
*/

/* cexp, cexpf, cexpl - complex exponential functions

   AUTHOR: Gregory Pietsch

   DESCRIPTION:

   The functionality described on this reference page is aligned with the 
   ISO C standard. Any conflict between the requirements described here and 
   the ISO C standard is unintentional. This volume of POSIX.1-2008 defers 
   to the ISO C standard.

   These functions shall compute the complex exponent of z, defined as e^z.

   RETURN VALUE:

   These functions shall return the complex exponential value of z.

   ERRORS

   No errors are defined.
*/

#include <complex.h>
#include "xmath.h"

double _Complex (cexp) (double _Complex z)
{
  double x;

  x = exp (creal (z));
  return CMPLX (x * cos (cimag (z)), x * sin (cimag (z)));
}

float _Complex (cexpf) (float _Complex z)
{
  float x;

  x = expf (crealf (z));
  return CMPLXF (x * cosf (cimagf (z)), x * sinf (cimagf (z)));
}

long double _Complex (cexpl) (long double _Complex z)
{
  long double x;

  x = expl (creall (z));
  return CMPLXL (x * cosl (cimagl (z)), x * sinl (cimagl (z)));
}

/* END OF FILE */
