/* This file has been put into the public domain by its author.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
   OTHER DEALINGS IN THE SOFTWARE.
*/

/* csqrt, csqrtf, csqrtl - complex square root functions

   AUTHOR: Gregory Pietsch

   DESCRIPTION:

   The functionality described on this reference page is aligned with the 
   ISO C standard. Any conflict between the requirements described here and 
   the ISO C standard is unintentional. This volume of POSIX.1-2008 defers 
   to the ISO C standard.

   These functions shall compute the complex square root of z, with a branch 
   cut along the negative real axis.

   RETURN VALUE:

   These functions shall return the complex square root value, in the range 
   of the right half-plane (including the imaginary axis).

   ERRORS:

   No errors are defined.
*/

#include <complex.h>
#include "xmath.h"

double _Complex (csqrt) (double _Complex z)
{
  double mag, x, y, q;

  x = creal (z);
  y = cimag (z);
  mag = cabs (z);
  if (mag == 0.0)
    return CMPLX (0.0, 0.0);
  else if (x > 0.0)
    {
      q = sqrt (0.5 * (mag + x));
      return CMPLX (q, y / q / 2.0);
    }
  else
    {
      q = sqrt (0.5 * (mag - x));
      if (y < 0.0)
	q = -q;
      return CMPLX (y / q / 2.0, q);
    }
}

float _Complex (csqrtf) (float _Complex z)
{
  float mag, x, y, q;

  x = crealf (z);
  y = cimagf (z);
  mag = cabsf (z);
  if (mag == 0.0F)
    return CMPLXF (0.0F, 0.0F);
  else if (x > 0.0F)
    {
      q = sqrtf (0.5F * (mag + x));
      return CMPLXF (q, y / q / 2.0F);
    }
  else
    {
      q = sqrtf (0.5F * (mag - x));
      if (y < 0.0F)
	q = -q;
      return CMPLXF (y / q / 2.0F, q);
    }
}

long double _Complex (csqrtl) (long double _Complex z)
{
  long double mag, x, y, q;

  x = creall (z);
  y = cimagl (z);
  mag = cabsl (z);
  if (mag == 0.0L)
    return CMPLX (0.0L, 0.0L);
  else if (x > 0.0L)
    {
      q = sqrtl (0.5L * (mag + x));
      return CMPLXL (q, y / q / 2.0L);
    }
  else
    {
      q = sqrtl (0.5L * (mag - x));
      if (y < 0.0L)
	q = -q;
      return CMPLXL (y / q / 2.0L, q);
    }
}

/* END OF FILE */
