/* This file has been put into the public domain by its author.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
   OTHER DEALINGS IN THE SOFTWARE.
*/

/* catanh, catanhf, catanhl - complex hyperbolic arc tangent functions

   AUTHOR: Gregory Pietsch

   DESCRIPTION

   The functionality described on this reference page is aligned with the 
   ISO C standard. Any conflict between the requirements described here and 
   the ISO C standard is unintentional. This volume of POSIX.1-2017 defers 
   to the ISO C standard.

   These functions shall compute the complex hyperbolic arc tangent of z, 
   with branch cuts outside the interval [-1, +1] along the real axis.

   RETURN VALUE

   These functions shall return the complex arc tangent value, in the range 
   of a strip mathematically unbounded along the real axis and in the 
   interval [-i(pi/2), +i(pi/2)] along the imaginary axis.

   ERRORS

   No errors are defined.
*/

#include <complex.h>
#include "xmath.h"

double _Complex (catanh) (double _Complex z)
{
  double x = creal (z);
  double y = cimag (z);
  double x2 = x * x;
  double y2 = y * y;
  double denom = (1.0 - x) * (1.0 - x) + y2;
  double real = 0.5 * log ((1.0 + x) * (1.0 - x) + y2) - 0.5 * log (denom);
  double imag = 0.5 * atan2 (2.0 * y, (1.0 - x) * (1.0 - x) - y2);
  return CMPLX (real, imag);
}

float _Complex (catanhf) (float _Complex z)
{
  float x = crealf (z);
  float y = cimagf (z);
  float x2 = x * x;
  float y2 = y * y;
  float denom = (1.0F - x) * (1.0F - x) + y2;
  float real =
    0.5F * logf ((1.0F + x) * (1.0F - x) + y2) - 0.5F * logf (denom);
  float imag = 0.5F * atan2f (2.0F * y, (1.0F - x) * (1.0F - x) - y2);
  return CMPLXF (real, imag);
}

long double _Complex (catanhl) (long double _Complex z)
{
  long double x = creall (z);
  long double y = cimagl (z);
  long double x2 = x * x;
  long double y2 = y * y;
  long double denom = (1.0L - x) * (1.0L - x) + y2;
  long double real =
    0.5L * logl ((1.0L + x) * (1.0L - x) + y2) - 0.5L * logl (denom);
  long double imag = 0.5L * atan2l (2.0L * y, (1.0L - x) * (1.0L - x) - y2);
  return CMPLXL (real, imag);
}

/* END OF FILE */
