/* This file has been put into the public domain by its author.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
   OTHER DEALINGS IN THE SOFTWARE.
*/

/* csqrt, csqrtf, csqrtl - complex square root functions

   AUTHOR: Gregory Pietsch

   DESCRIPTION:

   The functionality described on this reference page is aligned with the 
   ISO C standard. Any conflict between the requirements described here and 
   the ISO C standard is unintentional. This volume of POSIX.1-2008 defers 
   to the ISO C standard.

   These functions shall compute the complex square root of z, with a branch 
   cut along the negative real axis.

   RETURN VALUE:

   These functions shall return the complex square root value, in the range 
   of the right half-plane (including the imaginary axis).

   ERRORS:

   No errors are defined.

   Appendix G of the C99 Standard mandates this behavior:
   
   csqrt(conj(z)) = conj(csqrt(z)).

   csqrt(+/-0 + i0) returns +0 + i0.
   
   csqrt(x + i(Inf)) returns +Inf + i(Inf), for all x (including NaN).

   csqrt(x + iNaN) returns NaN + iNaN and optionally raises the "invalid" 
   floating-point exception, for finite x.

   csqrt(−Inf + iy) returns +0 + i(Inf), for finite positive-signed y.

   csqrt(+Inf + iy) returns +Inf + i0, for finite positive-signed y.
   
   csqrt(−Inf + iNaN) returns NaN +/- i(Inf) (where the sign of the imaginary 
   part of the result is unspecified).

   csqrt(+Inf + iNaN) returns +Inf + iNaN.

   csqrt(NaN + iy) returns NaN + iNaN and optionally raises the "invalid" 
   floating-point exception, for finite y.

   csqrt(NaN + iNaN) returns NaN + iNaN.

*/

#include <complex.h>
#include "xmath.h"

double _Complex (csqrt) (double _Complex z)
{
  double x, y, r, s, t;
  int cx, cy, sx;

  x = creal (z);
  y = cimag (z);
  cx = _Fpclassify (x);
  cy = _Fpclassify (y);
  if (cx == FP_ZERO && cy == FP_ZERO)
    return CMPLX (+0.0, +0.0);
  else if (cy == FP_INFINITE)
    return CMPLX (_Dbl->_Inf._D, _Dbl->_Inf._D);
  else if (cx == FP_NAN || cy == FP_NAN)
    {
      if (cx == FP_NAN && cy == FP_NAN)
	return z;
      else if (cx == FP_INFINITE)
	{
	  sx = _Getsign ((unsigned char *) &x, _Dbl);
	  if (sx)
	    return CMPLX (y, x);
	  else
	    return z;
	}
      else
	{
	  _Matherr (EDOM, FE_INVALID);
	  return CMPLX (y, y);
	}
    }
  else if (cx == FP_INFINITE)
    {
      sx = _Getsign ((unsigned char *) &x, _Dbl);
      return sx ? CMPLX (+0.0, _Dbl->_Inf._D) : CMPLX (_Dbl->_Inf._D, +0.0);
    }
  else if (cy == FP_ZERO)
    {
      sx = _Getsign ((unsigned char *) &x, _Dbl);
      return sx ? CMPLX (+0.0, sqrt (-x)) : CMPLX (sqrt (x), +0.0);
    }
  r = hypot (x, y);
  s = sqrt ((r + fabs (x)) / 2.0);
  t = (y / s) / 2.0;
  sx = _Getsign ((unsigned char *) &x, _Dbl);
  return sx ? CMPLX (fabs (t), copysign (s, y)) : CMPLX (s, t);
}

float _Complex (csqrtf) (float _Complex z)
{
  float x, y, r, s, t;
  int cx, cy, sx;

  x = crealf (z);
  y = cimagf (z);
  cx = _Fpclassifyf (x);
  cy = _Fpclassifyf (y);
  if (cx == FP_ZERO && cy == FP_ZERO)
    return CMPLXF (+0.0F, +0.0F);
  else if (cy == FP_INFINITE)
    return CMPLXF (_Flt->_Inf._F, _Flt->_Inf._F);
  else if (cx == FP_NAN || cy == FP_NAN)
    {
      if (cx == FP_NAN && cy == FP_NAN)
	return z;
      else if (cx == FP_INFINITE)
	{
	  sx = _Getsign ((unsigned char *) &x, _Flt);
	  if (sx)
	    return CMPLXF (y, x);
	  else
	    return z;
	}
      else
	{
	  _Matherr (EDOM, FE_INVALID);
	  return CMPLXF (y, y);
	}
    }
  else if (cx == FP_INFINITE)
    {
      sx = _Getsign ((unsigned char *) &x, _Flt);
      return sx ? CMPLXF (+0.0F, _Flt->_Inf._F) : CMPLX (_Flt->_Inf._D,
							 +0.0F);
    }
  else if (cy == FP_ZERO)
    {
      sx = _Getsign ((unsigned char *) &x, _Flt);
      return sx ? CMPLXF (+0.0F, sqrtf (-x)) : CMPLXF (sqrtf (x), +0.0F);
    }
  r = hypotf (x, y);
  s = sqrtf ((r + fabsf (x)) / 2.0F);
  t = (y / s) / 2.0F;
  sx = _Getsign ((unsigned char *) &x, _Flt);
  return sx ? CMPLXF (fabsf (t), copysignf (s, y)) : CMPLXF (s, t);
}

long double _Complex (csqrtl) (long double _Complex z)
{
  long double x, y, r, s, t;
  int cx, cy, sx;

  x = creall (z);
  y = cimagl (z);
  cx = _Fpclassifyl (x);
  cy = _Fpclassifyl (y);
  if (cx == FP_ZERO && cy == FP_ZERO)
    return CMPLXL (+0.0L, +0.0L);
  else if (cy == FP_INFINITE)
    return CMPLXL (_Ldbl->_Inf._L, _Ldbl->_Inf._L);
  else if (cx == FP_NAN || cy == FP_NAN)
    {
      if (cx == FP_NAN && cy == FP_NAN)
	return z;
      else if (cx == FP_INFINITE)
	{
	  sx = _Getsign ((unsigned char *) &x, _Ldbl);
	  if (sx)
	    return CMPLXL (y, x);
	  else
	    return z;
	}
      else
	{
	  _Matherr (EDOM, FE_INVALID);
	  return CMPLXL (y, y);
	}
    }
  else if (cx == FP_INFINITE)
    {
      sx = _Getsign ((unsigned char *) &x, _Ldbl);
      return sx ? CMPLXL (+0.0L, _Ldbl->_Inf._L)
	: CMPLXL (_Ldbl->_Inf._L, +0.0L);
    }
  else if (cy == FP_ZERO)
    {
      sx = _Getsign ((unsigned char *) &x, _Ldbl);
      return sx ? CMPLXL (+0.0L, sqrtl (-x)) : CMPLXL (sqrtl (x), +0.0L);
    }
  r = hypotl (x, y);
  s = sqrtl ((r + fabsl (x)) / 2.0L);
  t = (y / s) / 2.0L;
  sx = _Getsign ((unsigned char *) &x, _Ldbl);
  return sx ? CMPLXL (fabsl (t), copysignl (s, y)) : CMPLXL (s, t);
}

/* END OF FILE */
