/* This file has been put into the public domain by its author.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
   OTHER DEALINGS IN THE SOFTWARE.
*/

/* tanh, tanhf, tanhl - hyperbolic tangent functions

   AUTHOR: Gregory Pietsch

   DESCRIPTION

   The functionality described on this reference page is aligned with the 
   ISO C standard. Any conflict between the requirements described here and 
   the ISO C standard is unintentional. This volume of POSIX.1-2008 defers 
   to the ISO C standard.

   These functions shall compute the hyperbolic tangent of their argument x.

   An application wishing to check for error situations should set errno to 
   zero and call feclearexcept(FE_ALL_EXCEPT) before calling these functions. 
   On return, if errno is non-zero or fetestexcept(FE_INVALID | FE_DIVBYZERO 
   | FE_OVERFLOW | FE_UNDERFLOW) is non-zero, an error has occurred.

   RETURN VALUE

   Upon successful completion, these functions shall return the hyperbolic 
   tangent of x.

   If x is NaN, a NaN shall be returned.

   If x is +/-0, x shall be returned.

   If x is +/-Inf, +/-1 shall be returned.

   If x is subnormal, a range error may occur and x should be returned.

   If x is not returned, tanh(), tanhf(), and tanhl() shall return an 
   implementation-defined value no greater in magnitude than DBL_MIN, 
   FLT_MIN, and LDBL_MIN, respectively.

   ERRORS

   These functions may fail if:
   Range Error
   The value of x is subnormal. 
   If the integer expression (math_errhandling & MATH_ERRNO) is non-zero, 
   then errno shall be set to ERANGE. If the integer expression 
   (math_errhandling & MATH_ERREXCEPT) is non-zero, then the underflow 
   floating-point exception shall be raised.

   ALGORITHM:
   
   1.   For small inputs (|x| <= 0.5), a polynomial approximation is used
        to compute tanh(x) directly. The coefficients of the polynomial 
        are precomputed and optimized for accuracy in each precision.
   2.   For larger inputs (|x| > 0.5), the hyperbolic tangent identity is
        used to compute tanh(x) as 1 - 2 / (exp(2x) + 1). This expression 
        is rearranged to avoid overflow/underflow issues and computed 
        using another polynomial approximation for the term 
        2 / (exp(2x) + 1).

*/

#include "xmath.h"

/* Constants for double precision */
static const double P[] = {
  4.16938922512284326504311638314e-8,
  2.76076847742355635309623107416e-6,
  1.16888925859191382142080315187e-4,
  3.13770543556508811624629418145e-3,
  4.38203601801624068368709547775e-2,
  2.85619036796905312063712786668e-1,
  1.00000000000000000000000000000e+0,
};

static const double Q[] = {
  2.94078955048598507346796876262e-9,
  2.51945159528138168910638186282e-7,
  1.22426109148261976594219487892e-5,
  3.34702365976030692301624856229e-4,
  5.84915919393474247594408882431e-3,
  6.23489932084269003455154890283e-2,
  4.00000000000000000000000000000e-1,
  1.00000000000000000000000000000e+0,
};

/* Constants for single precision */
static const float Pf[] = {
  9.64399179359752452503e-7f,
  2.26390925031595409704e-5f,
  2.76282708911095083252e-4f,
  1.96777106701602722294e-3f,
  8.33028376239439503789e-3f,
  2.05058574770865580699e-2f,
  2.49015124357883938627e-2f,
  1.33314516438595153238e-1f,
};

static const float Qf[] = {
  1.57799883256658803364e-7f,
  2.50804902663263153538e-6f,
  2.23548839548079259746e-5f,
  1.12656624633284594914e-4f,
  3.61207315337924129465e-4f,
  7.69706159362492592469e-4f,
  1.08858621546801774648e-3f,
  1.00000000000000000000e+0f,
};

/* Constants for long double precision */
static const long double Pl[] = {
  4.16938922512284326504311638314e-8L,
  2.76076847742355635309623107416e-6L,
  1.16888925859191382142080315187e-4L,
  3.13770543556508811624629418145e-3L,
  4.38203601801624068368709547775e-2L,
  2.85619036796905312063712786668e-1L,
  1.00000000000000000000000000000e+0L,
};

static const long double Ql[] = {
  2.94078955048598507346796876262e-9L,
  2.51945159528138168910638186282e-7L,
  1.22426109148261976594219487892e-5L,
  3.34702365976030692301624856229e-4L,
  5.84915919393474247594408882431e-3L,
  6.23489932084269003455154890283e-2L,
  4.00000000000000000000000000000e-1L,
  1.00000000000000000000000000000e+0L,
};

static long double
_Tanh (long double x, const _FP_METRIC_T const *m, long double maxval)
{
  long double q, r, s, t, z, zz;
  int sign;

  if (fabsl (x) <= 0.5L)
    {
      z = x * x;
      switch (m->_Size)
	{
	case 4:
	  r =
	    z * (Pf[0] +
		 z * (Pf[1] +
		      z * (Pf[2] +
			   z * (Pf[3] +
				z * (Pf[4] +
				     z * (Pf[5] +
					  z * (Pf[6] + z * Pf[7])))))));
	  break;
	case 8:
	  r =
	    z * (P[0] +
		 z * (P[1] +
		      z * (P[2] +
			   z * (P[3] + z * (P[4] + z * (P[5] + z * P[6]))))));
	  break;
	default:
	  r =
	    z * (Pl[0] +
		 z * (Pl[1] +
		      z * (Pl[2] +
			   z * (Pl[3] +
				z * (Pl[4] + z * (Pl[5] + z * Pl[6]))))));
	}
      return x * (1.0L + r);
    }
  else
    {
      sign = _Getsign ((unsigned char *) &x, _Ldbl) ? -1 : +1;
      z = 1.0L - 1.0L / (expl (fabsl (x)) + 1.0L);
      zz = z * z;
      switch (m->_Size)
	{
	case 4:
	  s =
	    sign * (Qf[0] +
		    zz * (Qf[1] +
			  zz * (Qf[2] +
				zz * (Qf[3] +
				      zz * (Qf[4] +
					    zz * (Qf[5] +
						  zz * (Qf[6] +
							zz * Qf[7])))))));
	  break;
	case 8:
	  s =
	    sign * (Q[0] +
		    zz * (Q[1] +
			  zz * (Q[2] +
				zz * (Q[3] +
				      zz * (Q[4] +
					    zz * (Q[5] +
						  zz * (Q[6] +
							zz * Q[7])))))));
	  break;
	default:
	  s =
	    sign * (Ql[0] +
		    zz * (Ql[1] +
			  zz * (Ql[2] +
				zz * (Ql[3] +
				      zz * (Ql[4] +
					    zz * (Ql[5] +
						  zz * (Ql[6] +
							zz * Ql[7])))))));
	}
      t = z + s;
      if (t > maxval)
	q = sign;
      else if (t < -maxval)
	q = -sign;
      else
	q = t;
      return q;
    }
}

double (tanh) (double x)
{
  double z;

  switch (_Fpclassify (x))
    {
    case FP_SUBNORMAL:
      _Matherr (ERANGE, FE_UNDERFLOW);	/* FALL THROUGH */
    case FP_NAN:
    case FP_ZERO:
      return x;
    case FP_INFINITE:
      return _Getsign ((unsigned char *) &x, _Dbl) ? -1.0 : +1.0;
    default:
      z = _Tanh (x, _Dbl, (long double) _Dbl->_Max._D);
      return z;
    }
}

float (tanhf) (float x)
{
  float z;

  switch (_Fpclassifyf (x))
    {
    case FP_SUBNORMAL:
      _Matherr (ERANGE, FE_UNDERFLOW);	/* FALL THROUGH */
    case FP_NAN:
    case FP_ZERO:
      return x;
    case FP_INFINITE:
      return _Getsign ((unsigned char *) &x, _Flt) ? -1.0F : +1.0F;
    default:
      z = _Tanh (x, _Flt, (long double) _Flt->_Max._F);
      return z;
    }
}

long double (tanhl) (long double x)
{
  long double z;

  switch (_Fpclassifyl (x))
    {
    case FP_SUBNORMAL:
      _Matherr (ERANGE, FE_UNDERFLOW);	/* FALL THROUGH */
    case FP_NAN:
    case FP_ZERO:
      return x;
    case FP_INFINITE:
      return _Getsign ((unsigned char *) &x, _Ldbl) ? -1.0L : +1.0L;
    default:
      z = _Tanh (x, _Ldbl, _Ldbl->_Max._D);
      return z;
    }
}

/* END OF FILE */
