/* cpy.c - copy bytes in memory with overlapping areas

   AUTHOR: Gregory Pietsch

   DESCRIPTION:
  
   This function is identical to the memmove function in the C library
   and is here only because this library should not be dependent on
   the C library.

   This file has been put into the public domain by its author.
   
   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
   OTHER DEALINGS IN THE SOFTWARE.
*/

#include "mpi.h"

#ifndef _OPTIMIZED_FOR_SIZE

/* Nonzero if either X or Y is not aligned on an "unsigned long" boundary.  */
#ifdef _ALIGN
#define UNALIGNED(X,Y) \
  (((unsigned long)X&(sizeof(unsigned long)-1))\
  |((unsigned long)Y&(sizeof(unsigned long)-1)))
#else
#define UNALIGNED(X,Y) (0)
#endif

/* How many bytes are copied each interation of the word copy loop.  */
#define LITTLEBLOCKSIZE (sizeof(unsigned long))

/* How many bytes are copied each interation of the 4X unrolled loop.  */
#define BIGBLOCKSIZE (sizeof(unsigned long)<<2)

/* Threshhold for punting to the byte copier.  */
#define TOO_SMALL(len) ((len)<BIGBLOCKSIZE)

#endif /* _OPTIMIZED_FOR_SIZE */

void *(_Cpy) (void *s1, const void *s2, size_t n)
{
  unsigned char *us1 = s1;
  const unsigned char *us2 = s2;

#ifndef _OPTIMIZED_FOR_SIZE
  unsigned long *pdst, *psrc;
#endif

  if (us2 < us1 && us1 < us2 + n)
    {
      /* Have to copy backwards.  */
      us1 += n;
      us2 += n;
      while (n--)
	*--us1 = *--us2;
      return s1;
    }
#ifndef _OPTIMIZED_FOR_SIZE
  /* If the size is small, or either s1 or s2 is unaligned, punt into the
     byte copy loop.  This should be rare.  */
  if (!TOO_SMALL (n) && !UNALIGNED (s2, s1))
    {
      pdst = (unsigned long *) s1;
      psrc = (unsigned long *) s2;

      /* Copy a big block at a time if possible. */
      while (n >= BIGBLOCKSIZE)
	{
	  *pdst++ = *psrc++;
	  *pdst++ = *psrc++;
	  *pdst++ = *psrc++;
	  *pdst++ = *psrc++;
	  n -= BIGBLOCKSIZE;
	}

      /* Copy a little block at a time if possible. */
      while (n >= LITTLEBLOCKSIZE)
	{
	  *pdst++ = *psrc++;
	  n -= LITTLEBLOCKSIZE;
	}

      /* Pick up any residual with a byte copier.  */
      us1 = (unsigned char *) pdst;
      us2 = (unsigned char *) psrc;
    }
#endif
  /* The normal byte-copy loop.  */

  while (n--)
    *us1++ = *us2++;
  return s1;
}

#ifndef _OPTIMIZED_FOR_SIZE
#undef UNALIGNED
#undef LITTLEBLOCKSIZE
#undef BIGBLOCKSIZE
#undef TOO_SMALL
#endif /* _OPTIMIZED_FOR_SIZE */

/* END OF FILE */
