/* zer.c - zero bytes in memory

   AUTHOR: Gregory Pietsch

   DESCRIPTION:

   This is similar to the code for memset from the C library with the 
   second parameter omitted and set to '\0'.  The function is in this
   library here because this library should not be dependent on the
   C library.
   
   This file has been put into the public domain by its author.
   
   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
   OTHER DEALINGS IN THE SOFTWARE.
*/

#include "mpi.h"

#ifndef _OPTIMIZED_FOR_SIZE
/* Nonzero if X is not aligned on an "unsigned long" boundary.  */
#ifdef _ALIGN
#define UNALIGNED(X) ((unsigned long)X&(sizeof(unsigned long)-1))
#else
#define UNALIGNED(X) 0
#endif
/* How many bytes are copied each iteration of the little loop.  */
#define LITTLEBLOCKSIZE (sizeof(unsigned long))
/* Threshhold for punting into the byte copier.  */
#define TOO_SMALL(X) ((X)<LITTLEBLOCKSIZE)
#endif /* _OPTIMIZED_FOR_SIZE */

  /* memset */
void *(_Zer) (void *s, size_t n)
{
  unsigned char *us = s;
#ifndef _OPTIMIZED_FOR_SIZE
  unsigned long *ps;
  size_t i;

  if (!TOO_SMALL (n) && !UNALIGNED (s))
    {
      ps = (unsigned long *) s;
      /* Copy a 4X block if possible. */
      while (n >= (LITTLEBLOCKSIZE * 4))
	{
	  *ps++ = 0UL;
	  *ps++ = 0UL;
	  *ps++ = 0UL;
	  *ps++ = 0UL;
	  n -= LITTLEBLOCKSIZE * 4;
	}
      /* Copy a 1X block if possible. */
      while (n >= LITTLEBLOCKSIZE)
	{
	  *ps++ = 0UL;
	  n -= LITTLEBLOCKSIZE;
	}
      /* Pick up any residual with a byte copier.  */
      us = (unsigned char *) ps;
    }
#endif /* _OPTIMIZED_FOR_SIZE */

  while (n-- != 0)
    *us++ = 0;
  return s;
}

/* END OF FILE */
