/*
    SUPPL - a supplemental library for common useage
    Copyright (C) 1995,1996  Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    See: COPYING.LB
*/
/* $RCSfile: DFNEXPAN.C $
   $Locker:  $	$Name:  $	$State: Exp $

	char *dfnexpand(const char fname[], const char path[])

	Expands a filename to it a fully-qualified DOS-style path relative
	to path.

	If SUPPORT_UNC_PATH is defined, UNC paths are supported. If fname[]
	starts with "\\\\", is is assumed to be an UNC path. UNC paths are
	always fully-qualified!

	The expanded filename is stored into a malloc'ed buffer.
	fname[] is preprocessed by dfnsqueeze().
	path[] is _not_ preprocessed by dfnsqueeze().

	If path == NULL, the current working directory is used.
	If path[] contains no drive spec, but a path spec, this path spec is
	assumed to be the current working directory of either the current
	drive or the drive specified by fname[].
	If the path spec of path[] is not fully-qualified, it is _not_ expanded;
	e.g. dfnexpand("x:foe/bar", "d:humpf/er") -> "x:humpf/er\\FOE\\BAR"

   Return:
   	NULL: malloc failed or fname == NULL
	else: pointer to the malloc'ed buffer

	Target compilers: Micro-C v3.13, v3.14; Borland C v2, v3.1, v4.52

*/

#include "initsupl.inc"

#ifndef _MICROC_
#include <string.h>
#include <stdlib.h>
#include <dos.h>
#include "dir.inc"
#endif
#include <portable.h>
#include "dfn.h"
#include "dynstr.h"
#include "suppl.h"
#include "dir.inc"

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: DFNEXPAN.C 1.8 1998/08/19 07:26:56 ska Exp $";
#endif

char *dfnexpand(const char fname[], char path[])
{	char *h, *p;				/* intermediate pointers */
	char *dr, *pa, *na, *ex;	/* filename components */
	char pathDr, *pathPa;		/* drive & path of 'path' */
	char *dynPath;

	if((h = dfnsqueeze(fname)) == NULL)
		return NULL;

#ifdef SUPPORT_UNC_PATH
	if(isUNCpath(h)) {			/* UNC paths are always fully-qualified */
		/* check if the trailing '\\' is present to mark the root direc */
		return (*UNCpath(h) != '\\')? StrAppChr(h, '\\') : h;
	}
#endif

	if(h[1] != ':' || h[2] != '\\') {	/* the spec is not fully-qualified */
		pathDr = 0;
		dynPath = NULL;
		if((pathPa = path) != NULL && *pathPa) {
			if(pathPa[1] == ':') {	/* path[] has drive spec */
				pathDr = *path;
				if(!*(pathPa += 2)) {
					pathPa = NULL;
					goto noPath;
				}
			}
			if((*pathPa == '\\' || *pathPa == '/') && !pathPa[1])
				++pathPa;		/* Trans "/" || "\\" --> "" */
noPath:;
		}
		if(dfnsplit(h, &dr, &pa, &na, &ex)) {
			StrFree(h);
			if(dr) {				/* drive specified */
				if(pathDr && toFUpper(pathDr) != *dr)
					/* The specified path is for a different drive */
					pathPa = NULL;
			}
			else {					/* drive spec missing */
				if((dr = StrChar(pathDr? pathDr: 'A' + getdisk())) == NULL)
					goto errRet;
			}

			if(!pa || *pa != '\\' && *pa != NUL) {
			/* no path or a relative one */
				if(!pathPa) {				/* path has no path spec in it */
					if((dynPath = dfnpath(*dr)) == NULL)
						goto errRet;
					/* Transform "?:\\" --> "" */
					StrStrip(pathPa = dynPath + 2, '\\');
				}

				if((p = dfnmerge(NULL, NULL, pathPa, pa, NULL)) == NULL)
					goto errRet;
				StrStrip(StrRepl(pa, p), '\\');
			}
			h = dfnmerge(NULL, dr, pa, na, ex);
		}
		else StrFree(h);

errRet:
		free(dr);
		free(pa);
		free(na);
		free(ex);
		free(dynPath);
	}

	return h;
}
