{
****************************************************************
****************************************************************
**                                                            **
**          FLISP,  a LISP interpreter for FreeDOS            **
**                                                            **
**          Copyright (C) 2005 by Francesco Zamblera          **
**           under the GNU General Public License             **
**                                                            **
**                       Version 1.0                          **
**                                                            **
****************************************************************
****************************************************************

    This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

    This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    Contact me at

    vilnergoy@yahoo.it
}




{---------------------------------------}
UNIT Scanner1;

{---------------------------------------}
INTERFACE

USES     crt,
         utils,
         header;

CONST Comment = ';';

VAR   UpCaseOn: boolean;

{---------------------------------------}
FUNCTION uppercase (c: char): char;
PROCEDURE GetChar;
FUNCTION IsAlpha (c: char): boolean;
FUNCTION IsDigit (c: char): boolean;
FUNCTION IsAlNum (c: char): boolean;
FUNCTION IsAddop (c: char): boolean;
FUNCTION IsWhite (c: char): boolean;
PROCEDURE SkipWhite;
PROCEDURE SkipComment;
PROCEDURE Match (c: char);
FUNCTION IsKeyword (s: string): integer;
FUNCTION str2int (s: string): integer;
PROCEDURE GetAtom (var token: string);
PROCEDURE GetName (var Node: PCellList);
PROCEDURE GetOp (var Node: PCellList);
PROCEDURE GetNum (var node: PCellList);
PROCEDURE AddToQueue (var List: PCellList; ListNode: PCellList);
PROCEDURE Scan (var Scanned: PCellList);
PROCEDURE AddQuote (var p: PCellList; quote: char);


{---------------------------------------}
IMPLEMENTATION

{---------------------------------------}

FUNCTION uppercase (c: char): char;
begin
 if (c in ['a'..'z']) and UpCaseOn
        then uppercase := upcase(c)
        else uppercase := c
end;


{---------------------------------------}
PROCEDURE GetChar;
begin
 if InputFile = 0
    then read (look)
    else read (Files [InputFile].FileVar,look);
{ if look = CR then inc (NL_Count) }
end;

{---------------------------------------}
FUNCTION IsAlpha (c: char): boolean;
begin
 IsAlpha := (c in ['A'..'Z','a'..'z'])
   or  (c in ['+', '*', '-', '/','.','<','>','=', '&',':','!','_','^',
             '%','$','~','|','\'])
   or (ord(c) > 127)
end;

{---------------------------------------}
FUNCTION IsDigit (c: char): boolean;
begin
 IsDigit := c in ['0'..'9']
end;

{---------------------------------------}
FUNCTION IsAlNum (c: char): boolean;
begin
 IsAlNum := IsAlpha(c) or IsDigit(c)
end;

{---------------------------------------}
FUNCTION IsAddop (c: char): boolean;
begin
 IsAddop := c in ['+', '-']
end;

{---------------------------------------}
FUNCTION IsWhite (c: char): boolean;
begin
 IsWhite := c in [' ', TAB, CR, LF]
end;

{---------------------------------------}
PROCEDURE SkipWhite;
begin
 while IsWhite (Look) do GetChar
end;

{---------------------------------------}
PROCEDURE SkipComment;
begin
 while Look <> CR do getchar;
 SkipWhite
end;

{---------------------------------------}
PROCEDURE Match (c: char);
begin
 if c = Look then GetChar
             else Expected (' "' + c + '"')
end;

{---------------------------------------}
FUNCTION IsKeyword (s: string): integer;
var i: integer;
    found: boolean;
begin
 found := false;
 i := KEYWORDS;
 while (i > 0) and not found do
       if s = KWlist[i] then found := true
                        else dec(i);
 IsKeyword := i
end;

{---------------------------------------}
FUNCTION str2int (s: string): integer;
var n,i,k: integer;
begin
 n := 0;
 k := length(s);
 for i := 1 to k do begin
                     if IsDigit(s[i]) then n := 10*n + ord(s[i]) - ord('0')
                                      else Expected('Digit')
                    end;
 if exit_code then str2int := n else str2int := 0
end;


{---------------------------------------}
PROCEDURE GetAtom (var token: string);
begin
 token := ''; SkipWhite;
 if not IsAlpha (Look)
    then Expected ('Atom')
    else begin
          while IsAlNum(Look) do begin
                                  token := token + uppercase(Look);
                                  GetChar
                                 end
         end
end;

{---------------------------------------}
PROCEDURE GetName (var Node: PCellList);
var token: string; k: integer;
begin
 token := '';
 UpCaseOn := Look <> '|';
 GetAtom (token);
 UpCaseOn := true;
 k := IsKeyword (token);
 with node^.data
      do begin
          if k = 0
             then begin
                   if (token = 'T') or (token = 'NIL')
                      then begin
                            flag := _spec;
                            t := token = 'T'
                           end
                      else begin
                            Flag := _id;
                            varname := token
                           end
                  end
             else begin
                   Flag := _op;
                   OpCode := OpCodes(k-1)
                  end
      end;
 node^.next := nil
end;


{---------------------------------------}
PROCEDURE GetString (var Node: PCellList);
begin
 node^.next := nil;
 node^.data.flag := _string;
 node^.data.str := '';
 match('"');
 while look <> '"' do begin
                       node^.data.str := node^.data.str + Look;
                       GetChar
                      end;
 match('"')
end;

{---------------------------------------}
PROCEDURE GetOp (var Node: PCellList);
var k: integer; token: string;
begin
 token := ''; GetAtom (token);
 k := IsKeyword (token);
 if k = 0 then Expected ('Procedure')
          else begin
                Node^.Data.Flag := _op;
                Node^.Data.OpCode := OpCodes(k-1)
               end;
 node^.next := nil
end;

{---------------------------------------}
PROCEDURE Getint (var token: string; var digits: byte);
begin
 digits := 0; token := '';
 while IsDigit (Look) do begin
                           token := token + Look;
                           inc(Digits);
                           GetChar
                         end
end;

{---------------------------------------}
PROCEDURE ShiftPoint (var d: real; e: integer);
var i: byte;
begin
 if e > 0 then for i := 1 to e do d := d * 10
          else if e < 0 then begin
                              e := - e;
                              for i := 1 to e do d := d / 10
                             end
end;

{---------------------------------------}
PROCEDURE GetNum (var node: PCellList);
var token: string;
    n: integer;
    m: real;
    i,digits,expt: byte;
    IsNeg: boolean;
begin
 SkipWhite;
 if not IsDigit (Look)
    then Expected ('Integer')
    else begin
          GetInt(Token,Digits);
          n := str2int(Token);
          case look of
               '/': begin
                     match('/');
                     node^.data.flag := _ratio;
                     node^.data.num := n;
                     GetInt (Token,Digits);
                     n := str2int(Token);
                     if n = 0 then begin
                                    error ('Zero division');
                                    node^.data.flag := _null
                                   end
                              else node^.data.denom := n
                    end;
               '.': begin
                     match('.');
                     node^.data.flag := _float;
                     node^.data.float := n;
                     GetInt(Token,Digits);
                     m := str2int(Token);
                     for i := 1 to Digits do m := m / 10;
                     node^.data.float := node^.data.float + m;
                     if Look = 'E'
                        then begin
                              match('E');
                              if look in ['+','-']
                                 then begin
                                       IsNeg := look = '-';
                                       getchar
                                      end
                                 else IsNeg := false;
                              GetInt(Token,Digits);
                              Expt := str2int(Token);
                              if IsNeg then Expt := - Expt;
                              ShiftPoint (node^.data.float,expt)
                             end
                    end
               else begin
                     with Node^.Data do begin
                                         Flag := _int;
                                         int := n
                                        end
                     end
              end;
          node^.next := nil
         end
end;


{---------------------------------------}
PROCEDURE AddToQueue (var List: PCellList; ListNode: PCellList);
var p: PCellList;
begin
 ListNode^.next := nil;
 if List = nil then List := ListNode
               else begin
                     p := List;
                     while p^.next <> nil do p := p^.next;
                     p^.next := ListNode
                    end
end;

{---------------------------------------}
PROCEDURE GetList (var node: PCellList);
var ListNode: pcelllist;
begin
 Match('(');
 with node^.data do begin
                     flag := _list;
                     list := NIL;
                     while (Look <> ')') and exit_code
                            do begin
                                scan(ListNode);
                                AddToQueue (List, ListNode);
                                ListNode := nil;
                                SkipWhite
                               end
                    end;
 Match(')')
end;

{---------------------------------------}
PROCEDURE AddQuote (var p: PCellList; quote: char);
var q: PCellList;
begin
 new(q);
 q^.data.flag := _list;
 new(q^.data.list);
 q^.data.list^.data.flag := _op;
 case quote of
      '''': q^.data.list^.data.opcode := _quote;
      '`' : q^.data.list^.data.opcode := _backquote;
      ',' : q^.data.list^.data.opcode := _comma;
      '@' : q^.data.list^.data.opcode := _at
     end;
 q^.data.list^.next := p;
 p := q
end;

{---------------------------------------}
PROCEDURE GetCharacter (var Scanned: PCellList);
begin
 match ('#'); match ('\');
 scanned^.data.flag := _char;
 scanned^.data.ch := look;
 GetChar
end;

{---------------------------------------}
PROCEDURE Scan (var Scanned: PCellList);
var k: integer; quote: char;
begin
 if exit_code
    then begin
          SkipWhite;
          while Look = Comment do SkipComment;
          new (Scanned);
          Scanned^.Next := nil;
          if look = '('
             then GetList (Scanned)
             else if IsAlpha(Look)
                     then GetName (Scanned)
                     else if IsDigit(Look)
                             then GetNum (Scanned)
                             else if Look in ['''','`',',']
                                     then begin
                                           quote := look;
                                           GetChar;
                                           if Look = '@'
                                              then begin
                                                    if quote <> ','
                                                       then error('"," only'+
                                                            ' in ",@"')
                                                       else begin
                                                             quote := '@';
                                                             getchar
                                                            end
                                                   end;
                                           Scan (Scanned);
                                           AddQuote(Scanned,quote)
                                          end
                                     else if look ='#'
                                          then GetCharacter (Scanned)
                                          else if look ='"'
                                                  then GetString(Scanned)
                                                  else GetChar
             end
end;

{---------------------------------------}
begin
 look := ' '
end.

