/*
* Disc I/O routines for FDISK
*/

#include "fdisk.h"

extern struct drive_map **Map;
extern int Drive;

/******************************************************
* Read a sector from the selected drive               *
* Provides a front-end to the do_read_sector function *
* Drive should have bit 7 set for hard discs          *
******************************************************/

int read_sector(UINT drive, UINT head, UINT cylinder, UINT sector, UCHAR *buffer)
{
	return(do_read_sector((head * 256) + drive, tocylsec(cylinder, sector), buffer));
}


/*****************************************************
* Read a sector from the selected drive              *
* Usually called from read_sector() which has a more *
* understandable list of paramaters.                 *
*****************************************************/

do_read_sector(drvhd, cylsec, buffer) asm
{
		PUSH	DS			; Save DS
		POP		ES			; Set ES
		MOV		BX,4[BP]	; Get buffer
		MOV		CX,6[BP]	; Get sector & cylinder numbers
		MOV		DX,8[BP]	; Get drive & head numbers
		MOV		AX,0201H	; Read 1 sector
		INT		13h			; Call BIOS
		JC		drs_quit	; Error occurred
		MOV		AX,0		; Successful => return 0
drs_quit:	NOP					; Return
}


/*****************************
* Write a sector out to disc *
*****************************/

int write_sector(UINT drive, UINT head, UINT cylinder, UINT sector, UCHAR *buffer)
{
	return(do_write_sector((head * 256) + drive, tocylsec(cylinder, sector), buffer));
}


/******************************************************
* Write a sector to the selected drive                *
* Usually called from write_sector() which has a more *
* understandable list of paramaters.                  *
******************************************************/

do_write_sector(drvhd, cylsec, buffer) asm
{
		PUSH	DS			; Save DS
		POP		ES			; Set ES
		MOV		BX,4[BP]	; Get buffer
		MOV		CX,6[BP]	; Get sector & cylinder numbers
		MOV		DX,8[BP]	; Get drive & head numbers
		MOV		AX,0301H	; Write 1 sector
		INT		13h			; Call BIOS
		JC		dws_quit	; Error occurred
		MOV		AX,0		; Successful => return 0
dws_quit:	NOP					; Return
}

/****************************************
* Ask the BIOS for the drive parameters *
****************************************/

get_drive_params(drive, buffer) asm
{
		MOV		SI,4[BP]	; Get buffer
		MOV		DX,6[BP]	; Get drive number (80h = HD0)
		MOV		AH,48H		; Function number
		INT		13h			; Call BIOS
		JC		gdp_quit	; Error occurred
		MOV		AX,0		; Successful => return 0
gdp_quit:	NOP					; Return
}


/*
* Extract a sector number from a cylsec value
*/

UINT cylsectosec(WORD cylsec)
{
	return(cylsec % 64);
}


/*
* Extract a cylinder number from a cylsec value
*/

UINT cylsectocyl(WORD cylsec)
{
	return((cylsec * 4) & 768) + (cylsec / 256);
}


/*
* Combine sector and cylinder numbers into a cylsec value
*/

WORD tocylsec(UINT cylinder, UINT sector)
{
	return(((cylinder & 255) << 8) | ((cylinder & 768) >> 2) | sector);
}


/************************************************************
* Calculate the logical sector                              *
* = (sector - 1) + (head * spt) + (cyl * spt * total_heads) *
************************************************************/

int get_num_secs(UINT head, UINT cyl, UINT sec, char *total)
{
	char temp1[LSIZE],
		 temp2[LSIZE],
		 one[LSIZE];

	longset(one, 1);
	longset(total, sec);
	longsub(total, one);
	longset(temp1, head);
	longmul(temp1, Map[Drive]->dp.spt);
	longadd(total, temp1);
	longset(temp1, cyl);
	longmul(temp1, Map[Drive]->dp.spt);
	longcpy(temp2, Map[Drive]->dp.heads);	/* Plus 1, as it starts at zero */
	longadd(temp2, one);
	longmul(temp1, temp2);
	longadd(total, temp1);

	return(0);
}


/*
* Fill in the correct values into the partition table *
*/

make_part(int drive, UINT mbr_no, UINT tab_no, struct par_details details, UINT type)
{
	char secs_high[LSIZE],
		 secs_low[LSIZE],
		 secs_mbr[LSIZE],
		 temp[LSIZE],
		 two[LSIZE];

	Map[drive]->mbr[mbr_no]->table[tab_no].status = INACTIVE;
	Map[drive]->mbr[mbr_no]->table[tab_no].head_start = details.start_head;
	Map[drive]->mbr[mbr_no]->table[tab_no].cylsec_start = tocylsec(details.start_cyl, details.start_sec);
	Map[drive]->mbr[mbr_no]->table[tab_no].head_end = details.end_head;
	Map[drive]->mbr[mbr_no]->table[tab_no].cylsec_end = tocylsec(details.end_cyl, details.end_sec);

	get_num_secs(details.start_head + 1, details.start_cyl + 1, details.start_sec, secs_low);
	get_num_secs(details.end_head + 1, details.end_cyl + 1, details.end_sec, secs_high);
	get_num_secs(details.mbr_head + 1, details.mbr_cyl + 1, details.mbr_sec, secs_mbr);

	longcpy(Map[drive]->mbr[mbr_no]->table[tab_no].offset, secs_low);
	longsub(Map[drive]->mbr[mbr_no]->table[tab_no].offset, secs_mbr);

	/* Total sectors = high sector number - low sector number + 1 */
	longcpy(Map[drive]->mbr[mbr_no]->table[tab_no].total_secs, secs_high);
	longsub(Map[drive]->mbr[mbr_no]->table[tab_no].total_secs, secs_low);
	longset(temp, 1);
	longadd(Map[drive]->mbr[mbr_no]->table[tab_no].total_secs, temp);

	/* Determine type of DOS partition */
	if (type == UNUSED)
	{
		longset(two, 2);
		longset(temp, 32768);
		if (longcmp(Map[drive]->mbr[mbr_no]->table[tab_no].total_secs, temp) == -1)
			type = DOS12;
		else
		{
			longmul(temp, two);
			if (longcmp(Map[drive]->mbr[mbr_no]->table[tab_no].total_secs, temp) == -1)
				type = DOS16;
			else
				type = BIGDOS;
		}
	}

	Map[drive]->mbr[mbr_no]->table[tab_no].type = (UCHAR)type;

	return(0);
}


/*****************
* Erase a sector *
*****************/

erase_sector(UINT drive, UINT head, UINT cyl, UINT sec)
{
	UCHAR buffer[512];

	memset(buffer, 0, 512);
	write_sector(drive + 0x80, head, cyl, sec, buffer);
	return(0);
}
