/*
* A replacement FDISK
* Written for the FDOS project
* By Chris Stratford
* v0.1	16.7.95	- Early experiments
* v0.2	29.7.95 - Started to look as though it might work
* v0.3	29.8.95 - Actually did appear to work
* v0.4	21.9.95 - First alpha release
*/

#include "fdisk.h"
#include "bootcode.h"

int No_drives,						/* Number of installed hard drives */
	Drive,							/* Current drive letter */
	No_partitions,					/* Number of partitions */
	Selected;						/* Currently selected partition */
struct drive_map **Map;				/* Full drive details */
struct del_list *Delhd,				/* Start of unwanted sector list */
				*Del;				/* Current point on list */

char *Cyl_form[] = { 22 << 8 | 2,	/* 22 x 2 form */
					 "\x01\x00\x84Start cylinder:",
					 "\x01\x01\x84End cylinder  :",
					 0 };

/****************
* Main function *
****************/

main()
{
	Drive = 0;		/* Default start drive */
	Selected = 1;	/* Default selected partition */
	init();

	do
		display_partition_info();
	while (get_choice() == 0);

	closedown();
	return(0);
}


/********************
* Initialize things *
********************/

init()
{
	int drive,
		partn,
		extended,
		mbr_count;

	UINT head,
		 cyl,
		 sec;

	struct mbrec buffer;

	vopen();
	vcursor_off();

	/* Initialise the list of deleted sectors */
	if ((Delhd = calloc(sizeof(struct del_list))) == NULL)
		w_abort("Not enough memory - aborting [init - 1]");
	Delhd->next = NULL;
	Del = Delhd;

	/* Get number of available drives */
	No_drives = peek(0x40, 0x75);

	if ((Map = calloc(No_drives * sizeof(Map))) == NULL)
		w_abort("Not enough memory - aborting [init - 2]");

	/* Get information about them */
	for (drive = 0; drive < No_drives; drive++)
	{
		if ((Map[drive] = calloc(sizeof(struct drive_map))) == NULL)
			w_abort("Not enough memory - aborting [init - 3]");
		Map[drive]->dp.size = sizeof(struct drive_params);

		/* Get the drive parameters */
		get_drive_params(drive + 0x80, Map[drive]->dp);

		/* Make space for the master mbr table */
		mbr_count = 2;	/* Master MBR and a terminator */
		if ((Map[drive]->mbr = calloc(mbr_count * sizeof(char *))) == NULL)
			w_abort("Not enough memory - aborting [init - 4]");
		if ((Map[drive]->mbr[mbr_count - 2] = calloc(sizeof(struct mbrec))) == NULL)
			w_abort("Not enough memory - aborting [init - 5]");
		Map[drive]->mbr[mbr_count - 1] = NULL;

		/* Get the master boot record */
		read_sector(drive + 0x80, 0, 0, 1, Map[drive]->mbr[0]);

		/* Perhaps check for a valid boot signature here? */

		/* Search for extended partition tables */
		extended = -1;

		for (partn = 0; partn < 4; partn++)
			if (Map[drive]->mbr[mbr_count - 2]->table[partn].type == EXTEND)
				extended = partn;

		while (extended >= 0)
		{
			head = Map[drive]->mbr[mbr_count - 2]->table[extended].head_start;
			cyl = cylsectocyl(Map[drive]->mbr[mbr_count - 2]->table[extended].cylsec_start);
			sec = cylsectosec(Map[drive]->mbr[mbr_count - 2]->table[extended].cylsec_start);

			read_sector(Drive + 0x80, head, cyl, sec, buffer);

			if (buffer.table[0].type != UNUSED || buffer.table[1].type != UNUSED)
			{
				mbr_count++;

				/* Allocate some space for the new table */
				if ((Map[drive]->mbr = realloc(Map[drive]->mbr, mbr_count * sizeof(char *))) == NULL)
					w_abort("Not enough memory - aborting [init - 6]");
				if ((Map[drive]->mbr[mbr_count - 2] = calloc(sizeof(struct mbrec))) == NULL)
					w_abort("Not enough memory - aborting [init - 7]");
				Map[drive]->mbr[mbr_count - 1] = NULL;

				memcpy(Map[drive]->mbr[mbr_count - 2], buffer, sizeof(struct mbrec));
			}

			extended = -1;	/* Reset, ready for the next one */

			for (partn = 0; partn < 2; partn++)
				if (buffer.table[partn].type == EXTEND)
					extended = partn;
		}
	}

	wclose();

	return(0);
}


/*****************************
* Tidy-up before exiting     *
* Free any allocated memory  *
* Write out the MBRs to disc *
*****************************/

closedown()
{
	int	mbrnum,
		ext_par;
	UINT drive,
		 head,
		 cyl,
		 sec;
	char saveit;

	vcursor_line();
	vclscr();

	if (tolower(message_box("Save this partition layout?")) == 'y')
		saveit = TRUE;
	else
		saveit = FALSE;

	/* Delete any partitions on the deletions list */
	if (saveit == TRUE)
	{
		Del = Delhd;
		while (Del->next != NULL)
		{
			erase_sector(Del->drive, Del->head, Del->cyl, Del->sec);
			Del = Del->next;
		}
	}

	for (drive = 0; drive < No_drives; drive++)
	{
		mbrnum = 0;
		head = 0;	/* Primary MBR is always here */
		cyl = 0;
		sec = 1;
		while (Map[drive]->mbr[mbrnum] != NULL)
		{
			/* Write out the MBR if told to do so */
			if (saveit == TRUE)
			{
				printf("Writing. Dv: %d Hd: %d Cyl: %d Sec: %d\n", drive + 0x80, head, cyl, sec);
				write_sector(drive + 0x80, head, cyl, sec, Map[drive]->mbr[mbrnum]);
			}

			/* Get the location of the next one */
			for (ext_par = 0; (Map[drive]->mbr[mbrnum]->table[ext_par].type != EXTEND) && (ext_par < 4); ext_par++);
			head = Map[drive]->mbr[mbrnum]->table[ext_par].head_start;
			cyl = cylsectocyl(Map[drive]->mbr[mbrnum]->table[ext_par].cylsec_start);
			sec = cylsectosec(Map[drive]->mbr[mbrnum]->table[ext_par].cylsec_start);

			/* Remove this one from memory */
			free(Map[drive]->mbr[mbrnum++]);
		}
		free(Map[drive]->mbr);
		free(Map[drive]);
	}

	free(Map);

	return(0);
}


/*******************************
* Create a new partition entry *
*******************************/

create_partition()
{
	UINT part,		/* Counter */
		 type,		/* Type of partition to create */
		 mbr_num,	/* Which MBR to add the partition to */
		 ext_pos;	/* Position of extended partition in table (0-3) */

	int used_parts,	/* Number of partition slots in use */
		 free_part,	/* Free slot in partition table */
		 dos_part,	/* DOS entry in partition table */
		 ext_part;	/* Extended partition entry in table */

	char title[30],	/* title string for window */
		 new_disc;	/* True if primary partition is empty */

	struct par_details details;		/* Details of partition to be created */

	struct WINDOW *whnd;			/* Window handle */

	/* Find a free slot - check the primary partition table first */
	free_part = dos_part = ext_part = -1;
	used_parts = 0;

	for (part = 0; part < 4; part++)
	{
		type = Map[Drive]->mbr[0]->table[part].type;

		if (type != UNUSED)
			used_parts++;

		/* Is there already a DOS partition? */
		if ((type == DOS12) || (type == DOS16) || (type == BIGDOS))
			dos_part = part;

		/* Is there an extended partition? */
		if (type == EXTEND)
			ext_part = part;

		/* What's the first free space? */
		if ((type == UNUSED) && (free_part < 0))
			free_part = part;
	}

	/* Are there any partitions already set? */
	if (used_parts == 0)
		new_disc = TRUE;
	else
		new_disc = FALSE;

	/* Is there space for a new partition? */
	if ((ext_part == -1) && (free_part == -1)) 
	{
		message_box("No space available for new partition!");
		return(-1);	/* No */
	}

	/* If we've already got a DOS partition, but no extended partition... */
	if ((dos_part != -1) && (ext_part == -1))
	{
		strcpy(title, " Creating extended partition:");
		type = EXTEND;
	}
	else
	{
		strcpy(title, "   Creating DOS partition:");
		type = UNUSED;		/* Not sure what type just yet */
	}

	/* Make a nice window around the form */
	whnd = wopen(24, 10, 32, 5, WCOPEN | WBOX2 | WIN_COL);
	wputs(title);

	/* Ask for start and end cylinders (form) */
	details.start_cyl = 0;
	details.end_cyl = 0;
	wform(29, 12, WCOPEN | WIN_COL, Cyl_form, &details.start_cyl, &details.end_cyl);
	wclose();

	/* If no values have been entered, just forget about it */
	if (details.start_cyl == 0 && details.end_cyl == 0)
		return(0);

	details.start_head = 0;
	details.start_sec = 1;
	details.end_head = ltoui(Map[Drive]->dp.heads);
	details.end_sec = ltoui(Map[Drive]->dp.spt);

	/* If we're missing either a primary DOS partition or a extended partition... */
	if ((dos_part == -1) || (ext_part == -1))
	{
		/* Create a primary table entry */
		details.mbr_head = 0;
		details.mbr_cyl = 0;
		details.mbr_sec = 1;

		details.start_head++;	/* To allow room for the mbr */

		make_part(Drive, 0, free_part, details, type);

		/* Add the boot code and signature if this is a new disc */
		if (new_disc == TRUE)
		{
			memcpy(Map[Drive]->mbr[0]->code, Bootcode, sizeof(Bootcode));
			Map[Drive]->mbr[0]->signature = 0xaa55;			
		}
	}
	else
	{
		/* Otherwise create a new extended partition and DOS partition
		   inside it */

/***************************************************************
* Can't just add to the end of the partition list              *
* We must be able to insert into the middle of a list          *
* if that's where the partition belongs, but for the moment... *
***************************************************************/

		/* Find end of partition list */
		for (mbr_num = 0; Map[Drive]->mbr[mbr_num] != NULL; mbr_num++);

		/* Add an extended partition entry to the previous mbr (unless that's the primary) */
		if (mbr_num > 1)
		{
			ext_pos = (Map[Drive]->mbr[mbr_num - 2]->table[0].type == EXTEND) ? 0 : 1;
			details.mbr_head = Map[Drive]->mbr[mbr_num - 2]->table[ext_pos].head_start;
			details.mbr_cyl = cylsectocyl(Map[Drive]->mbr[mbr_num - 2]->table[ext_pos].cylsec_start);
			details.mbr_sec = cylsectosec(Map[Drive]->mbr[mbr_num - 2]->table[ext_pos].cylsec_start);

			free_part = (Map[Drive]->mbr[mbr_num - 1]->table[0].type == UNUSED) ? 0 : 1;

			make_part(Drive, mbr_num - 1, free_part, details, EXTEND);
		}

		/* Allocate some space for a new mbr record */
		if ((Map[Drive]->mbr = realloc(Map[Drive]->mbr, (mbr_num + 2) * sizeof(char *))) == NULL)
			w_abort("Not enough memory - aborting [7]");
		if ((Map[Drive]->mbr[mbr_num] = calloc(sizeof(struct mbrec))) == NULL)
			w_abort("Not enough memory - aborting [8]");
		Map[Drive]->mbr[mbr_num + 1] = NULL;

		/* Add the new DOS partition */

		for (part = 0; part < 4; part++)
			if (Map[Drive]->mbr[mbr_num - 1]->table[0].type == EXTEND)
				ext_part = part;
		details.mbr_head = Map[Drive]->mbr[mbr_num - 1]->table[ext_pos].head_start;
		details.mbr_cyl = cylsectocyl(Map[Drive]->mbr[mbr_num - 1]->table[ext_pos].cylsec_start);
		details.mbr_sec = cylsectosec(Map[Drive]->mbr[mbr_num - 1]->table[ext_pos].cylsec_start);

		details.start_head++;	/* To leave enough space for the mbr */

		clear_partition(Map[Drive]->mbr[mbr_num]->table[0]);
		clear_partition(Map[Drive]->mbr[mbr_num]->table[1]);
		clear_partition(Map[Drive]->mbr[mbr_num]->table[2]);
		clear_partition(Map[Drive]->mbr[mbr_num]->table[3]);

		make_part(Drive, mbr_num, 0, details, type);	/* We know position 0 is free, so use it */

		/* Since this is a new mbr, we need to add the signature as well */
		Map[Drive]->mbr[mbr_num]->signature = 0xaa55;
	}

	return(0);
}


/************************************************
* Make the selected partition active (bootable) *
************************************************/

set_active_partition()
{
	int count;

	for (count = 0; count < 4; count++)
		if (count == Selected)
			Map[Drive]->mbr[0]->table[count].status = ACTIVE;
		else
			Map[Drive]->mbr[0]->table[count].status = INACTIVE;

	return(0);
}


/*********************
* Remove a partition *
*********************/

delete_partition()
{
	int partn,
		mbrnum,
		parcount,
		mbr_needed,		/* MBR containing the partition we want */
		par_needed,		/* Position in partition table */
		tabnum,
		count;

	char ignore;		/* Flag */

	struct partable *buffer;

	/* Find the mbr containing the partition chosen */
	mbrnum = 0;
	parcount = 1;
	while (Map[Drive]->mbr[mbrnum] != NULL)
	{
		for (partn = 0; partn < 4; partn++)
		{
			if (parcount == Selected)
			{
				par_needed = partn;
				mbr_needed = mbrnum;
			}

			if (Map[Drive]->mbr[mbrnum]->table[partn].type != UNUSED)
				if ((Map[Drive]->mbr[mbrnum]->table[partn].type != EXTEND) || (mbrnum == 0))
					parcount++;
		}
		mbrnum++;
	}

	/* Delete it */

	buffer = Map[Drive]->mbr[mbr_needed]->table[par_needed];

	/* Normal partitions can just be removed */
	if (buffer->type != EXTEND)
	{
		clear_partition(buffer);
		No_partitions--;
	}
	else
		/* Extended partitions can only be removed if they are empty */
		if (Map[Drive]->mbr[mbr_needed + 1] == NULL)
		{
			if (mbr_needed == 0)
				No_partitions--;
			clear_partition(buffer);
		}
		else
			message_box("Extended partition is still in use");

	/* Now check that we don't have any unwanted extended partitions left */

	for (mbrnum--; mbrnum > 0; mbrnum--)
	{
		ignore = FALSE;

		/* Table entry contains an extended partition but there's nothing in it */
		/* So clear the extended partition entry */
		for (tabnum = 0; tabnum < 2; tabnum++)
		{
			if ((Map[Drive]->mbr[mbrnum]->table[tabnum].type == EXTEND) &&
				(Map[Drive]->mbr[mbrnum + 1] == NULL))
			{
				clear_partition(Map[Drive]->mbr[mbrnum]->table[tabnum]);
			}
		}

		/* Nothing left in the table, so we can remove it */
		if ((Map[Drive]->mbr[mbrnum]->table[0].type == UNUSED) &&
			(Map[Drive]->mbr[mbrnum]->table[1].type == UNUSED))
		{
			add_to_del_list(Drive, mbrnum);
			free(Map[Drive]->mbr[mbrnum]);
			Map[Drive]->mbr[mbrnum] = NULL;
			Map[Drive]->mbr = realloc(Map[Drive]->mbr, (mbrnum  + 1) * sizeof(char *));
			ignore = TRUE;	/* Ignore the rest of the tests */
		}

		/* Table entry contains 1 extended partition and 1 unused */
		/* Check for both 0=EXT, 1=unused and 0=unused, 1=EXT */
		/* Ignore the first extended partition, as this is a special case */
		for (tabnum = 0; (tabnum < 2) && (ignore == FALSE); tabnum++)
		{
			if ((Map[Drive]->mbr[mbrnum]->table[tabnum].type == EXTEND) &&
				(Map[Drive]->mbr[mbrnum]->table[(tabnum + 1) % 2].type == UNUSED) &&
				(mbrnum != 1))
			{
				/* Copy the table entry to the previous mbr then delete it */
				for (partn = 0; Map[Drive]->mbr[mbrnum - 1]->table[partn].type != EXTEND; partn++);

				memcpy(Map[Drive]->mbr[mbrnum - 1]->table[partn], Map[Drive]->mbr[mbrnum]->table[tabnum], sizeof(struct partable));

				add_to_del_list(Drive, mbrnum);
				free(Map[Drive]->mbr[mbrnum]);

				/* Shuffle the remaining entries down */
				for (count = mbrnum; Map[Drive]->mbr[count] != NULL; count++)
					Map[Drive]->mbr[count] = Map[Drive]->mbr[count + 1];

				Map[Drive]->mbr = realloc(Map[Drive]->mbr, count * sizeof(char *));
				ignore = TRUE;	/* Ignore the rest of the tests */
			}
		}
	}

	if ((Selected > No_partitions) && (Selected != 1))
		Selected--;

	return(0);
}


/********************************************
* Show the complete partition table details *
********************************************/

display_partition_info()
{
	int partn,
		mbrnum,
		pri_y,		/* Cursor line in primary window */
		ext_y,		/* Cursor line in extended window */
		y;			/* Current cursor line */
	struct mbrec *buffer;
	struct WINDOW *whole,	/* The entire screen */
				  *pri_win,	/* Window to display the primary partitions */
				  *ext_win,	/* And one to display the extended partitions */
				  *display;	/* Holds the current window */

	No_partitions = 0;

	/* Set up the screen */
	whole = wopen(0, 0, 80, 25, WCOPEN | WIN_COL);
	pri_win = wopen(0, 1, 80, 7, WBOX2 | WCOPEN | WIN_COL);
	ext_win = wopen(0, 8, 80, 17, WBOX2 | WCOPEN | WIN_COL);

	w_gotoxy(0, 0, whole);
	w_printf(whole, "Drive number %d", Drive);

	w_gotoxy(2, 1, whole);
	w_printf(whole, "Primary partitions:");
	w_gotoxy(0, 0, pri_win);
	w_printf(pri_win, "Partn   Status   System   StCyl   EndCyl   Offset   Size");
	pri_y = y = 1;

	w_gotoxy(2, 8, whole);
	w_printf(whole, "Extended partitions:");
	w_gotoxy(0, 0, ext_win);
	w_printf(ext_win, "Partn   Status   System   StCyl   EndCyl   Offset   Size");
	ext_y = 1;

	/* And display something in it */
	mbrnum = 0;
	display = pri_win;		/* We'll always be displaying this one first */
	while (Map[Drive]->mbr[mbrnum] != NULL)
	{
		buffer = Map[Drive]->mbr[mbrnum];

		/* Choose which window to display to and the cursor position */
		if (mbrnum == 0)
		{
			if (display == pri_win)
				pri_y = y;
			else
			{
				ext_y = y;
				y = pri_y;
			}
			display = pri_win;
		}
		else
		{
			if (display == ext_win)
				ext_y = y;
			else
			{
				pri_y = y;
				y = ext_y;
			}
			display = ext_win;
		}

		for (partn = 0; partn < 4; partn++)
		{
			if (buffer->table[partn].type != 0)
			{
				if ((mbrnum == 0) || (buffer->table[partn].type != EXTEND))
				{
					w_gotoxy(0, y++, display);
					No_partitions++;
					if (Selected == No_partitions)
						display->WINattr = SEL_COL;
					else
						display->WINattr = WIN_COL;
					w_printf(display, "  %-7d", No_partitions);				/* partition number */
					display_status(display, buffer->table[partn].status);
					display_type(display, buffer->table[partn].type);
					w_printf(display, " %-8d", cylsectocyl(buffer->table[partn].cylsec_start));	/* Start cylinder */
					w_printf(display, "%-8d", cylsectocyl(buffer->table[partn].cylsec_end));	/* End cylinder */
					longprint(display, buffer->table[partn].offset);
					longprint(display, buffer->table[partn].total_secs);		/* Size (in sectors) */
				}
			}
		}
		mbrnum++;
	}

	w_close(ext_win);
	w_close(pri_win);
	w_close(whole);

	return(0);
}


/**************************************
* Change which drive we're looking at *
**************************************/

int change_drive()
{
	int new_drive;

	new_drive = -1;

	if (No_drives == 1)
		message_box("There is only one drive present.");
	else
	{
		while (new_drive < 0 || new_drive > No_drives)
		{
			printf("Select drive: (1-%d) ", No_drives);
			new_drive = getchar() - '0';
		}
	}	

	Drive = new_drive;

	return(0);
}


/**************************************
* See what the user wants to do next. *
*									  *
* Choices are:						  *
*  INS - Add a new partition		  *
*  DEL - Delete an existing partition *
*  b   - Make partition bootable	  *
*  d   - Change to another drive	  *
*  ESC - Quit program				  *
**************************************/

get_choice()
{
	int input;

	input = INVALID;

	while (input == INVALID)
	{
		input = wait_key();

		switch (input)
		{
			case 'b':
			case 'B':	input = MAKE_BOOTABLE;
						set_active_partition();
						break;

			case 'd':
			case 'D':	input = CHANGE_DRIVE;
						change_drive();
						break;

			case _KIN:	input = CREATE;
						create_partition();
						break;

			case _KDL:	input = DELETE;
						delete_partition();
						break;

			case _KUA:	input = UP;
						if (Selected > 1)
							Selected--;
						break;

			case _KDA:	input = DOWN;
						if (Selected < No_partitions)
							Selected++;
						break;

			case 0x1b:	/* Esc - Quit program */
						return(-1);

			default:	input = INVALID;
						break;
		}
	}

	return(0);
}


/*************************
* Wipe a partition entry *
*************************/

clear_partition(struct partable table)
{
	memset(table, 0, sizeof(struct partable));

	return(0);
}


/******************************************************
* Something's gone wrong (probably run out of memory) *
* Tell the user, then quit.                           *
* Not a very neat way to deal with this, but it       *
* should work, assuming DOS tidys up the allocated    *
* memory blocks.                                      *
******************************************************/

w_abort(char *message)
{
	message_box(message);
/*	while(vtstc() == 0); */
	wclose();

	exit(1);
}


/************************************
* Add a sector to the deletion list *
************************************/

add_to_del_list(UINT drive, int mbrnum)
{
	int ext_par;

	for (ext_par = 0; (Map[drive]->mbr[mbrnum - 1]->table[ext_par].type != EXTEND) && (ext_par < 4); ext_par++);
			
	if ((Del->next = malloc(sizeof(struct del_list))) == NULL)
		w_abort("Not enough memory - aborting [atdl - 1]");
	Del->head = Map[drive]->mbr[mbrnum - 1]->table[ext_par].head_start;
	Del->cyl = cylsectocyl(Map[drive]->mbr[mbrnum - 1]->table[ext_par].cylsec_start);
	Del->sec = cylsectosec(Map[drive]->mbr[mbrnum - 1]->table[ext_par].cylsec_start);
	Del = Del->next;
	Del->next = NULL;

	return(0);
}
