/*
* Output routines for FDISK
*/

#include "fdisk.h"

extern char Longreg[];

char *Type[] = {"UNUSED   ", "DOS-12   ", "XENIX    ", "XENIX    ", "DOS-16   ",
				"EXTEND   ", "BIGDOS   ", "HPFS     "};

/*
* Print a long number to the specified window
* padded to 8 characters
*/

longprint(struct WINDOW *display, char *number)
{
	char outstr[11];	/* Output string */
	char numcopy[LSIZE]	/* Copy of input number */
	char base[LSIZE];	/* Number base */
	int position;		/* Position in output string */

	longset(base, 10);
	longcpy(numcopy, number);
	position = 0;		/* Start position in string */

	while (longtst(numcopy) != 0)
	{
		longdiv(numcopy, base);
		outstr[position++] = Longreg[0] + '0';
	}
	outstr[position] = '\0';	/* Terminate string */

	/* Now reverse the string */
	strrev(outstr);

	/* And print it */
	w_printf(display, "%-8s", outstr);

	return(0);
}


/***********************************************
* Convert a long number to an unsigned integer *
***********************************************/

ltoui(char *number)
{
	return(number[1] * 256 + number[0]);
}


/********************************************
* Open a window and display a message in it *
* Window will be centred on the screen      *
* Tabs in the message will be ignored       *
********************************************/

int message_box(char *message)
{
	int width,
		height,
		x, y,
		count,
		ccount,
		key;

	struct WINDOW *handle;

	ccount = width = height = 0;

	/* Establish size of message */
	for (count = 0; count <= strlen(message); count++)
	{
		if ((message[count] == '\n') || (message[count] == '\0'))
		{
			width = max(ccount, width);
			ccount = 0;
			height++;
		}
		else 
			ccount++;
	}

	width += 4;		/* Adjust to allow for a border */
	height += 2;

	x = (SCR_WIDTH - width) / 2;
	y = (SCR_HEIGHT - height) / 2;

	handle = wopen(x, y, width, height, WSAVE | WCOPEN | WBOX2 | B_RED | F_GREEN | F_BLUE | F_RED);

	x = 1;	/* Start coordinate */
	y = 0;

	for (count = 0; count < strlen(message); count++)
	{
		if (message[count] == '\n')
		{
			x = 1;
			y++;
		}
		else
		{
			wgotoxy(x++, y);
			wputc(message[count]);
		}
	}

	key = wait_key();

	wclose();

	return(key);
}


/**************************************************
* Display the status field of the partition entry *
**************************************************/

display_status(struct WINDOW *display, UCHAR status)
{
	switch (status)
	{
		case 0:		/* Inactive */
					w_printf(display, "----    ");
					break;

		case 0x80:	/* Active (bootable) */
					w_printf(display, "BOOT    ");
					break;

		default:	/* Invalid value */
					w_printf(display, "ERR     ");
					break;
	}

	return(0);
}


/*****************************
* Display the partition type *
*****************************/

display_type(struct WINDOW *display, UCHAR type)
{
	if (type < 8)
		w_printf(display, "%s", Type[type]);
	else
		w_printf(display, "UNKNOWN  ");

	return(0);
}


/*************************
* Wait for a keypress    *
* Return the key pressed *
*************************/

wait_key()
{
	int input;

	do NOTHING while ((input = vtstc()) == 0);

	return(input);
}
