/*
   filename: WCD.C

   WCD - Powerful chdir for Dos and Unix.

   See file wcd.txt

Author: Erwin Waterlander

Address :

	  Jongemastate 125
	  5655 HS Eindhoven, The Netherlands

E-mail  : waterlan@natlab.research.philips.com
 or     : waterlan@xs4all.nl
WWW     : http://www.xs4all.nl

======================================================================
= Copyright                                                          =
======================================================================
Copyright (C) 1997-1998 Erwin Waterlander

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

=======================================================================
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <memory.h>
#include <dirent.h>
#include <dir.h>
#include <errno.h>
#include <unistd.h>
#include "match.h"

#define DD_MAXDIR 1024
#define DD_MAXDRIVE 3
#define EXIT_OK 0
#define ALT_SW || *argv[i]=='/'
#define TREEFILE "/treedata.wcd"
#define EXTRA_TREEFILE "/extra.wcd"
#define BANFILE "/ban.wcd"
#define ALIASFILE "/alias.wcd"
#define ROOTDIR "/"
#define DIR_SEPARATOR '/'
#define DIR_END '/'
#define ALL_FILES_MASK "*"
#define DIR_CUR	"."
#define DIR_PARENT  ".."
#define DIR_END	'/'

#define MAX_LIST 22   /* max list of perfect match directories */
#define MAX_WILD_LIST 22   /* max list of wild match directories */
#define MAX_EF 10         /* max nr. of extra tree-files */
#define VERSION      "2.0.5"
#define VERSION_DATE "Mar 26 1998"

#ifdef BASH
#  define GO_FILE "/wcd.go"
#endif
#define DD_NORMAL   0x00	/* Normal file, no attributes */
#define DD_MAXFILE	256
#define DD_MAXEXT	256
#define DD_ISDIREC(m)  ((m) & (FA_DIREC | S_IFDIR))
#define WILDCARDS	0x01


/* Global variables */

FILE *outfile;
char filemask[DD_MAXFILE+DD_MAXEXT];
int  attrib = DD_NORMAL;
unsigned dir_files;
char dir[DD_MAXDIR];
char org_dir[DD_MAXDIR];
char best_match[DD_MAXDIR];
char pm[MAX_LIST][DD_MAXDIR];
char wm[MAX_WILD_LIST][DD_MAXDIR];
char no_match =1, perfect_match = 0;
int  perfect_match_nr = 0, wild_match_nr = 0;
char quiet = 1;
const char *default_mask = ALL_FILES_MASK;

typedef struct TDirTag {
	char* dirname;
	struct TDirTag *next;
} TDirEntry;

typedef struct {
	TDirEntry *head, *tail;
} TDirList;

/* Function prototypes */

void finddirs(char *dir);
int  checkban(char *dir, char *banfilename);
int check_double_match(char *dir, int perfect);
void scanfile(char *filename, char *banfilename);
void scanaliasfile(char *filename);
int  strcompare(const char*, const char*, int);
void q_insert(TDirList *list, const char *s);
int  q_remove(TDirList *list, char *s);
void printhelp(void);
int  wcd_get_int(void);
void empty_wcdgo(char *go_file, int changedrive, char *drive);
int SpecialDir(const char *path);
int (*filematch)( const char*, const char*, int );

/* No valid MS-DOS directories start with a period (.),
 * except for .. or ., so anything with a period prefix
 * must be special.
 */
#define SpecialDir(f) (*(f) == '.')

int strcompare(const char *s1,const char * s2,int ignore_case)
{
	return !strcmp(s1, s2);
}

/********************************************************************
 *
 *                    finddirs(dir)
 *
 ********************************************************************/

void finddirs(char* dir)
{
	static struct ffblk fb;       /* file block structure */
	static char tmp[DD_MAXDIR];      /* tmp string buffer */
	int rc;                       /* error code */
	TDirList list;                /* directory queue */
	char *ptr;


	if (dir)
	{
		int len = strlen(dir);
		/* strip ending separator (if present), which DOS doesn't like. */
		if (len > 1 && dir[len-1]==DIR_END) dir[len-1] = 0;
		if (chdir(dir)) return; /* ?err */
	}

	rc = findfirst( default_mask, &fb, attrib | FA_DIREC );
	list.head = list.tail = 0;
	dir_files = 0;

	while (rc==0)
	{
		if (attrib & FA_DIREC  && DD_ISDIREC(fb.ff_attrib) )
		{
			/*  Ignore directory entries starting with '.'
	   *  which includes the current and parent directories.
	   */
			if (!SpecialDir(fb.ff_name))
				q_insert(&list, fb.ff_name);
		}

		/* if match then do something with the file */
		if (filematch(fb.ff_name, filemask, 0))
		{
			if (!dir_files++)
			{
				int len;
				getcwd(tmp, sizeof(tmp));
				len = strlen(tmp);
				if (len==0 || tmp[--len] == DIR_END)
					tmp[len] = '\0';

				/* Change backslashes to DIR_SEPARATOR
		   Avoid doing compares on strings
		   with backslashes. */
/*				while ((ptr=strchr(tmp,'\\'))) * ptr=DIR_SEPARATOR;
  */			if ( (ptr=strstr(tmp,"/")) != NULL)
					fprintf(outfile,"%s\n",ptr);

			}
		}

		rc = findnext(&fb);
	} /* while !rc */

	/* recursively parse subdirectories (if any) */
	while (q_remove(&list, tmp))
		finddirs(tmp);

	if (dir) chdir(DIR_PARENT); /* go to parent directory */
}

/******************************************************************
 *
 *          q_insert - insert directory name to queue
 *
 ******************************************************************/

void q_insert(TDirList *list,const char *s)
{
	TDirEntry *ptr;
	int len = strlen(s);
	if (!len) return;
	if ((ptr = (TDirEntry*) malloc(sizeof(TDirEntry))) == NULL )
	{
		perror("malloc");
		return;
	}
	if ((ptr->dirname = (char*) malloc(len+1)) == NULL )
	{
		perror("malloc");
		free(ptr);
		return;
	}
	strcpy(ptr->dirname, s);
	ptr->next = NULL;
	if (!list->head) list->head = ptr;
	else list->tail->next = ptr;
	list->tail = ptr;
}

/*******************************************************************
 *
 *         q_remove - remove directory name from queue
 *
 *******************************************************************/

int q_remove(TDirList *list,char *s)
{
	TDirEntry *ptr = list->head;
	if (!ptr) return 0;		/* queue empty? */
	strcpy(s, ptr->dirname);
	list->head = ptr->next;
	free(ptr->dirname);
	free(ptr);
	return 1;			/* okay */
}


/********************************************************************
 *
 *                    checkban(char *dir,char *banfilename)
 *
 *  Returns 0 if directory is not banned.
 *  Returns 1 if directory is banned.
 *
 ********************************************************************/

int checkban(char *dir,char *banfilename)
{
	FILE *infile;
	char ban[DD_MAXDIR];
	char temp[DD_MAXDIR];
	int banned = 0;

	/* open treedata-file */
	if  ((infile = fopen(banfilename,"r")) == NULL)
	{
		banned = 0;
	}
	else
	{
		while ((!feof(infile) )&&(banned==0))
		{
			if(fscanf(infile,"%s",ban)==1)
			{
	            if (ban)
	            {
		        	int len = strlen(ban);
		        	/* strip ending separator (if present) */
		        	if (len > 1 && ban[len-1]==DIR_SEPARATOR) ban[len-1] = '\0';
	            }

				if(strlen(dir)>=strlen(ban))
				{
					strncpy(temp,dir,strlen(ban));
					temp[strlen(ban)]='\0';
					if (stricmp(temp,ban)== 0)
					{
						banned = 1;
					}
				}
			}
		} /* while (!feof(infile) ) */
		fclose(infile);
	}
	return(banned);
}
/********************************************************************
 *
 *                    check_double_match(char *dir, int perfect)
 *
 *  Returns 0 if directory had no match before.
 *  Returns 1 if directory is double matched.
 *
 ********************************************************************/

int check_double_match(char *dir, int perfect)
{
	int i = 0,doublematch = 0;
	char *ptr;

	ptr = dir;

	if (ptr)
    if (perfect)
    {
    	if (perfect_match_nr > 0)  /* test for double perfect match */
    	{
     		while( (doublematch==0) && (i < perfect_match_nr))
     		{
        		if( stricmp(pm[i],ptr) == 0) doublematch = 1;
        		i++;
      		}
    	}
    }
    else
    {
    	if (wild_match_nr > 0)  /* test for double wild match */
    	{
     		while( (doublematch==0) && (i < wild_match_nr))
     		{
        		if( stricmp(wm[i],ptr) == 0) doublematch = 1;
        		i++;
     		}
     	}

    }

	return(doublematch);
}

/********************************************************************
 *
 *                    scanfile(char *filename)
 *
 *
 ********************************************************************/

void scanfile(char *filename, char *banfilename)
{
	FILE *infile;
	char line[DD_MAXDIR];
	char help1_str[DD_MAXDIR];
	char c;
	char *line_end,*help_str;
	int  i,j;
	char error = 0;

	if  ((infile = fopen(filename,"r")) == NULL)
	{
		fprintf(stderr,"Error opening file %s.\n",filename);
	}
	else
	{

		if( (help_str = strrchr(org_dir,DIR_SEPARATOR)) != NULL)
		 help_str++;
		else help_str = org_dir;

		if (!dd_iswild(help_str))
			{
			  strcpy(dir,org_dir);
			  strcat(dir,"*");
			}

		while (!feof(infile) )
		{
			j=0;                  /* read a line */
			error = 0;
			for(i=0; ((c=getc(infile)) != '\n') && (!feof(infile)) ;i++)
				if (i < (DD_MAXDIR-1))
				{
					line[i]=c;
					j++;
				}
				else error = 1;
			line[j]='\0';  /* end string */


			if(error)
				printf("Error: line too long ( %d > %d). Fix: increase DD_MAXDIR.\n",i,(DD_MAXDIR-1));

			if (line)
	        {
		       	int len = strlen(line);
		       	/* strip ending separator (if present) */
		       	if (len > 1 && line[len-1]==DIR_SEPARATOR) line[len-1] = '\0';
	        }

			if( (line_end = strrchr(line,DIR_SEPARATOR)) != NULL)
				line_end++;
			else
				line_end = line;

			/* does the search string have a DIR_SEPARATOR? */

			  if( (help_str = strrchr(org_dir,DIR_SEPARATOR)) != NULL)
			    help_str++;
			  else help_str = org_dir;

			  strcpy(help1_str,"*");
			  strcat(help1_str,dir);

		 /* test for a perfect match */
			/* ignore case on a DOS platform */


			if ( (stricmp(line_end,help_str)==0) && (dd_match(line,help1_str,1)) )
			{

				  if(!quiet) printf("%s\n",line);
				  if(perfect_match_nr<=MAX_LIST)
				  if ((checkban(line,banfilename) == 0) && (check_double_match(line,1)==0))
				  {
				  	perfect_match =1;
					if(perfect_match_nr<MAX_LIST) strcpy(pm[perfect_match_nr],line);
					strcpy(best_match,line);
					perfect_match_nr++;
					no_match = 0;
				  }
			}
			else
			{
			  if( (help_str = strrchr(dir,'/')) != NULL)
			    help_str++;
			  else help_str = dir;


			/* test for a wild match if no perfect match */
			/* ignore case on a DOS platform */

			if ( (dd_match(line_end,help_str,1)) && (dd_match(line,help1_str,1)) && !perfect_match)
			{
				  if(!quiet) printf("%s\n",line);
				  if(wild_match_nr<=MAX_WILD_LIST)
	              if((checkban(line,banfilename) == 0) && (check_double_match(line,0)==0))
	              {
		              if(wild_match_nr<MAX_WILD_LIST) strcpy(wm[wild_match_nr],line);
		              strcpy(best_match,line);
		              wild_match_nr++;
		              no_match = 0;
				  }
			}
			}

		}   /* while (!feof(infile) ) */
		fclose(infile);
	}


}
/********************************************************************
 *
 *                    scanaliasfile(char *filename)
 *
 *
 ********************************************************************/

void scanaliasfile(char *filename)
{
	FILE *infile;
	char line[DD_MAXDIR];
	char alias[256];

	/* open treedata-file */
	if  ((infile = fopen(filename,"r")) != NULL)
	{

		while (!feof(infile) )
		{

			if(fscanf(infile,"%s %s",alias,line)==2)
			{


			/* Only a perfect match counts, case sensitive */
				if  (strcmp(alias,org_dir)==0)
					{
						if(!quiet) printf("%s\n",line);

				/*		if(perfect_match_nr<=MAX_LIST)
							if (checkban(dir,banfilename) == 0)
							{    */
								perfect_match =1;
								if(perfect_match_nr<MAX_LIST) strcpy(pm[perfect_match_nr],
								    line);
								strcpy(best_match,line);
								perfect_match_nr++;
								no_match = 0;

						/*	} */
					}
			}
		}   /* while (!feof(infile) ) */
	fclose(infile);
	}
}

/********************************************************************
 *
 *                 Get int
 *
 ********************************************************************/

int wcd_get_int()
{
  int i;
  char string[256];

    gets(string);
    fflush(stdin); /* flush the input stream in case of bad input */

    i=atoi(string);

  return(i);
}

/********************************************************************
 *
 *                 Print help
 *
 ********************************************************************/
void printhelp()
{
#ifdef BASH
	printf("WCD  - Waterlander Change Directory %s (32 bit DOS BASH version)\n",VERSION);
#else
	printf("WCD  - Waterlander Change Directory %s (32 bit)\n",VERSION);
#endif
	printf("     - by Erwin Waterlander, %s.   Powerful chdir for Dos and Unix.\n\n",VERSION_DATE);
	printf("Usage: wcd [drive:][dir] [-h] [-q] [-Q] [-b] [-l] [-c] [-v] [-e] [-E <dir>]\n");
	printf("       [-g] [-s] [-S <dir>] [-a] [-A <dir>] [-u <username>] [-f <treefile>]\n");
	printf("  dir (partial) name of directory to change to.\n");
	printf("      Wildcards *, ? and [SET] are supported!\n");
	printf("  -u  add treefile of other User (Unix only)\n");
	printf("  -q  Quiet operation (Unix)  unQuiet operation (DOS)\n");
	printf("  -Q  Quieter operation               -c  direct CD mode\n");
	printf("  -f  add extra treeFile              -l  aLias current dir\n");
	printf("  -h  show this Help                  -b  Ban current path\n");
	printf("  -s  (re)Scan disk from $HOME        -v  print Version info\n");
	printf("  -S  Scan disk from directory        -g  print software licence\n");
	printf("  -a  Add current dir to treedata     -e  add current dir to Extra treedata\n");
	printf("  -A  Add tree from dir               -E  add tree from dir to Extra treedata\n");
	printf("Examples:\n");
	printf("   wcd -s                  wcd src/wcd\n");
	printf("   wcd -S /                wcd doe*/vhdl -u doe\n");
	printf("   wcd w*[1-3]             wcd src -q -f /home/doe/.extra.wcd\n");
	printf("   wcd d:games             wcd -a\n\n");
	printf("e-mail: waterlan@natlab.research.philips.com\n");
#ifdef BASH
	printf("---\nThis version works only in the DJGPP DOS bash\n");
	printf("---\nBourne Again Shell installation:\n");
	printf("Copy wcd.exe to your ~/bin directory.\n");
	printf("Add the following function to your ~/_bashrc file.\n");
	printf("   function wcd\n   {\n   c:/bin/wcd.exe $*\n   source ~/wcd.go\n   }\n");
	printf("Start a new bash and go.\n");
#endif
}
/********************************************************************
 *
 *             empty wcd.go file
 *
 ********************************************************************/

#ifdef BASH
void empty_wcdgo(char *go_file, int changedrive, char *drive)
{

	if  ((outfile = fopen(go_file,"w")) == NULL)
	{
		fprintf(stderr,"Error opening file %s for write.\n", go_file);
		exit(0);
	}
	if(changedrive == 1)
	fprintf(outfile,"cd %s\n",drive);
	else
	fprintf(outfile,"\n");
	fclose(outfile);

}
#endif

/********************************************************************
 *
 *                             MAIN
 *
 ********************************************************************/

int main(int argc, char **argv)
{

	int  i;
	int  flags = 0;
	char *path = 0;
	char rootdir[DD_MAXDIR],treefile[DD_MAXDIR],banfile[DD_MAXDIR],aliasfile[DD_MAXDIR];
	char extratreefile[DD_MAXDIR];
	char scandir[DD_MAXDIR];
	FILE *infile;
	char scan_disk = 0;
	char *ptr;
	char extra_files[MAX_EF][DD_MAXDIR];
	char nr_extra_files = 0;
	int  quieter = 0, cd = 0, scandirectory =0, append =0;
	DIR* dirp; /* GJM */
	int len;
	static char tmp[DD_MAXDIR];      /* tmp string buffer */
	int destDisk;
	char drive[DD_MAXDRIVE];
	char string[256];

#ifdef BASH
	int  j,k;
	char help1_str[DD_MAXDIR];
	char go_file[DD_MAXDIR];
	int changedrive = 0;
	int disk = -1;
	strcpy(go_file,GO_FILE);
#endif
	strcpy(rootdir,ROOTDIR);
	strcpy(treefile,TREEFILE);
	strcpy(extratreefile,EXTRA_TREEFILE);
	strcpy(banfile,BANFILE);
	strcpy(aliasfile,ALIASFILE);

    strcpy(scandir,rootdir);

	if (argc == 1 )
	{
		printhelp();
		return EXIT_OK;
	}

	strcpy(dir,"");
	strcpy(org_dir,"");

 	attrib |= FA_DIREC;  /* set recursive search on */

   /* ---------------------- parse the commandline ------------*/


	for (i=1; i < argc; i++)
	{
		if (*argv[i]=='-') /* is it a switch? */
			switch (argv[i][1]) {
			case 'S':
				  scandirectory = 1 ;
				  scan_disk = 1;
				break;
			case 's':
				  scan_disk = 1;
				break;
			case 'A':
				  append =1;
			      scandirectory = 1 ;
				  scan_disk = 1;
				break;
			case 'a':
				getcwd(tmp, sizeof(tmp));
				if(tmp != NULL)
				{
				  len = strlen(tmp);
				  if (len==0 || tmp[--len] == DIR_END)
					tmp[len] = '\0';

				  /* open the treedata file */
				  if  ((outfile = fopen(treefile,"a")) == NULL)
				  {
				   fprintf(stderr,"Error opening file %s for write.\n", treefile);
				   exit(0);
				  }
				  else
				  {
				   while ((ptr=strchr(tmp,'\\'))) * ptr= DIR_SEPARATOR;
				   if ( (ptr=strstr(tmp,"/")) != NULL)
				   {
				   	fprintf(outfile,"%s\n",ptr);
				    printf("%s added.\n",ptr);
				   }
				  }
				  fclose(outfile);
				}
				break;
			case 'E':
				append =2;
				scandirectory = 1 ;
				scan_disk = 1;
				break;
			case 'e':
				getcwd(tmp, sizeof(tmp));
				if(tmp != NULL)
				{
					len = strlen(tmp);
					if (len==0 || tmp[--len] == DIR_END)
						tmp[len] = '\0';
					if  ((outfile = fopen(extratreefile,"a")) == NULL)
					{
						fprintf(stderr,"Error opening file %s for write.\n", extratreefile);
					}
					else
					{
						while ((ptr=strchr(tmp,'\\'))) * ptr= DIR_SEPARATOR;
				        if ( (ptr=strstr(tmp,"/")) != NULL)
				        {
						  fprintf(outfile,"%s\n",ptr);
						  printf("%s added.\n",ptr);
						}
						fclose(outfile);
					}
				}
				break;
			case 'c':
			case 'C':
				  cd = 1;
				break;
			case 'B':
			case 'b':
				getcwd(tmp, sizeof(tmp));
				if(tmp != NULL)
				{
					len = strlen(tmp);
					if (len==0 || tmp[--len] == DIR_END)
						tmp[len] = '\0';

					/* open the treedata file */
					if  ((outfile = fopen(banfile,"a")) == NULL)
					{
						fprintf(stderr,"Error opening file %s for write.\n", banfile);
					}
					else
					{
						while ((ptr=strchr(tmp,'\\'))) * ptr= DIR_SEPARATOR;
						if ( (ptr=strstr(tmp,"/")) != NULL)
						{
							fprintf(outfile,"%s\n",ptr);
							printf("%s added to banfile %s.\n",ptr,banfile);
						}
						fclose(outfile);
					}
				}
				break;
			case 'L':
			case 'l':
				printf("Enter alias for current directory: ");
				gets(string);
				fflush(stdin); /* flush the input stream in case of bad input */

				if(strcmp(string,"")!=0)
				{

				getcwd(tmp, sizeof(tmp));
				if(tmp != NULL)
				{
					len = strlen(tmp);
					if (len==0 || tmp[--len] == DIR_END)
						tmp[len] = '\0';

					/* open the treedata file */
					if  ((outfile = fopen(aliasfile,"a")) == NULL)
					{
						fprintf(stderr,"Error opening file %s for write.\n", aliasfile);
					}
					else
					{
						while ((ptr=strchr(tmp,'\\'))) * ptr= DIR_SEPARATOR;
						if ( (ptr=strstr(tmp,"/")) != NULL)
						{
							fprintf(outfile,"%s %s\n",string,ptr);
							printf("%s added to aliasfile %s.\n",ptr,aliasfile);
						}
						fclose(outfile);
					}
				}
				}
				break;
			case 'q':
				  quiet = 0;
				break;
			case 'Q':
				  quiet = 1;
				  quieter = 1;
				break;
			case 'v':
			case 'V':
				printf("WCD %s, %s\n",VERSION,VERSION_DATE);
				printf("Powerful chdir for Dos and Unix.\n\n");
				printf("See also file wcd.txt\n\n");
				printf("Download the latest executables and sources from:\n");
				printf("http://www.xs4all.nl/~waterlan/\n");

				break;
			case 'g':
			case 'G':
				printf("WCD %s, %s\n",VERSION,VERSION_DATE);
				printf("Powerful chdir for Dos and Unix.\n\n");
				printf("Copyright (C) 1997-1998 Erwin Waterlander\n");
				printf("Copyright (C) 1994-1996 Jason Mathews on DOSDIR\n");
				printf("Copyright (C) 1990-1992 Mark Adler, Richard B. Wales, Jean-loup Gailly,\n");
				printf("Kai Uwe Rommel and Igor Mandrichenko on recmatch()\n\n");

				printf("This program is free software; you can redistribute it and/or\n");
				printf("modify it under the terms of the GNU General Public License\n");
				printf("as published by the Free Software Foundation; either version 2\n");
				printf("of the License, or (at your option) any later version.\n\n");

				printf("This program is distributed in the hope that it will be useful,\n");
				printf("but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
				printf("MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
				printf("GNU General Public License for more details.\n\n");

				printf("You should have received a copy of the GNU General Public License\n");
				printf("along with this program; if not, write to the Free Software\n");
				printf("Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.\n\n");

				printf("See also file copying.txt and wcd.txt\n");

				break;
			case 'f':
				break;
			default:               /* any switch except the above */
				printhelp();
#ifdef BASH       /* empty wcd.go file */
		empty_wcdgo(go_file,0,drive);
#endif
				return EXIT_OK;
			}
		else /* Not a switch. Must be a dir of filename. */
		{
           if ((strcmp(argv[i-1],"-f") == 0 )
                       && (nr_extra_files < MAX_EF))
           {
           		strcpy(extra_files[nr_extra_files],argv[i]);
                nr_extra_files++;
           }
           else
				if (strcmp(argv[i-1],"-S") == 0 )
				{
				    strcpy(scandir,argv[i]);
				}
				else
				if (strcmp(argv[i-1],"-A") == 0 )
				{
				    strcpy(scandir,argv[i]);
				}
				else
				if (strcmp(argv[i-1],"-E") == 0 )
				{
				    strcpy(scandir,argv[i]);
				}
				else
						 {

						 strcpy(dir,argv[i]);
			             strcpy(org_dir,argv[i]);

		/* Change DIR_SEPARATOR to '/' if a '\' was typed
		   Avoid doing compares on strings
		   with backslashes. */
		                 while ((ptr=strchr(dir,'\\'))) * ptr='/';
		                 while ((ptr=strchr(org_dir,'\\'))) *ptr='/';
						 }
		}
	} /* for */


	/*--- end parsing commandline -----------*/



	if ((cd == 1)&&(strcmp(dir,"")!=0)) /* Try open dir direct. */
	{

		if((dirp=opendir(org_dir)) != NULL) /* GJM */
		{
			closedir(dirp); /* GJM */
			strcpy(best_match,org_dir);
			if (!quieter)
	   			printf("-> %s\n",best_match);

#ifdef BASH
	        j = strlen(best_match);
	        k = 0;
			for (i=0; i < j ; i++)
			{ if ( (best_match[i] == '$') || (best_match[i] == '"') )
	   			{
	   			help1_str[k] = '\\'; k++;
	   			}
	  		help1_str[k] = best_match[i];
	  		k++ ;
			}
			help1_str[k] = '\0' ;

			strcpy(best_match,"\"");
			strcat(best_match,help1_str);
			strcat(best_match,"\"");

			if  ((outfile = fopen(go_file,"w")) == NULL)
			{
				fprintf(stderr,"Error opening file %s for write.\n", go_file);
				exit(0);
			}
			fprintf(outfile,"cd %s\n", best_match);
			fclose(outfile);

#else
	   		chdir(best_match); /* change to directory */
#endif
	  		return EXIT_OK;
		}


	}

	flags |= WILDCARDS;
	strcpy(filemask, default_mask);

	/* set comparison function */
	filematch = (flags & WILDCARDS) ? *dd_match : *strcompare;

	/* is there a drive to go to ? */

	if (strlen(dir)>1)
	{

	strncpy(drive,dir,2);
	drive[DD_MAXDRIVE-1] = '\0';
	if (dd_match(drive,"[a-z]:",1))
	{
		destDisk = islower(*drive) ? *drive-'a' : *drive-'A';
#ifdef BASH
		changedrive = 1;
		disk = getdisk();
#endif
		if (destDisk >= 0)
		{
	   		setdisk(destDisk);
		}
		ptr = dir + 2;
		strcpy(org_dir,ptr);
		strcpy(dir,org_dir);

	}
	}


	/* does treedata-file exist? */
	if  ((infile = fopen(treefile,"r")) == NULL)
		scan_disk = 1;
	else fclose(infile);


	if (scan_disk)     /* (re)Scan the disk? */
	{
		if (scandirectory)
		{
			if((dirp=opendir(scandir)) != NULL)
		    {
				closedir(dirp); /* GJM */
				path=scandir;
			}
			else
			{
  		 		printf("Error: Directory %s does not exist.\n",scandir);
  		 		exit (1);
  		 	}
		}
		else
		path=rootdir;

		printf("Please wait. (re)Scanning disk. Building treedata-file from %s\n",path);

	/* open the output file */
		if (append == 1)
		outfile = fopen(treefile,"a");  /* append to default database */
		else
		if (append == 2)
		outfile = fopen(extratreefile,"a");  /* append to extra database */
		else
		outfile = fopen(treefile,"w");  /* create new database */

		if (!outfile) /* Try to open in a temp dir */
		{
			if  ( (ptr = getenv("TEMP")) != NULL )
			{
			strcpy(treefile,ptr);
			strcat(treefile,TREEFILE);
			outfile = fopen(treefile,"w");
			}

			if (!outfile)
			{
				if  ( (ptr = getenv("TMP")) != NULL )
					{
					strcpy(treefile,ptr);
					strcat(treefile,TREEFILE);
					outfile = fopen(treefile,"w");
					}

				if (!outfile)
				{
					if  ( (ptr = getenv("TMPDIR")) != NULL )
						{
						strcpy(treefile,ptr);
						strcat(treefile,TREEFILE);
						outfile = fopen(treefile,"w");
						}
				}
			}
		}

		if (!outfile) /* Did we succeed? */
		{
			fprintf(stderr,"Error opening treefile for write.\n");
			fprintf(stderr,"Set TEMP environment variable if this is a read-only disk.\n");
			exit(1);
		}

		finddirs( path ); /* Build treedata-file */
		fclose(outfile);

	}

	if (strcmp(dir,"")==0) /* is there a directory to go to? */
	{
#ifdef BASH       /* empty wcd.go file */
         /* restore original disk (if necessary) */
        if (disk >= 0) setdisk(disk);
		empty_wcdgo(go_file,changedrive,drive);
#endif
		return EXIT_OK;
	}
       scanfile(treefile,banfile); /* scan the treedata file */


	   if  ((outfile = fopen(extratreefile,"r")) != NULL)
	   {
		 fclose(outfile);
		 scanfile(extratreefile,banfile); /* scan the extra treedata file */
	   }

       /* search extra files */
       for (i=0;i<nr_extra_files;i++)
       {
         scanfile(extra_files[i],banfile); /* scan the extra treedata file */
       }

	   /* search alias file */
	   scanaliasfile(aliasfile);

	/* search is done */

#ifdef BASH
  /* restore original disk (if necessary) */
  if (disk >= 0) setdisk(disk);
#endif


	if ((perfect_match_nr > 1)&&(perfect_match_nr <= MAX_LIST))
	{
		for (i=0;(i<perfect_match_nr)&&(i<MAX_LIST);i++)
			printf("%d  %s\n",i+1,pm[i]);
		i= -1;
		printf("\nPerfect match for %d directories.\n",perfect_match_nr);
		printf("Please choose one (<Enter> to abort): ");
		i = wcd_get_int();

		if ( (i>0) && (i<=perfect_match_nr))
		{
			i--;
			strcpy(best_match,pm[i]);
		}
		else
		{
#ifdef BASH       /* empty wcd.go file */
		empty_wcdgo(go_file,changedrive,drive);
#endif
		return EXIT_OK;
		}
	}
	else if ((perfect_match==0)&&(wild_match_nr > 1)&&(wild_match_nr <= MAX_WILD_LIST))
	{
		for (i=0;(i<wild_match_nr)&&(i<MAX_WILD_LIST);i++)
			printf("%d  %s\n",i+1,wm[i]);
		printf("\nWild match for %d directories.\n",wild_match_nr);
		printf("Please choose one (<Enter> to abort): ");
		i=wcd_get_int();

		if ( (i>0) && (i<=wild_match_nr))
		{
			i--;
			strcpy(best_match,wm[i]);
		}
		else
		{
#ifdef BASH       /* empty wcd.go file */
		empty_wcdgo(go_file,changedrive,drive);
#endif
		return EXIT_OK;
		}
	} else if ((perfect_match==0)&&(wild_match_nr > MAX_WILD_LIST))
		no_match = 1;


	/*******************************/


	if ((no_match)&&(wild_match_nr > MAX_WILD_LIST))
	{  /* No perfect match, to many wild matches */

		printf("More than %d directories found matching %s\n",MAX_WILD_LIST,dir);
		printf("Try a more unique searchstring.\n");
#ifdef BASH       /* empty wcd.go file */
		empty_wcdgo(go_file,changedrive,drive);
#endif

	}
	else if ((no_match)&&(cd==0)) /* No match at all */
	{

		if((dirp=opendir(org_dir)) == NULL) /* GJM */
		{
			printf("No directory found matching %s\nPerhaps you need to rescan the disk or path is banned.\n",dir);
#ifdef BASH       /* empty wcd.go file */
		empty_wcdgo(go_file,changedrive,drive);
#endif
		}
		else
		{
			closedir(dirp); /* GJM */
			no_match = 0;
			strcpy(best_match,org_dir);
		}


	}else if (no_match)
	  	printf("No directory found matching %s\nPerhaps you need to rescan the disk or path is banned.\n",dir);


	/*******************************/


 	if ((no_match==0)&&(perfect_match_nr > MAX_LIST))
	{  /* Too many perfect matches */

		printf("More than %d directories found matching %s\n",MAX_LIST,org_dir);
		printf("Use subdirectory definition. Eg. wcd src/wcd\n");
#ifdef BASH      /* empty wcd.go file */
		empty_wcdgo(go_file,changedrive,drive);
#endif

	}
	else
    if (no_match==0) /* yes, a match */
	{
	   if (!quieter)
	   printf("-> %s\n",best_match);
#ifdef BASH
	j = strlen(best_match);
	k = 0;
	for (i=0; i < j ; i++)
	{ if ( (best_match[i] == '$') || (best_match[i] == '"') )
	   {
	   help1_str[k] = '\\'; k++;
	   }
	  help1_str[k] = best_match[i];
	  k++ ;
	}
	help1_str[k] = '\0' ;

	strcpy(best_match,"\"");
	strcat(best_match,help1_str);
	strcat(best_match,"\"");

		if  ((outfile = fopen(go_file,"w")) == NULL)
		{
			fprintf(stderr,"Error opening file %s for write.\n", go_file);
			exit(0);
		}

		if (changedrive)
		fprintf(outfile,"cd %s ; cd %s\n", drive, best_match);
		else
		fprintf(outfile,"cd %s\n", best_match);
		fclose(outfile);

#else
	   chdir(best_match); /* change to directory */
#endif
	}
	return EXIT_OK;
}
