/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: FGETS.C $
   $Locker:  $	$Name:  $	$State: Exp $

	int fgets(char *buf, size_t bufsiz, FILE *fp)

	Get characters from a stream and put them into the buffer.
	The get-loop stops if: a) (bufsiz - 1) characters are read,
	b) an '\n' (line end, newline) was read, c) EOF is encountered,
	or an error occured.

	Except in case b) no '\n' character will be placed into the
	buffer.

	A NUL character is placed behind all read characters.

	Input:
		buf != NULL
		fp != NULL; a valid FILE pointer

	Return:
		NULL: 0<=bufsiz<=1 || an error occured || EOF, but no character read
		buf: otherwise

	Note:
		<none>

	Conforms to:
		ANSI

	See also:
		fgetc, getc, fread, fputs

	Target compilers:
		Any C compiler

	Origin:
		1997/11/03 Robert de Bath (see CONTRIB\STDIO2.ZIP)

	Revised by:
		1997/11/23 Steffen Kaiser (ska)

	File Revision:    Revision 1.2  1998/01/29 07:10:01  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include "stdio.h"

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: FGETS.C 1.2 1998/01/29 07:10:01 ska Exp $";
#endif

_CLibFunc char *
fgets(char *s, size_t count, FILE *f)
{
	REG char *p, *end;	/* current & end pointer of buffer s */
	REG int ch;		/* currently read character */

	assert(s != NULL);
	assert(f != NULL);

	switch(count) {
	case 0:			/* nothing to do */
	case 1:			/* nothing to do either, but buffer exists */
		return NULL;
	default:
		end = (p = s) + count;
		if(count != 1) do {
			ch = getc(f);
			if (ch == EOF) {
				if (s == p)			/* nothing read? */
					return NULL;	/* no change of the buffer! */
				break;
			}
		} while((*p++ = (char) ch) != '\n' && p < end);
		*p = '\0';
		break;
	}

	return ferror(f)? NULL: s;
}
