/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: PRINTF.C $
   $Locker:  $	$Name:  $	$State: Exp $

	int printf(char *fmt, ...)
	int sprintf(char *buf, char *fmt, ...)
	int fprintf(FILE *fp, char *fmt, ...)
	int vprintf(char *fmt, va_list ap)
	int vsprintf(char *buf, char *fmt, va_list ap)
	int vfprintf(FILE *fp, char *fmt, va_list ap)

	Translate a format string and arguments into printable characters.
	[v]printf() writes the result into stdout.
	f[v]printf() writes the result into fp.
	s[v]printf() writes the result onto buf. buf must be large enough to
	hold the result.

	Input:
		fmt != NULL
		buf != NULL
		fp != NULL, opened by fopen()

	Return:
		number of characters written

	Note:
		<none>

	Conforms to:
		ANSI

	See also:
		scanf, fopen

	Target compilers:
		Any C compiler

	Origin:
		Robert De Bath (see CONTRIB\STDIO2.ZIP)

	Revised by:
		1997/11/11	Steffen Kaiser (ska)

	File Revision:    Revision 1.2  1998/01/29 07:10:01  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include <stdio.h>
#include <stdlib.h>			/* _ltoa(), _ultoa() */
#include <string.h>

static int					/* shut up warning */
prtfld(FILE *op, uchar_t *buf, int ljustf, uchar_t sign, uchar_t pad, int width
	, int preci, int buffer_mode);

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: PRINTF.C 1.2 1998/01/29 07:10:01 ska Exp $";
#endif

_CLibFunc int
printf(const char * fmt
	, ...)
{
  va_list ptr;
  int rv;

	assert(fmt != NULL);

  va_start(ptr, fmt);
  rv = vfprintf(stdout,fmt,ptr);
  va_end(ptr);
  return rv;
}

_CLibFunc int
sprintf(char *buf
	, const char * fmt
	, ...)
{
static FILE  string[1] =
{
   {0, 0, (char*)(unsigned) -1, 0, (char*) (unsigned) -1, -1,
    _IOFBF | __MODE_WRITE}
};

  va_list ptr;
  int rv;

  assert(buf != NULL);
  assert(fmt != NULL);

  va_start(ptr, fmt);
  string->bufpos = buf;
  rv = vfprintf(string,fmt,ptr);
  va_end(ptr);
  *(string->bufpos) = 0;
  return rv;
}

_CLibFunc int
fprintf(FILE * fp
	, const char * fmt
	, ...)
{
  va_list ptr;
  int rv;

  assert(fp != NULL);
  assert(fmt != NULL);

  va_start(ptr, fmt);
  rv = vfprintf(fp,fmt,ptr);
  va_end(ptr);
  return rv;
}

_CLibFunc int
vprintf(const char *fmt
	, va_list ap)
{
  	assert(fmt != NULL);
  	assert(ap != NULL);
  return vfprintf(stdout,fmt,ap);
}

int vsprintf(char *buf, const char *fmt, va_list ap)
{
static FILE  string[1] =
{
   {0, 0, (char*)(unsigned) -1, 0, (char*) (unsigned) -1, -1,
    _IOFBF | __MODE_WRITE}
};

  int rv;

  assert(buf != NULL);
  assert(fmt != NULL);
  assert(ap != NULL);

  string->bufpos = buf;
  rv = vfprintf(string,fmt,ap);
  *(string->bufpos) = 0;
  return rv;
}

#ifdef _STDIO_FLOATS
int (*__fp_print)() = 0;
#endif		/* _STDIO_FLOATS */

/*
 * Output the given field in the manner specified by the arguments. Return
 * the number of characters output.
 */
static int
prtfld(FILE *op, uchar_t *buf, int ljustf, uchar_t sign, uchar_t pad, int width
	, int preci, int buffer_mode)
{
   int cnt = 0, len;
   unsigned ch;

   len = strlen(buf);

   if (*buf == '-')
      sign = *buf++;
   else if (sign)
      len++;

   if ((preci != -1) && (len > preci))	/* limit max data width */
      len = preci;

   if (width < len)		/* flexible field width or width overflow */
      width = len;

   /*
    * at this point: width = total field width len   = actual data width
    * (including possible sign character)
    */
   cnt = width;
   width -= len;

   while (width || len)
   {
      if (!ljustf && width)	/* left padding */
      {
	 if (len && sign && (pad == '0'))
	    goto showsign;
	 ch = pad;
	 --width;
      }
      else if (len)
      {
	 if (sign)
	 {
	  showsign:ch = sign;	/* sign */
	    sign = '\0';
	 }
	 else
	    ch = *buf++;	/* main field */
	 --len;
      }
      else
      {
	 ch = pad;		/* right padding */
	 --width;
      }
      putc(ch, op);
      if( ch == '\n' && buffer_mode == _IOLBF ) fflush(op);
   }

   return (cnt);
}

_CLibFunc int
vfprintf(FILE *fp
	, REG const char *fmt
	, va_list ap)
{
   REG int i, cnt = 0, ljustf, lval;
   int   preci, dpoint, width;
   char  pad, sign, radix, hash;
   REG char *ptmp;
   char  tmp[64];
   int buffer_mode;

	assert(fp != NULL);
	assert(fmt != NULL);
	assert(ap != NULL);
#define op fp		/* For assert() have the *fp !! */

   /* We set the buffer mode to fully-buffered and handle the actual
   		buffer mode later. This can speed up the things for unbuffered
   		streams. */
   buffer_mode = (op->mode&__MODE_BUF);
   op->mode &= (~__MODE_BUF);
#if _IOFBF		/* Possibly someone will change this for unknown reason */
	op->mode |= _IOFBF;
#endif	/* _IOFBF */

   while (*fmt) {
      if (*fmt == '%') {
         /* Why shall we flush the stream at this point? wouldn't it
         	enough to flush it at the very end of this function?
         			- 1997/11/15 ska*/
         if( buffer_mode == _IONBF ) fflush(op);
	 ljustf = 0;		/* left justify flag */
	 sign = '\0';		/* sign char & status */
	 pad = ' ';		/* justification padding char */
	 width = -1;		/* min field width */
	 dpoint = 0;		/* found decimal point */
	 preci = -1;		/* max data width */
	 radix = 10;		/* number base */
	 ptmp = tmp;		/* pointer to area to print */
	 hash = 0;
	 lval = (sizeof(int)==sizeof(long));	/* long value flaged */
fmtnxt:
	 i = 0;
	 for(;;) {
	    ++fmt;
	    if(*fmt < '0' || *fmt > '9' ) break;
	    i = (i * 10) + (*fmt - '0');
	    if (dpoint)
	       preci = i;
	    else if (!i && (pad == ' '))
	    {
	       pad = '0';
	       goto fmtnxt;
	    }
	    else
	       width = i;
	 }

	 switch (*fmt)
	 {
	 case '\0':		/* early EOS */
	    --fmt;
	    goto charout;

	 case '-':		/* left justification */
	    ljustf = 1;
	    goto fmtnxt;

	 case ' ':
	 case '+':		/* leading sign flag */
	    sign = *fmt;
	    goto fmtnxt;

	 case '*':		/* parameter width value */
	    i = va_arg(ap, int);
	    if (dpoint)
	       preci = i;
	    else
	       width = i;
	    goto fmtnxt;

	 case '.':		/* secondary width field */
	    dpoint = 1;
	    goto fmtnxt;

	 case 'l':		/* long data */
	    lval = 1;
	    goto fmtnxt;

	 case 'h':		/* short data */
	    lval = 0;
	    goto fmtnxt;

	 case 'd':		/* Signed decimal */
	 case 'i':
	    ptmp = _ltostr((long) ((lval)
			 ? va_arg(ap, long)
			 : va_arg(ap, short)),
		 10);
	    goto printit;

	 case 'b':		/* Unsigned binary */
	    radix = 2;
	    goto usproc;

	 case 'o':		/* Unsigned octal */
	    radix = 8;
	    goto usproc;

	 case 'p':		/* Pointer */
	    lval = (sizeof(char*) == sizeof(long));
	    pad = '0';
	    width = 6;
	    preci = 8;
	    /* fall thru */

	 case 'x':		/* Unsigned hexadecimal */
	 case 'X':
	    radix = 16;
	    /* fall thru */

	 case 'u':		/* Unsigned decimal */
	  usproc:
	    ptmp = _ultostr((unsigned long) ((lval)
				   ? va_arg(ap, unsigned long)
				   : va_arg(ap, unsigned short)),
		  radix);
	    if( hash && radix == 8 ) {
	    	width = strlen(ptmp)+1; pad='0';
	    }
	    goto printit;

	 case '#':
	    hash=1;
	    goto fmtnxt;

	 case 'c':		/* Character */
	    ptmp[0] = va_arg(ap, int);
	    ptmp[1] = '\0';
	    goto nopad;

	 case 's':		/* String */
	    ptmp = va_arg(ap, char*);
	  nopad:
	    sign = '\0';
	    pad = ' ';
	  printit:
	    cnt += prtfld(op, ptmp, ljustf,
			   sign, pad, width, preci, buffer_mode);
	    break;

#ifdef _STDIO_FLOATS
	 case 'e':		/* float */
	 case 'f':
	 case 'g':
	 case 'E':
	 case 'G':
	    if ( __fp_print )
	    {
	       (*__fp_print)(&va_arg(ap, double), *fmt, preci, ptmp);
	       preci = -1;
	       goto printit;
	    }
	    /* FALLTHROUGH if no floating printf available */
#endif		/* _STDIO_FLOATS */

	 default:		/* unknown character */
	    goto charout;
	 }
      }
      else {
charout:
		putc(*fmt, op);	/* normal char out */
		++cnt;
		/* shouldn't we flush at the very end of the function?
			- 1997/11/15 ska*/
		if( *fmt == '\n' && buffer_mode == _IOLBF ) fflush(op);
      }
      ++fmt;
   }

#if _IOFBF		/* someone might find it reasonable to change _IOFBF */
   op->mode &= (~__MODE_BUF);
#endif	/* _IOFBF */
   op->mode |= buffer_mode;		/* reset the buffer mode to the original */

   /* Although the switch statement looks heavier than two if's,
   	most compilers have a special optimization for it. Some compilers
   	even transalte it into two if's - 1997/11/15 ska*/
   switch(buffer_mode) {
   case _IONBF:	fflush(op); break;
   case _IOLBF: op->bufwrite = op->bufstart; break;
   }

   return (cnt);
}
