/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: _HPFREE.C $
   $Locker: ska $	$Name:  $	$State: Exp $

	_heapNode *_heapfree(_heapNode *thisNode, _heapNode *prevNode)

	Mark the node thisNode as unused and join thisNode with the
	previous node prevNode and/or the following node, if the
	particular node was unused already.

	This function returns the address of the joined together node.

	If the resulting node is situated at the end of the heap, the
	node structure is initialized as if would not, but the node type
	is set to HEAP_END.

	Input:
		thisNode != NULL
		prevNode == NULL => thisNode == _heap
		both must point to valid heap entries

	Return:
		thisNode: if prevNode was non-existent or allocated
		prevNode: if prevNode was existent and unused

	Note:
		Local heap only

	Conforms to:
		<none>

	See also:
		malloc, realloc, calloc, _fmalloc, free

	Target compilers:
		Any C compiler

	Origin:
		1997/10/10 Steffen Kaiser (ska)

	Revised by:
		<none>

	File Revision:    Revision 1.3  1998/02/11 07:40:34  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include <_alloc.h>
#include <dos.h>
#include <string.h>

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: _HPFREE.C 1.3 1998/02/11 07:40:34 ska Exp ska $";
#endif

_CLibFunc _pHeapNode
_heapfree(REG _pHeapNode thisNode
	, _pHeapNode const prevNode)
{	REG _pHeapNode nextNode;

	assert(thisNode != NULL);
	assert(_heapfind(nodeData(thisNode),  NULL));
	assert(!prevNode || _heapfind(nodeData(prevNode),  NULL));

	/* join previous node if it is unused, too */
	if(prevNode && prevNode->type == HEAP_UNUSED) {
		joinNodes(prevNode, thisNode);
		thisNode = prevNode;
	}
	else thisNode->type = HEAP_UNUSED;

	/* join the next node if it is unused or the end of heap */
	nextNode = nxtNode(thisNode);
	switch(nextNode->type) {
	case HEAP_UNUSED:		/* join this node, too */
		joinNodes(prevNode, thisNode);
		break;
	case HEAP_END:	/* lower the heap border */
		thisNode->type = HEAP_END;
		break;
	}
	
	return thisNode;
}
