/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $Id: STDIO.H 1.2 1998/01/29 07:10:10 ska Exp $
   $Locker:  $	$Name:  $	$State: Exp $

	Declarations and macros for the FILE* (stdio) package

*/

#ifndef __STDIO_H
#define __STDIO_H

#include <_null.h>		/* NULL */
#include <_seek.h>		/* seek whence macros */
#include <limits.h>		/* PATH_MAX */
#include <stdarg.h>		/* va_list for v*printf() & v*scanf() */

/* Parameters of STDIO */
#define EOF	(-1)		/* end of file condition */
#define BUFSIZ	(256)	/* default size of buffers */
#define FOPEN_MAX (20)	/* min #of files that can be opened simultaneously */
#define FILENAME_MAX PATH_MAX	/* max length of filename the OS can open */
#define TMP_MAX	65535	/* min #of unique filenames generated by tmpnam() */

/* What kind of buffering is supported */
#define _IOFBF		0x00	/* full buffering */
#define _IOLBF		0x01	/* line buffering */
#define _IONBF		0x02	/* no buffering */
#define __MODE_BUF	0x03	/* Modal buffering dependent on isatty */

/* Resource management */
#define __MODE_FREEBUF	0x04	/* Buffer allocated with malloc, can free */
#define __MODE_FREEFIL	0x08	/* FILE allocated with malloc, can free */

/* Requested permissions */
#define __MODE_READ	0x10	/* Opened in read only */
#define __MODE_WRITE	0x20	/* Opened in write only */
#define __MODE_RDWR	0x30	/* Opened in read/write */

/* Type of last access */
#define __MODE_READING	0x40	/* Buffer has pending read data */
#define __MODE_WRITING	0x80	/* Buffer has pending write data */

/* Status flags */
#define __MODE_EOF	0x100	/* EOF status */
#define __MODE_ERR	0x200	/* Error status */
#define __MODE_UNGOT	0x400	/* Buffer has been polluted by ungetc */

#ifdef __MSDOS__
#define __MODE_IOTRAN	0x1000	/* MSDOS nl <-> cr,nl translation */
#else
#define __MODE_IOTRAN	0
#endif
#define __MODE_TEXT __MODE_IOTRAN	/* text specific conversations */

/* when you add or change fields here, be sure to change the initialization
 * in stdio_init and fopen, printf & scanf as well */
struct __stdio_file {
  uchar_t *bufpos;   /* the next byte to write to or read from */
  uchar_t *bufread;  /* the end of data returned by last read() */
  uchar_t *bufwrite; /* highest address writable by macro */
  uchar_t *bufstart; /* the start of the buffer */
  uchar_t *bufend;   /* the end of the buffer; ie the byte after the last
                              malloc()ed byte */

  int fd; /* the file descriptor associated with the stream */
  int mode;

  uchar_t unbuf[8];	   /* The buffer for 'unbuffered' streams */

  struct __stdio_file * next;
};

typedef struct __stdio_file FILE;

extern FILE stdin[1];
extern FILE stdout[1];
extern FILE stderr[1];
extern FILE stdprn[1];
extern FILE stdaux[1];

#ifdef __MSDOS__			/* macros don't handle character rewrite rules */
#define putc(c, fp) fputc(c, fp)
#define getc(fp) fgetc(fp)
#else
#define putc(c, stream)	\
    (((stream)->bufpos >= (stream)->bufwrite) ? fputc((c), (stream))	\
                          : (unsigned char) (*(stream)->bufpos++ = (c))	)

#define getc(stream)	\
  (((stream)->bufpos >= (stream)->bufread) ? fgetc(stream):		\
    (*(stream)->bufpos++))
#endif

#define putchar(c) putc((c), stdout)  
#define getchar() getc(stdin)

#define ferror(fp)	(((fp)->mode&__MODE_ERR) != 0)
#define feof(fp)   	(((fp)->mode&__MODE_EOF) != 0)
#define clearerr(fp)	(void)((fp)->mode &= ~(__MODE_EOF|__MODE_ERR))
#define fileno(fp)	((fp)->fd)

extern off_t ftell(FILE*);
extern int fseek(FILE*, off_t, int);

/* These two call malloc */
#define setlinebuf(__fp)             setvbuf((__fp), (char*)0, _IOLBF, 0)
extern int setvbuf(FILE*, char*, int, size_t);

/* These don't */
#define setbuf(__fp, __buf) setbuffer((__fp), (__buf), BUFSIZ)
extern void setbuffer(FILE*, char*, size_t);

extern int fgetc(FILE*);
extern int ungetc(int, FILE*);
extern int fputc(int, FILE*);

extern int fclose(FILE*);
extern int fflush(FILE*);
extern int fflushall(void);
extern int fcloseall(void);

#define fopen(__file, __mode)         __fopen((__file), -1, (FILE*)0, (__mode))
#define freopen(__file, __mode, __fp) __fopen((__file), -1, (__fp), (__mode))
#define fdopen(__file, __mode)  __fopen((char*)0, (__file), (FILE*)0, (__mode))
extern FILE *__fopen(const char*, int, FILE*, const char*);

extern int fputs(const char*, FILE*);
extern int puts(const char*);
extern char *gets(char*);
extern char *fgets(char*, size_t, FILE*);
extern size_t fget(char *, size_t, FILE*);
extern size_t fread(void *, size_t, size_t, FILE*);
extern size_t fwrite(const void *, size_t, size_t, FILE*);

extern int printf(const char *fmt, ...);
extern int fprintf(FILE *f, const char *fmt, ...);
extern int sprintf(char *buf, const char *fmt, ...);
extern int vprintf(const char *fmt, va_list ap);
extern int vfprintf(FILE *f, const char *fmt, va_list ap);
extern int vsprintf(char *buf, const char *fmt, va_list ap);

extern int scanf(const char *fmt, ...);
extern int fscanf(FILE *f, const char *fmt, ...);
extern int sscanf(const char *buf, const char *fmt, ...);
extern int vscanf(const char *fmt, va_list ap);
extern int vfscanf(FILE *f, const char *fmt, va_list ap);
extern int vsscanf(char *buf, const char *fmt, va_list ap);


#define stdio_pending(fp) ((fp)->bufread>(fp)->bufpos)

#endif
