/*
    This file is part of the CLib sub-project of the FreeDOS project
    Copyright (C) 1997 by the author see below

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: CHSIZE.C $
   $Locker:  $	$Name:  $	$State: Exp $

	int chsize(int fd, off_t offset)

	Change size of file.
	If the file is currently larger, the file is truncated at the position
	"offset". If it is smaller, it is extended by appending NUL bytes.

	Input:
		0 <= offset < (off_t)-1

	Return:
		0 on success
		else on failure

	Note:
		<none>

	Conforms to:
		<none>

	See also:
		_truncate

	Target compilers:
		Any C compiler

	Origin:
		1997/10/10 Steffen Kaiser (ska)

	Revised by:
		<none>

	File Revision:    Revision 1.1  1997/10/13 07:06:59  ska
*/

#include <_clib.h>			/* standard include, must be the first! */
#include <io.h>
#include <string.h>

#ifdef RCS_Version
static char const rcsid[] = 
	"$Id: CHSIZE.C 1.1 1997/10/13 07:06:59 ska Exp $";
#endif

_CLibFunc int
chsize(const int fd
      ,const off_t offset)
{	off_t curpos;
	REG off_t length;
	int errVal;

	if((curpos = tell(fd)) == _SEEK_FAIL
	 || (length = lseek(fd, 0, SEEK_END)) == _SEEK_FAIL)
		return -1;

	errVal = 0;			/* no error */

	if(offset > length) {			/* extend file */
		byte buf[__CHSIZE_BUFFER];			/* temporary buffer */

		length = offset - length;
		/* we have to write length NUL bytes to the file. For
			performance reasons we block the write using "buf". */

		memset(buf, 0, sizeof(buf));
		for(;length > sizeof(buf); length -= sizeof(buf))
			if(write(fd, buf, sizeof(buf)) != sizeof(buf)) {
				/* write failed. We have to assume disk full */
				errVal = 1;
				goto errRet;
			}

		if(write(fd, buf, (size_t)length) != (size_t)length)
			errVal = 1;
	}
	else {			/* truncate file */
		if(offset != length) {
			if(lseek(fd, offset, SEEK_SET) == _SEEK_FAIL)
				errVal = _SEEK_FAIL;
			else errVal = _truncate(fd);
		}
	}

errRet:
	/* Restore filepointer even if an error occured */
	return lseek(fd, curpos, SEEK_SET) == _SEEK_FAIL? _SEEK_FAIL: errVal;
}
